/* SPDX-License-Identifier: BSD-3-Clause
 * Copyright(c) 2015-2020 Beijing WangXun Technology Co., Ltd.
 * Copyright(c) 2010-2017 Intel Corporation
 */

#include <sys/queue.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <stdint.h>
#include <stdarg.h>
#include <unistd.h>
#include <inttypes.h>

#include <rte_byteorder.h>
#include <rte_common.h>
#include <rte_cycles.h>
#include <rte_log.h>
#include <rte_debug.h>
#include <rte_ethdev.h>
#include <ethdev_driver.h>
#include <rte_security_driver.h>
#include <rte_memzone.h>
#include <rte_atomic.h>
#include <rte_mempool.h>
#include <rte_malloc.h>
#include <rte_mbuf.h>
#include <rte_ether.h>
#include <rte_prefetch.h>
#include <rte_udp.h>
#include <rte_tcp.h>
#include <rte_sctp.h>
#include <rte_string_fns.h>
#include <rte_errno.h>
#include <rte_ip.h>
#include <rte_net.h>

#include "txgbe_logs.h"
#include "base/txgbe.h"
#include "txgbe_ethdev.h"
#include "txgbe_rxtx.h"

#ifdef RTE_LIBRTE_IEEE1588
#define TXGBE_TX_IEEE1588_TMST RTE_MBUF_F_TX_IEEE1588_TMST
#else
#define TXGBE_TX_IEEE1588_TMST 0
#endif

/* Bit Mask to indicate what bits required for building TX context */
static const u64 TXGBE_TX_OFFLOAD_MASK = (RTE_MBUF_F_TX_IP_CKSUM |
		RTE_MBUF_F_TX_OUTER_IPV6 |
		RTE_MBUF_F_TX_OUTER_IPV4 |
		RTE_MBUF_F_TX_IPV6 |
		RTE_MBUF_F_TX_IPV4 |
		RTE_MBUF_F_TX_VLAN |
		RTE_MBUF_F_TX_L4_MASK |
		RTE_MBUF_F_TX_TCP_SEG |
		RTE_MBUF_F_TX_TUNNEL_MASK |
		RTE_MBUF_F_TX_OUTER_IP_CKSUM |
		RTE_MBUF_F_TX_OUTER_UDP_CKSUM |
#ifdef RTE_LIB_SECURITY
		RTE_MBUF_F_TX_SEC_OFFLOAD |
#endif
		TXGBE_TX_IEEE1588_TMST);

#define TXGBE_TX_OFFLOAD_NOTSUP_MASK \
		(RTE_MBUF_F_TX_OFFLOAD_MASK ^ TXGBE_TX_OFFLOAD_MASK)

/*
 * Prefetch a cache line into all cache levels.
 */
#define rte_txgbe_prefetch(p)   rte_prefetch0(p)

static int
txgbe_is_vf(struct rte_eth_dev *dev)
{
	struct txgbe_hw *hw = TXGBE_DEV_HW(dev);

	switch (hw->mac.type) {
	case txgbe_mac_raptor_vf:
		return 1;
	default:
		return 0;
	}
}

/*********************************************************************
 *
 *  TX functions
 *
 **********************************************************************/

/*
 * Check for descriptors with their DD bit set and free mbufs.
 * Return the total number of buffers freed.
 */
static __rte_always_inline int
txgbe_tx_free_bufs(struct txgbe_tx_queue *txq)
{
	struct txgbe_tx_entry *txep;
	uint32_t status;
	int i, nb_free = 0;
	struct rte_mbuf *m, *free[RTE_TXGBE_TX_MAX_FREE_BUF_SZ];

	/* check DD bit on threshold descriptor */
	status = txq->tx_ring[txq->tx_next_dd].dw3;
	if (!(status & rte_cpu_to_le_32(TXGBE_TXD_DD))) {
		if (txq->nb_tx_free >> 1 < txq->tx_free_thresh)
			txgbe_set32_masked(txq->tdc_reg_addr,
				TXGBE_TXCFG_FLUSH, TXGBE_TXCFG_FLUSH);
		return 0;
	}

	/*
	 * first buffer to free from S/W ring is at index
	 * tx_next_dd - (tx_free_thresh-1)
	 */
	txep = &txq->sw_ring[txq->tx_next_dd - (txq->tx_free_thresh - 1)];
	for (i = 0; i < txq->tx_free_thresh; ++i, ++txep) {
		/* free buffers one at a time */
		m = rte_pktmbuf_prefree_seg(txep->mbuf);
		txep->mbuf = NULL;

		if (unlikely(m == NULL))
			continue;

		if (nb_free >= RTE_TXGBE_TX_MAX_FREE_BUF_SZ ||
		    (nb_free > 0 && m->pool != free[0]->pool)) {
			rte_mempool_put_bulk(free[0]->pool,
					     (void **)free, nb_free);
			nb_free = 0;
		}

		free[nb_free++] = m;
	}

	if (nb_free > 0)
		rte_mempool_put_bulk(free[0]->pool, (void **)free, nb_free);

	/* buffers were freed, update counters */
	txq->nb_tx_free = (uint16_t)(txq->nb_tx_free + txq->tx_free_thresh);
	txq->tx_next_dd = (uint16_t)(txq->tx_next_dd + txq->tx_free_thresh);
	if (txq->tx_next_dd >= txq->nb_tx_desc)
		txq->tx_next_dd = (uint16_t)(txq->tx_free_thresh - 1);

	return txq->tx_free_thresh;
}

/* Populate 4 descriptors with data from 4 mbufs */
static inline void
tx4(volatile struct txgbe_tx_desc *txdp, struct rte_mbuf **pkts)
{
	uint64_t buf_dma_addr;
	uint32_t pkt_len;
	int i;

	for (i = 0; i < 4; ++i, ++txdp, ++pkts) {
		buf_dma_addr = rte_mbuf_data_iova(*pkts);
		pkt_len = (*pkts)->data_len;

		/* write data to descriptor */
		txdp->qw0 = rte_cpu_to_le_64(buf_dma_addr);
		txdp->dw2 = cpu_to_le32(TXGBE_TXD_FLAGS |
					TXGBE_TXD_DATLEN(pkt_len));
		txdp->dw3 = cpu_to_le32(TXGBE_TXD_PAYLEN(pkt_len));

		rte_prefetch0(&(*pkts)->pool);
	}
}

/* Populate 1 descriptor with data from 1 mbuf */
static inline void
tx1(volatile struct txgbe_tx_desc *txdp, struct rte_mbuf **pkts)
{
	uint64_t buf_dma_addr;
	uint32_t pkt_len;

	buf_dma_addr = rte_mbuf_data_iova(*pkts);
	pkt_len = (*pkts)->data_len;

	/* write data to descriptor */
	txdp->qw0 = cpu_to_le64(buf_dma_addr);
	txdp->dw2 = cpu_to_le32(TXGBE_TXD_FLAGS |
				TXGBE_TXD_DATLEN(pkt_len));
	txdp->dw3 = cpu_to_le32(TXGBE_TXD_PAYLEN(pkt_len));

	rte_prefetch0(&(*pkts)->pool);
}

/*
 * Fill H/W descriptor ring with mbuf data.
 * Copy mbuf pointers to the S/W ring.
 */
static inline void
txgbe_tx_fill_hw_ring(struct txgbe_tx_queue *txq, struct rte_mbuf **pkts,
		      uint16_t nb_pkts)
{
	volatile struct txgbe_tx_desc *txdp = &txq->tx_ring[txq->tx_tail];
	struct txgbe_tx_entry *txep = &txq->sw_ring[txq->tx_tail];
	const int N_PER_LOOP = 4;
	const int N_PER_LOOP_MASK = N_PER_LOOP - 1;
	int mainpart, leftover;
	int i, j;

	/*
	 * Process most of the packets in chunks of N pkts.  Any
	 * leftover packets will get processed one at a time.
	 */
	mainpart = (nb_pkts & ((uint32_t)~N_PER_LOOP_MASK));
	leftover = (nb_pkts & ((uint32_t)N_PER_LOOP_MASK));
	for (i = 0; i < mainpart; i += N_PER_LOOP) {
		/* Copy N mbuf pointers to the S/W ring */
		for (j = 0; j < N_PER_LOOP; ++j)
			(txep + i + j)->mbuf = *(pkts + i + j);
		tx4(txdp + i, pkts + i);
	}

	if (unlikely(leftover > 0)) {
		for (i = 0; i < leftover; ++i) {
			(txep + mainpart + i)->mbuf = *(pkts + mainpart + i);
			tx1(txdp + mainpart + i, pkts + mainpart + i);
		}
	}
}

static inline uint16_t
tx_xmit_pkts(void *tx_queue, struct rte_mbuf **tx_pkts,
	     uint16_t nb_pkts)
{
	struct txgbe_tx_queue *txq = (struct txgbe_tx_queue *)tx_queue;
	uint16_t n = 0;

	/*
	 * Begin scanning the H/W ring for done descriptors when the
	 * number of available descriptors drops below tx_free_thresh.  For
	 * each done descriptor, free the associated buffer.
	 */
	if (txq->nb_tx_free < txq->tx_free_thresh)
		txgbe_tx_free_bufs(txq);

	/* Only use descriptors that are available */
	nb_pkts = (uint16_t)RTE_MIN(txq->nb_tx_free, nb_pkts);
	if (unlikely(nb_pkts == 0))
		return 0;

	/* Use exactly nb_pkts descriptors */
	txq->nb_tx_free = (uint16_t)(txq->nb_tx_free - nb_pkts);

	/*
	 * At this point, we know there are enough descriptors in the
	 * ring to transmit all the packets.  This assumes that each
	 * mbuf contains a single segment, and that no new offloads
	 * are expected, which would require a new context descriptor.
	 */

	/*
	 * See if we're going to wrap-around. If so, handle the top
	 * of the descriptor ring first, then do the bottom.  If not,
	 * the processing looks just like the "bottom" part anyway...
	 */
	if ((txq->tx_tail + nb_pkts) > txq->nb_tx_desc) {
		n = (uint16_t)(txq->nb_tx_desc - txq->tx_tail);
		txgbe_tx_fill_hw_ring(txq, tx_pkts, n);
		txq->tx_tail = 0;
	}

	/* Fill H/W descriptor ring with mbuf data */
	txgbe_tx_fill_hw_ring(txq, tx_pkts + n, (uint16_t)(nb_pkts - n));
	txq->tx_tail = (uint16_t)(txq->tx_tail + (nb_pkts - n));

	/*
	 * Check for wrap-around. This would only happen if we used
	 * up to the last descriptor in the ring, no more, no less.
	 */
	if (txq->tx_tail >= txq->nb_tx_desc)
		txq->tx_tail = 0;

	PMD_TX_LOG(DEBUG, "port_id=%u queue_id=%u tx_tail=%u nb_tx=%u",
		   (uint16_t)txq->port_id, (uint16_t)txq->queue_id,
		   (uint16_t)txq->tx_tail, (uint16_t)nb_pkts);

	/* update tail pointer */
	rte_wmb();
	txgbe_set32_relaxed(txq->tdt_reg_addr, txq->tx_tail);

	return nb_pkts;
}

uint16_t
txgbe_xmit_pkts_simple(void *tx_queue, struct rte_mbuf **tx_pkts,
		       uint16_t nb_pkts)
{
	uint16_t nb_tx;

	/* Try to transmit at least chunks of TX_MAX_BURST pkts */
	if (likely(nb_pkts <= RTE_PMD_TXGBE_TX_MAX_BURST))
		return tx_xmit_pkts(tx_queue, tx_pkts, nb_pkts);

	/* transmit more than the max burst, in chunks of TX_MAX_BURST */
	nb_tx = 0;
	while (nb_pkts) {
		uint16_t ret, n;

		n = (uint16_t)RTE_MIN(nb_pkts, RTE_PMD_TXGBE_TX_MAX_BURST);
		ret = tx_xmit_pkts(tx_queue, &tx_pkts[nb_tx], n);
		nb_tx = (uint16_t)(nb_tx + ret);
		nb_pkts = (uint16_t)(nb_pkts - ret);
		if (ret < n)
			break;
	}

	return nb_tx;
}

static inline void
txgbe_set_xmit_ctx(struct txgbe_tx_queue *txq,
		volatile struct txgbe_tx_ctx_desc *ctx_txd,
		uint64_t ol_flags, union txgbe_tx_offload tx_offload,
		__rte_unused uint64_t *mdata)
{
	union txgbe_tx_offload tx_offload_mask;
	uint32_t type_tucmd_mlhl;
	uint32_t mss_l4len_idx;
	uint32_t ctx_idx;
	uint32_t vlan_macip_lens;
	uint32_t tunnel_seed;

	ctx_idx = txq->ctx_curr;
	tx_offload_mask.data[0] = 0;
	tx_offload_mask.data[1] = 0;

	/* Specify which HW CTX to upload. */
	mss_l4len_idx = TXGBE_TXD_IDX(ctx_idx);
	type_tucmd_mlhl = TXGBE_TXD_CTXT;

	tx_offload_mask.ptid |= ~0;
	type_tucmd_mlhl |= TXGBE_TXD_PTID(tx_offload.ptid);

	/* check if TCP segmentation required for this packet */
	if (ol_flags & RTE_MBUF_F_TX_TCP_SEG) {
		tx_offload_mask.l2_len |= ~0;
		tx_offload_mask.l3_len |= ~0;
		tx_offload_mask.l4_len |= ~0;
		tx_offload_mask.tso_segsz |= ~0;
		mss_l4len_idx |= TXGBE_TXD_MSS(tx_offload.tso_segsz);
		mss_l4len_idx |= TXGBE_TXD_L4LEN(tx_offload.l4_len);
	} else { /* no TSO, check if hardware checksum is needed */
		if (ol_flags & RTE_MBUF_F_TX_IP_CKSUM) {
			tx_offload_mask.l2_len |= ~0;
			tx_offload_mask.l3_len |= ~0;
		}

		switch (ol_flags & RTE_MBUF_F_TX_L4_MASK) {
		case RTE_MBUF_F_TX_UDP_CKSUM:
			mss_l4len_idx |=
				TXGBE_TXD_L4LEN(sizeof(struct rte_udp_hdr));
			tx_offload_mask.l2_len |= ~0;
			tx_offload_mask.l3_len |= ~0;
			break;
		case RTE_MBUF_F_TX_TCP_CKSUM:
			mss_l4len_idx |=
				TXGBE_TXD_L4LEN(sizeof(struct rte_tcp_hdr));
			tx_offload_mask.l2_len |= ~0;
			tx_offload_mask.l3_len |= ~0;
			break;
		case RTE_MBUF_F_TX_SCTP_CKSUM:
			mss_l4len_idx |=
				TXGBE_TXD_L4LEN(sizeof(struct rte_sctp_hdr));
			tx_offload_mask.l2_len |= ~0;
			tx_offload_mask.l3_len |= ~0;
			break;
		default:
			break;
		}
	}

	vlan_macip_lens = TXGBE_TXD_IPLEN(tx_offload.l3_len >> 1);

	if (ol_flags & RTE_MBUF_F_TX_TUNNEL_MASK) {
		tx_offload_mask.outer_tun_len |= ~0;
		tx_offload_mask.outer_l2_len |= ~0;
		tx_offload_mask.outer_l3_len |= ~0;
		tx_offload_mask.l2_len |= ~0;
		tunnel_seed = TXGBE_TXD_ETUNLEN(tx_offload.outer_tun_len >> 1);
		tunnel_seed |= TXGBE_TXD_EIPLEN(tx_offload.outer_l3_len >> 2);

		switch (ol_flags & RTE_MBUF_F_TX_TUNNEL_MASK) {
		case RTE_MBUF_F_TX_TUNNEL_IPIP:
			/* for non UDP / GRE tunneling, set to 0b */
			break;
		case RTE_MBUF_F_TX_TUNNEL_VXLAN:
		case RTE_MBUF_F_TX_TUNNEL_VXLAN_GPE:
		case RTE_MBUF_F_TX_TUNNEL_GENEVE:
			tunnel_seed |= TXGBE_TXD_ETYPE_UDP;
			break;
		case RTE_MBUF_F_TX_TUNNEL_GRE:
			tunnel_seed |= TXGBE_TXD_ETYPE_GRE;
			break;
		default:
			PMD_TX_LOG(ERR, "Tunnel type not supported");
			return;
		}
		vlan_macip_lens |= TXGBE_TXD_MACLEN(tx_offload.outer_l2_len);
	} else {
		tunnel_seed = 0;
		vlan_macip_lens |= TXGBE_TXD_MACLEN(tx_offload.l2_len);
	}

	if (ol_flags & RTE_MBUF_F_TX_VLAN) {
		tx_offload_mask.vlan_tci |= ~0;
		vlan_macip_lens |= TXGBE_TXD_VLAN(tx_offload.vlan_tci);
	}

#ifdef RTE_LIB_SECURITY
	if (ol_flags & RTE_MBUF_F_TX_SEC_OFFLOAD) {
		union txgbe_crypto_tx_desc_md *md =
				(union txgbe_crypto_tx_desc_md *)mdata;
		tunnel_seed |= TXGBE_TXD_IPSEC_SAIDX(md->sa_idx);
		type_tucmd_mlhl |= md->enc ?
			(TXGBE_TXD_IPSEC_ESP | TXGBE_TXD_IPSEC_ESPENC) : 0;
		type_tucmd_mlhl |= TXGBE_TXD_IPSEC_ESPLEN(md->pad_len);
		tx_offload_mask.sa_idx |= ~0;
		tx_offload_mask.sec_pad_len |= ~0;
	}
#endif

	txq->ctx_cache[ctx_idx].flags = ol_flags;
	txq->ctx_cache[ctx_idx].tx_offload.data[0] =
		tx_offload_mask.data[0] & tx_offload.data[0];
	txq->ctx_cache[ctx_idx].tx_offload.data[1] =
		tx_offload_mask.data[1] & tx_offload.data[1];
	txq->ctx_cache[ctx_idx].tx_offload_mask = tx_offload_mask;

	ctx_txd->dw0 = rte_cpu_to_le_32(vlan_macip_lens);
	ctx_txd->dw1 = rte_cpu_to_le_32(tunnel_seed);
	ctx_txd->dw2 = rte_cpu_to_le_32(type_tucmd_mlhl);
	ctx_txd->dw3 = rte_cpu_to_le_32(mss_l4len_idx);
}

/*
 * Check which hardware context can be used. Use the existing match
 * or create a new context descriptor.
 */
static inline uint32_t
what_ctx_update(struct txgbe_tx_queue *txq, uint64_t flags,
		   union txgbe_tx_offload tx_offload)
{
	/* If match with the current used context */
	if (likely(txq->ctx_cache[txq->ctx_curr].flags == flags &&
		   (txq->ctx_cache[txq->ctx_curr].tx_offload.data[0] ==
		    (txq->ctx_cache[txq->ctx_curr].tx_offload_mask.data[0]
		     & tx_offload.data[0])) &&
		   (txq->ctx_cache[txq->ctx_curr].tx_offload.data[1] ==
		    (txq->ctx_cache[txq->ctx_curr].tx_offload_mask.data[1]
		     & tx_offload.data[1]))))
		return txq->ctx_curr;

	/* What if match with the next context  */
	txq->ctx_curr ^= 1;
	if (likely(txq->ctx_cache[txq->ctx_curr].flags == flags &&
		   (txq->ctx_cache[txq->ctx_curr].tx_offload.data[0] ==
		    (txq->ctx_cache[txq->ctx_curr].tx_offload_mask.data[0]
		     & tx_offload.data[0])) &&
		   (txq->ctx_cache[txq->ctx_curr].tx_offload.data[1] ==
		    (txq->ctx_cache[txq->ctx_curr].tx_offload_mask.data[1]
		     & tx_offload.data[1]))))
		return txq->ctx_curr;

	/* Mismatch, use the previous context */
	return TXGBE_CTX_NUM;
}

static inline uint32_t
tx_desc_cksum_flags_to_olinfo(uint64_t ol_flags)
{
	uint32_t tmp = 0;

	if ((ol_flags & RTE_MBUF_F_TX_L4_MASK) != RTE_MBUF_F_TX_L4_NO_CKSUM) {
		tmp |= TXGBE_TXD_CC;
		tmp |= TXGBE_TXD_L4CS;
	}
	if (ol_flags & RTE_MBUF_F_TX_IP_CKSUM) {
		tmp |= TXGBE_TXD_CC;
		tmp |= TXGBE_TXD_IPCS;
	}
	if (ol_flags & RTE_MBUF_F_TX_OUTER_IP_CKSUM) {
		tmp |= TXGBE_TXD_CC;
		tmp |= TXGBE_TXD_EIPCS;
	}
	if (ol_flags & RTE_MBUF_F_TX_TCP_SEG) {
		tmp |= TXGBE_TXD_CC;
		/* implies IPv4 cksum */
		if (ol_flags & RTE_MBUF_F_TX_IPV4)
			tmp |= TXGBE_TXD_IPCS;
		tmp |= TXGBE_TXD_L4CS;
	}
	if (ol_flags & RTE_MBUF_F_TX_VLAN)
		tmp |= TXGBE_TXD_CC;

	return tmp;
}

static inline uint32_t
tx_desc_ol_flags_to_cmdtype(uint64_t ol_flags)
{
	uint32_t cmdtype = 0;

	if (ol_flags & RTE_MBUF_F_TX_VLAN)
		cmdtype |= TXGBE_TXD_VLE;
	if (ol_flags & RTE_MBUF_F_TX_TCP_SEG)
		cmdtype |= TXGBE_TXD_TSE;
	if (ol_flags & RTE_MBUF_F_TX_MACSEC)
		cmdtype |= TXGBE_TXD_LINKSEC;
	return cmdtype;
}

static inline uint32_t
tx_desc_ol_flags_to_ptype(uint64_t oflags)
{
	uint32_t ptype;
	bool tun;

	/* Only support flags in TXGBE_TX_OFFLOAD_MASK */
	tun = !!(oflags & RTE_MBUF_F_TX_TUNNEL_MASK);

	/* L2 level */
	ptype = RTE_PTYPE_L2_ETHER;
	if (oflags & RTE_MBUF_F_TX_VLAN)
		ptype |= (tun ? RTE_PTYPE_INNER_L2_ETHER_VLAN : RTE_PTYPE_L2_ETHER_VLAN);

	if (oflags & RTE_MBUF_F_TX_QINQ) /* tunnel + QINQ is not supported */
		ptype |= RTE_PTYPE_L2_ETHER_VLAN;

	/* L3 level */
	if (oflags & (RTE_MBUF_F_TX_OUTER_IPV4 | RTE_MBUF_F_TX_OUTER_IP_CKSUM))
		ptype |= RTE_PTYPE_L3_IPV4;
	else if (oflags & (RTE_MBUF_F_TX_OUTER_IPV6))
		ptype |= RTE_PTYPE_L3_IPV6;

	if (oflags & (RTE_MBUF_F_TX_IPV4 | RTE_MBUF_F_TX_IP_CKSUM))
		ptype |= (tun ? RTE_PTYPE_INNER_L3_IPV4 : RTE_PTYPE_L3_IPV4);
	else if (oflags & (RTE_MBUF_F_TX_IPV6))
		ptype |= (tun ? RTE_PTYPE_INNER_L3_IPV6 : RTE_PTYPE_L3_IPV6);

	/* L4 level */
	switch (oflags & (RTE_MBUF_F_TX_L4_MASK)) {
	case RTE_MBUF_F_TX_TCP_CKSUM:
		ptype |= (tun ? RTE_PTYPE_INNER_L4_TCP : RTE_PTYPE_L4_TCP);
		break;
	case RTE_MBUF_F_TX_UDP_CKSUM:
		ptype |= (tun ? RTE_PTYPE_INNER_L4_UDP : RTE_PTYPE_L4_UDP);
		break;
	case RTE_MBUF_F_TX_SCTP_CKSUM:
		ptype |= (tun ? RTE_PTYPE_INNER_L4_SCTP : RTE_PTYPE_L4_SCTP);
		break;
	}

	if (oflags & RTE_MBUF_F_TX_TCP_SEG)
		ptype |= (tun ? RTE_PTYPE_INNER_L4_TCP : RTE_PTYPE_L4_TCP);

	/* Tunnel */
	switch (oflags & RTE_MBUF_F_TX_TUNNEL_MASK) {
	case RTE_MBUF_F_TX_TUNNEL_VXLAN:
	case RTE_MBUF_F_TX_TUNNEL_VXLAN_GPE:
		ptype |= RTE_PTYPE_L2_ETHER |
			 RTE_PTYPE_L3_IPV4 |
			 RTE_PTYPE_TUNNEL_GRENAT;
		break;
	case RTE_MBUF_F_TX_TUNNEL_GRE:
		ptype |= RTE_PTYPE_L2_ETHER |
			 RTE_PTYPE_L3_IPV4 |
			 RTE_PTYPE_TUNNEL_GRE;
		ptype |= RTE_PTYPE_INNER_L2_ETHER;
		break;
	case RTE_MBUF_F_TX_TUNNEL_GENEVE:
		ptype |= RTE_PTYPE_L2_ETHER |
			 RTE_PTYPE_L3_IPV4 |
			 RTE_PTYPE_TUNNEL_GENEVE;
		ptype |= RTE_PTYPE_INNER_L2_ETHER;
		break;
	case RTE_MBUF_F_TX_TUNNEL_IPIP:
	case RTE_MBUF_F_TX_TUNNEL_IP:
		ptype |= RTE_PTYPE_L2_ETHER |
			 RTE_PTYPE_L3_IPV4 |
			 RTE_PTYPE_TUNNEL_IP;
		break;
	}

	return ptype;
}

static inline uint8_t
tx_desc_ol_flags_to_ptid(uint64_t oflags)
{
	uint32_t ptype;

	ptype = tx_desc_ol_flags_to_ptype(oflags);

	return txgbe_encode_ptype(ptype);
}

#ifndef DEFAULT_TX_FREE_THRESH
#define DEFAULT_TX_FREE_THRESH 32
#endif

/* Reset transmit descriptors after they have been used */
static inline int
txgbe_xmit_cleanup(struct txgbe_tx_queue *txq)
{
	struct txgbe_tx_entry *sw_ring = txq->sw_ring;
	volatile struct txgbe_tx_desc *txr = txq->tx_ring;
	uint16_t last_desc_cleaned = txq->last_desc_cleaned;
	uint16_t nb_tx_desc = txq->nb_tx_desc;
	uint16_t desc_to_clean_to;
	uint16_t nb_tx_to_clean;
	uint32_t status;

	/* Determine the last descriptor needing to be cleaned */
	desc_to_clean_to = (uint16_t)(last_desc_cleaned + txq->tx_free_thresh);
	if (desc_to_clean_to >= nb_tx_desc)
		desc_to_clean_to = (uint16_t)(desc_to_clean_to - nb_tx_desc);

	/* Check to make sure the last descriptor to clean is done */
	desc_to_clean_to = sw_ring[desc_to_clean_to].last_id;
	status = txr[desc_to_clean_to].dw3;
	if (!(status & rte_cpu_to_le_32(TXGBE_TXD_DD))) {
		PMD_TX_FREE_LOG(DEBUG,
				"TX descriptor %4u is not done"
				"(port=%d queue=%d)",
				desc_to_clean_to,
				txq->port_id, txq->queue_id);
		if (txq->nb_tx_free >> 1 < txq->tx_free_thresh)
			txgbe_set32_masked(txq->tdc_reg_addr,
				TXGBE_TXCFG_FLUSH, TXGBE_TXCFG_FLUSH);
		/* Failed to clean any descriptors, better luck next time */
		return -(1);
	}

	/* Figure out how many descriptors will be cleaned */
	if (last_desc_cleaned > desc_to_clean_to)
		nb_tx_to_clean = (uint16_t)((nb_tx_desc - last_desc_cleaned) +
							desc_to_clean_to);
	else
		nb_tx_to_clean = (uint16_t)(desc_to_clean_to -
						last_desc_cleaned);

	PMD_TX_FREE_LOG(DEBUG,
			"Cleaning %4u TX descriptors: %4u to %4u "
			"(port=%d queue=%d)",
			nb_tx_to_clean, last_desc_cleaned, desc_to_clean_to,
			txq->port_id, txq->queue_id);

	/*
	 * The last descriptor to clean is done, so that means all the
	 * descriptors from the last descriptor that was cleaned
	 * up to the last descriptor with the RS bit set
	 * are done. Only reset the threshold descriptor.
	 */
	txr[desc_to_clean_to].dw3 = 0;

	/* Update the txq to reflect the last descriptor that was cleaned */
	txq->last_desc_cleaned = desc_to_clean_to;
	txq->nb_tx_free = (uint16_t)(txq->nb_tx_free + nb_tx_to_clean);

	/* No Error */
	return 0;
}

static inline uint8_t
txgbe_get_tun_len(struct rte_mbuf *mbuf)
{
	struct txgbe_genevehdr genevehdr;
	const struct txgbe_genevehdr *gh;
	uint8_t tun_len;

	switch (mbuf->ol_flags & RTE_MBUF_F_TX_TUNNEL_MASK) {
	case RTE_MBUF_F_TX_TUNNEL_IPIP:
		tun_len = 0;
		break;
	case RTE_MBUF_F_TX_TUNNEL_VXLAN:
	case RTE_MBUF_F_TX_TUNNEL_VXLAN_GPE:
		tun_len = sizeof(struct txgbe_udphdr)
			+ sizeof(struct txgbe_vxlanhdr);
		break;
	case RTE_MBUF_F_TX_TUNNEL_GRE:
		tun_len = sizeof(struct txgbe_nvgrehdr);
		break;
	case RTE_MBUF_F_TX_TUNNEL_GENEVE:
		gh = rte_pktmbuf_read(mbuf,
			mbuf->outer_l2_len + mbuf->outer_l3_len,
			sizeof(genevehdr), &genevehdr);
		tun_len = sizeof(struct txgbe_udphdr)
			+ sizeof(struct txgbe_genevehdr)
			+ (gh->opt_len << 2);
		break;
	default:
		tun_len = 0;
	}

	return tun_len;
}

static inline uint8_t
txgbe_parse_tun_ptid(struct rte_mbuf *tx_pkt)
{
	uint64_t l2_none, l2_mac, l2_mac_vlan;
	uint8_t ptid = 0;

	if ((tx_pkt->ol_flags & (RTE_MBUF_F_TX_TUNNEL_VXLAN |
				RTE_MBUF_F_TX_TUNNEL_VXLAN_GPE)) == 0)
		return ptid;

	l2_none = sizeof(struct txgbe_udphdr) + sizeof(struct txgbe_vxlanhdr);
	l2_mac = l2_none + sizeof(struct rte_ether_hdr);
	l2_mac_vlan = l2_mac + sizeof(struct rte_vlan_hdr);

	if (tx_pkt->l2_len == l2_none)
		ptid = TXGBE_PTID_TUN_EIG;
	else if (tx_pkt->l2_len == l2_mac)
		ptid = TXGBE_PTID_TUN_EIGM;
	else if (tx_pkt->l2_len == l2_mac_vlan)
		ptid = TXGBE_PTID_TUN_EIGMV;

	return ptid;
}

uint16_t
txgbe_xmit_pkts(void *tx_queue, struct rte_mbuf **tx_pkts,
		uint16_t nb_pkts)
{
	struct txgbe_tx_queue *t