"""Reminders service."""

import time
import uuid
from datetime import datetime
from typing import Any

from tzlocal import get_localzone_name

from pyicloud.services.base import BaseService
from pyicloud.session import PyiCloudSession


class RemindersService(BaseService):
    """The 'Reminders' iCloud service."""

    def __init__(
        self, service_root: str, session: PyiCloudSession, params: dict[str, Any]
    ) -> None:
        super().__init__(service_root, session, params)

        self.lists = {}
        self.collections = {}

        self.refresh()

    def refresh(self) -> None:
        """Refresh data."""
        params_reminders = dict(self.params)
        params_reminders.update(
            {
                "clientVersion": "4.0",
                "lang": "en-us",
                "usertz": get_localzone_name(),
            }
        )

        # Open reminders
        req = self.session.get(
            f"{self.service_root}/rd/startup", params=params_reminders
        )

        data = req.json()

        self.lists = {}
        self.collections = {}
        for collection in data["Collections"]:
            temp = []
            self.collections[collection["title"]] = {
                "guid": collection["guid"],
                "ctag": collection["ctag"],
            }
            for reminder in data["Reminders"]:
                if reminder["pGuid"] != collection["guid"]:
                    continue

                if reminder.get("dueDate"):
                    due = datetime(
                        reminder["dueDate"][0],
                        reminder["dueDate"][1],
                        reminder["dueDate"][2],
                        reminder["dueDate"][3],
                        reminder["dueDate"][4],
                        reminder["dueDate"][5],
                    )
                else:
                    due = None

                temp.append(
                    {
                        "title": reminder["title"],
                        "desc": reminder.get("description"),
                        "due": due,
                    }
                )
            self.lists[collection["title"]] = temp

    def post(self, title, description="", collection=None, due_date=None):
        """Adds a new reminder."""
        pguid = "tasks"
        if collection and collection in self.collections:
            pguid = self.collections[collection]["guid"]

        params_reminders = dict(self.params)
        params_reminders.update(
            {"clientVersion": "4.0", "lang": "en-us", "usertz": get_localzone_name()}
        )

        due_dates = None
        if due_date:
            due_dates = [
                int(f"{due_date.year}{due_date.month:02}{due_date.day:02}"),
                due_date.year,
                due_date.month,
                due_date.day,
                due_date.hour,
                due_date.minute,
            ]

        req = self.session.post(
            f"{self.service_root}/rd/reminders/tasks",
            json={
                "Reminders": {
                    "title": title,
                    "description": description,
                    "pGuid": pguid,
                    "etag": None,
                    "order": None,
                    "priority": 0,
                    "recurrence": None,
                    "alarms": [],
                    "startDate": None,
                    "startDateTz": None,
                    "startDateIsAllDay": False,
                    "completedDate": None,
                    "dueDate": due_dates,
                    "dueDateIsAllDay": False,
                    "lastModifiedDate": None,
                    "createdDate": None,
                    "isFamily": None,
                    "createdDateExtended": int(time.time() * 1000),
                    "guid": str(uuid.uuid4()),
                },
                "ClientState": {"Collections": list(self.collections.values())},
            },
            params=params_reminders,
        )
        return req.ok
