/* Table_dataSets.cpp
	 *
 * Copyright (C) 1997-2022 David Weenink
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This code is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this work. If not, see <http://www.gnu.org/licenses/>.
 */

/*	speaker type (m|w|c), sex(m|f), id, vowel_number, vowel_label
	F0, F1, F2, F3
*/

#include "NUM2.h"
#include "Table_extensions.h"

/*
	The Peterson & Barney data were once (1991) obtained by me (djmw) as a compressed tar-file
	by anonymous ftp from ftp://linc.cis.upenn.edu/pub,
	However, this site appears no longer to be an anonymous ftp site.
	The compressed tar file contained two files: a header file 'pb.header'
	and a data file 'verified_pb.data'.
	The header file reads:

	"This file contains the vowel formant data reported by Gordon E.
	Peterson and Harold L. Barney in their classic paper, "Control methods
	used in a study of the vowels", JASA 24(2) 175-184, 1952. This data
	was supplied in printed form by Ignatius Mattingly, April, 1990.

	The data consists of the formant values F0, F1, F2, and F3 for each of
	two repetitions of ten vowels by 76 speakers (1520 utterances). The
	vowels were pronounced in isolated words consisting of hVd. Of the
	speakers, 33 were men, 28 were women and 15 were children. Dr.
	Mattingly reported that he obtained from G. Peterson the information
	that children speakers 62, 63, 65, 66, 67, 68, 73 and 76 were female.

	The data are organized by speaker type, speaker, and vowel into 1520
	lines of 8 fields. The fields are: Speaker Type, Speaker Number,
	Phoneme Number, Phoneme Label, F0, F1, F2 and F3. The speaker types
	are type 1 (men), type 2 (women) and type 3 (children)."
*/

autoTable Table_create_petersonBarney1952 () {
	const integer nrows = 1520, ncols = 9;
	const conststring32 columnLabels [9] = {U"Type", U"Sex", U"Speaker", U"Vowel", U"IPA", U"F0", U"F1", U"F2", U"F3"};
	const conststring32 type [3] = {U"m", U"w", U"c"};
	/*
		Wrong order before 20080125
		char32 *vowel [10] = {U"iy", U"ih", U"eh", U"ae", U"aa", U"ao", U"uh", U"uw", U"ah", U"er"};
		char32 *ipa [10] = {U"i", U"\\ic", U"\\ep", U"\\ae", U"\\as", U"\\ct", U"\\hs", U"u",
			U"\\vt", U"\\er\\hr"};
	*/
	const conststring32 vowel [10] = {U"iy", U"ih", U"eh", U"ae", U"ah", U"aa", U"ao", U"uh", U"uw", U"er"};
	/*
		Watrous IPA symbols
		char32 *ipa [10] = {U"i", U"\\ic", U"e", U"\\ae", U"\\vt", U"\\as", U"o", U"\\hs", U"u", U"\\er"};
	*/
	// P & B IPA symbols
	const conststring32 ipa [10] = {U"i", U"\\ic", U"\\ef", U"\\ae", U"\\vt", U"\\as", U"\\ct", U"\\hs", U"u", U"\\er\\hr"};
	const conststring32 sex [2] = {U"m", U"f"};
	const struct pbdatum {
		short star; // was there a * in front of the vowel-type?
		short f [4];	// f0, f1, f2, f3
	} pbdata [] = {
		{0, {160, 240, 2280, 2850}},
		{0, {186, 280, 2400, 2790}},
		{0, {203, 390, 2030, 2640}},
		{0, {192, 310, 1980, 2550}},
		{0, {161, 490, 1870, 2420}},
		{1, {155, 570, 1700, 2600}},
		{1, {140, 560, 1820, 2660}},
		{0, {180, 630, 1700, 2550}},
		{0, {144, 590, 1250, 2620}},
		{0, {148, 620, 1300, 2530}},
		{0, {148, 740, 1070, 2490}},
		{0, {170, 800, 1060, 2640}},
		{1, {161, 600,  970, 2280}},
		{1, {158, 660,  980, 2220}},
		{0, {163, 440, 1120, 2210}},
		{0, {190, 400, 1070, 2280}},
		{0, {160, 240, 1040, 2150}},
		{0, {157, 270,  930, 2280}},
		{0, {177, 370, 1520, 1670}},
		{0, {164, 460, 1330, 1590}},
		{0, {147, 220, 2220, 2910}},
		{0, {148, 210, 2360, 3250}},
		{0, {141, 410, 1890, 2680}},
		{0, {139, 420, 1850, 2500}},
		{0, {136, 500, 1760, 2590}},
		{1, {135, 510, 1710, 2380}},
		{0, {128, 690, 1610, 2560}},
		{0, {131, 700, 1690, 2580}},
		{1, {140, 650, 1080, 2420}},
		{1, {125, 625, 1060, 2490}},
		{0, {140, 650, 1040, 2450}},
		{0, {136, 670, 1100, 2430}},
		{0, {149, 580,  580, 2470}},
		{0, {140, 560,  560, 2410}},
		{0, {145, 450,  940, 1910}},
		{0, {141, 410,  830, 2240}},
		{0, {140, 280,  650, 3300}},
		{0, {137, 260,  660, 3300}},
		{0, {145, 510, 1210, 1570}},
		{0, {145, 510, 1130, 1510}},
		{0, {105, 250, 2180, 2680}},
		{0, {111, 244, 2300, 2780}},
		{0, {100, 400, 1930, 2610}},
		{0, {104, 400, 1990, 2700}},
		{0, {100, 550, 1810, 2500}},
		{0,  {95, 540, 1810, 2480}},
		{0,  {93, 630, 1710, 2400}},
		{0,  {94, 658, 1755, 2305}},
		{1, {100, 600, 1200, 2320}},
		{0, {105, 612, 1160, 2350}},
		{0,  {91, 640, 1080, 2140}},
		{0,  {94, 720, 1090, 2230}},
		{0,  {92, 550,  870, 2300}},
		{0, {120, 540,  840, 2280}},
		{0, {114, 460, 1150, 2290}},
		{0, {114, 456, 1030, 2300}},
		{0, {112, 340,  950, 2240}},
		{1, {112, 326,  900, 2190}},
		{0, {100, 500, 1370, 1780}},
		{0, {106, 530, 1330, 1800}},
		{0, {150, 300, 2240, 3400}},
		{0, {156, 280, 2450, 3200}},
		{1, {156, 450, 1960, 2400}},
		{0, {146, 440, 2050, 2360}},
		{0, {130, 570, 1780, 2410}},
		{0, {150, 555, 1890, 2440}},
		{0, {125, 750, 1610, 2340}},
		{0, {136, 770, 1580, 2350}},
		{0, {132, 660, 1200, 2330}},
		{1, {150, 675, 1140, 2380}},
		{0, {125, 750, 1100, 2550}},
		{0, {138, 800, 1120, 2500}},
		{0, {143, 540,  850, 2320}},
		{0, {150, 555,  890, 2370}},
		{0, {136, 460,  960, 2210}},
		{0, {156, 460, 1000, 2350}},
		{0, {140, 380,  950, 2050}},
		{0, {148, 385,  850, 2330}},
		{0, {150, 590, 1400, 1840}},
		{0, {145, 555, 1430, 1730}},
		{0, {140, 310, 2310, 2820}},
		{0, {131, 260, 2250, 2850}},
		{0, {137, 440, 2060, 2640}},
		{0, {134, 430, 1880, 2450}},
		{0, {140, 580, 1910, 2500}},
		{0, {137, 550, 1770, 2400}},
		{0, {143, 830, 1720, 2180}},
		{0, {135, 750, 1690, 2320}},
		{0, {136, 630, 1300, 1950}},
		{0, {130, 650, 1170, 2000}},
		{0, {131, 760, 1220, 2140}},
		{1, {126, 720, 1260, 2020}},
		{0, {136, 540,  970, 1980}},
		{0, {124, 550,  880, 1950}},
		{0, {133, 470, 1040, 1990}},
		{0, {132, 490,  990, 1920}},
		{0, {141, 380,  950, 2140}},
		{0, {133, 330,  800, 2130}},
		{0, {143, 560, 1510, 1800}},
		{0, {136, 510, 1460, 1700}},
		{0, {125, 312, 2350, 2800}},
		{0, {119, 330, 2430, 2870}},
		{0, {133, 420, 2000, 2660}},
		{0, {125, 313, 2000, 2750}},
		{0, {120, 600, 1860, 2500}},
		{0, {114, 570, 1830, 2570}},
		{0, {119, 676, 1670, 2540}},
		{0, {125, 725, 1687, 2500}},
		{1, {118, 680, 1150, 2560}},
		{0, {125, 726, 1270, 2560}},
		{0, {125, 740, 1100, 2680}},
		{1, {113, 670,  960, 2650}},
		{1, {120, 660, 1030, 2690}},
		{1, {125, 720,  960, 2700}},
		{0, {120, 456, 1080, 2520}},
		{0, {120, 450, 1140, 2600}},
		{0, {125, 313,  838, 2340}},
		{0, {125, 288,  938, 2450}},
		{0, {120, 503, 1305, 1775}},
		{0, {120, 505, 1320, 1750}},
		{0, {186, 320, 2320, 3120}},
		{0, {172, 310, 2280, 3020}},
		{1, {167, 470, 2000, 2660}},
		{0, {170, 410, 2040, 2715}},
		{1, {167, 630, 1900, 2860}},
		{0, {146, 614, 1840, 2770}},
		{0, {143, 740, 1800, 2450}},
		{0, {162, 775, 1810, 2200}},
		{1, {167, 620, 1240, 2410}},
		{0, {160, 640, 1250, 2400}},
		{1, {162, 650,  970, 2580}},
		{0, {163, 650,  980, 2350}},
		{0, {145, 430,  720, 2450}},
		{0, {171, 510,  800, 2500}},
		{1, {170, 460, 1120, 2150}},
		{1, {170, 493, 1120, 2300}},
		{0, {175, 380, 1040, 2260}},
		{1, {200, 400, 1000, 2350}},
		{0, {167, 570, 1300, 1750}},
		{0, {157, 565, 1370, 1710}},
		{1, {105, 230, 2480, 3200}},
		{0, {109, 218, 2380, 3100}},
		{1, {110, 320, 2200, 2680}},
		{0, {103, 206, 2130, 2570}},
		{1, {107, 430, 2100, 2630}},
		{1, {105, 515, 1760, 2470}},
		{1, {107, 514, 2060, 2600}},
		{1, {106, 552, 1820, 2500}},
		{0, {108, 640, 1300, 2300}},
		{0, {104, 624, 1350, 2410}},
		{1, {111, 714, 1170, 2420}},
		{1,  {97, 650, 1150, 2350}},
		{1, {107, 590,  965, 2500}},
		{0, {109, 578,  970, 2460}},
		{0, {111, 467, 1110, 2400}},
		{0, {105, 475, 1220, 2310}},
		{0, {107, 270,  910, 2200}},
		{0, {108, 260,  975, 2320}},
		{0, {107, 460, 1400, 1790}},
		{0, {103, 425, 1410, 1760}},
		{0, {175, 316, 2200, 2800}},
		{0, {175, 280, 2275, 2775}},
		{0, {167, 450, 1820, 2475}},
		{0, {167, 434, 1850, 2425}},
		{0, {157, 582, 1725, 2375}},
		{0, {158, 586, 1800, 2425}},
		{0, {150, 600, 1750, 2375}},
		{1, {145, 582, 1775, 2375}},
		{1, {145, 626, 1125, 2200}},
		{0, {160, 641, 1120, 2225}},
		{0, {144, 708, 1054, 2420}},
		{1, {150, 705, 1050, 2375}},
		{0, {146, 614,  848, 2200}},
		{0, {143, 600,  860, 2175}},
		{0, {167, 500, 1000, 2325}},
		{0, {167, 500, 1000, 2325}},
		{0, {167, 334, 1150, 2200}},
		{0, {183, 312, 1020, 2300}},
		{0, {157, 518, 1305, 1570}},
		{0, {157, 504, 1210, 1510}},
		{0, {129, 260, 2260, 2820}},
		{0, {125, 250, 2200, 2825}},
		{0, {146, 400, 2040, 2500}},
		{0, {144, 389, 2000, 2425}},
		{0, {126, 500, 1870, 2500}},
		{0, {125, 500, 1775, 2450}},
		{0, {110, 660, 1650, 2500}},
		{0, {120, 624, 1700, 2475}},
		{0, {122, 650, 1220, 2550}},
		{0, {120, 672, 1260, 2500}},
		{0, {114, 750, 1080, 2680}},
		{0, {114, 777, 1026, 2625}},
		{0, {115, 580,  800, 2650}},
		{0, {117, 585,  819, 2625}},
		{1, {140, 480,  950, 2500}},
		{0, {127, 461,  993, 2350}},
		{0, {140, 280,  950, 2300}},
		{0, {133, 266,  920, 2300}},
		{0, {128, 500, 1340, 1700}},
		{0, {133, 532, 1275, 1600}},
		{0, {146, 248, 2225, 3100}},
		{0, {140, 238, 2175, 3075}},
		{0, {150, 405, 1925, 2550}},
		{0, {138, 416, 1940, 2600}},
		{1, {147, 588, 1790, 2500}},
		{1, {133, 586, 1725, 2650}},
		{0, {145, 725, 1700, 2425}},
		{0, {127, 710, 1650, 2220}},
		{1, {136, 586, 1078, 2300}},
		{0, {136, 627, 1038, 2360}},
		{1, {145, 725, 1046, 2325}},
		{1, {131, 746, 1018, 2300}},
		{0, {140, 560,  840, 2500}},
		{1, {140, 560,  924, 2350}},
		{0, {150, 495, 1080, 2275}},
		{0, {143, 430, 1030, 2275}},
		{0, {162, 290,  760, 2300}},
		{0, {157, 315,  850, 2025}},
		{0, {150, 511, 1561, 1876}},
		{0, {138, 530, 1450, 1887}},
		{0, {110, 220, 2410, 3000}},
		{0, {125, 240, 2440, 3280}},
		{0, {120, 450, 1880, 2450}},
		{0, {118, 380, 1930, 2420}},
		{0, {115, 560, 1650, 2300}},
		{0, {123, 560, 1720, 2300}},
		{0, {110, 680, 1720, 2330}},
		{0, {133, 630, 1680, 2280}},
		{1, {110, 560, 1430, 2250}},
		{1, {120, 560, 1390, 2240}},
		{1, {108, 800, 1330, 2260}},
		{0, {110, 740, 1240, 2280}},
		{1, {120, 600,  920, 2080}},
		{1, {133, 580,  910, 2000}},
		{0, {130, 400, 1200, 2210}},
		{0, {110, 420, 1230, 2230}},
		{0, {122, 300,  900, 2130}},
		{0, {123, 260, 1010, 2240}},
		{0, {125, 400, 1450, 1650}},
		{0, {128, 360, 1410, 1640}},
		{0, {142, 290, 2290, 2600}},
		{0, {135, 260, 2290, 2700}},
		{0, {132, 390, 1950, 2550}},
		{1, {135, 400, 1900, 2450}},
		{1, {124, 490, 1740, 2500}},
		{1, {125, 500, 1780, 2430}},
		{0, {125, 660, 1630, 2500}},
		{0, {132, 670, 1630, 2380}},
		{0, {140, 600, 1220, 2530}},
		{0, {125, 600, 1210, 2430}},
		{1, {125, 680, 1120, 2630}},
		{0, {128, 670, 1100, 2700}},
		{0, {127, 510,  720, 2450}},
		{0, {120, 480,  710, 2540}},
		{1, {133, 380,  910, 2350}},
		{0, {140, 440, 1030, 2400}},
		{0, {127, 350,  720, 2750}},
		{0, {140, 380,  740, 2880}},
		{0, {128, 430, 1370, 1610}},
		{0, {135, 440, 1360, 1600}},
		{0, {114, 228, 2350, 2860}},
		{0, {118, 220, 2350, 2920}},
		{1, {110, 407, 2070, 2500}},
		{1, {112, 420, 1900, 2450}},
		{0, {106, 445, 2020, 2420}},
		{0, {115, 470, 2020, 2500}},
		{0, {103, 721, 1680, 2400}},
		{0, {109, 750, 1710, 2440}},
		{1, {104, 552, 1122, 2500}},
		{1, {115, 580, 1150, 2600}},
		{0,  {98, 686, 1078, 2570}},
		{1, {103, 700, 1050, 2680}},
		{0, {102, 560,  665, 2620}},
		{0, {106, 550,  650, 2700}},
		{0, {112, 448,  980, 2370}},
		{0, {104, 410,  940, 2370}},
		{0, {116, 232,  696, 2200}},
		{0, {117, 222,  665, 2080}},
		{0, {120, 432, 1300, 1400}},
		{0, {111, 420, 1300, 1570}},
		{0, {121, 230, 2100, 2850}},
		{0, {118, 240, 2000, 2980}},
		{0, {130, 365, 1900, 2340}},
		{0, {119, 300, 2040, 2560}},
		{1, {112, 440, 1980, 2310}},
		{1, {120, 410, 2050, 2500}},
		{1, {133, 620, 1710, 2110}},
		{0, {124, 660, 1800, 2150}},
		{0, {120, 660, 1000, 2380}},
		{0, {110, 660,  960, 2450}},
		{1, {122, 600,  830, 2250}},
		{1, {119, 620,  820, 2400}},
		{0, {117, 500,  620, 2250}},
		{0, {106, 550,  700, 2550}},
		{0, {140, 390,  730, 2180}},
		{0, {130, 360,  740, 2200}},
		{0, {131, 260,  720, 2100}},
		{0, {132, 260,  740, 2040}},
		{0, {125, 450, 1230, 1600}},
		{0, {127, 460, 1300, 1650}},
		{0, {150, 300, 2355, 3250}},
		{0, {150, 300, 2460, 3280}},
		{0, {160, 385, 2242, 2805}},
		{0, {150, 407, 2250, 2780}},
		{0, {140, 504, 2090, 2720}},
		{0, {146, 543, 1980, 2640}},
		{0, {133, 680, 1958, 2542}},
		{0, {141, 708, 1840, 2535}},
		{0, {150, 675, 1320, 2550}},
		{0, {150, 704, 1393, 2550}},
		{0, {137, 825, 1168, 2750}},
		{0, {135, 840, 1210, 2680}},
		{1, {143, 671, 1000, 2670}},
		{1, {147, 690,  968, 2660}},
		{0, {143, 443, 1273, 2430}},
		{0, {153, 459, 1286, 2410}},
		{0, {146, 395, 1300, 2160}},
		{0, {153, 400, 1320, 2150}},
		{0, {140, 532, 1500, 1890}},
		{0, {146, 538, 1460, 1818}},
		{0, {120, 264, 2290, 2700}},
		{0, {128, 256, 2305, 2635}},
		{0, {112, 380, 1880, 2440}},
		{0, {115, 346, 1930, 2390}},
		{0, {100, 510, 1780, 2300}},
		{0, {108, 520, 1730, 2275}},
		{0, {100, 630, 1770, 2350}},
		{0, {105, 630, 1642, 2170}},
		{0, {103, 601, 1273, 2130}},
		{1, {105, 590, 1283, 2150}},
		{1, {100, 750, 1150, 2440}},
		{0,  {95, 703, 1092, 2320}},
		{0,  {97, 565,  780, 2350}},
		{0, {106, 584,  849, 2460}},
		{0, {105, 420, 1100, 2140}},
		{0, {111, 422, 1200, 2175}},
		{0, {117, 315, 1080, 2260}},
		{0, {125, 326, 1125, 2210}},
		{0, {111, 444, 1300, 1625}},
		{0, {109, 469, 1288, 1600}},
		{0, {124, 210, 2100, 3090}},
		{0, {130, 220, 2080, 3180}},
		{0, {128, 280, 2000, 2710}},
		{0, {130, 310, 1950, 2670}},
		{0, {121, 470, 1910, 2580}},
		{1, {129, 490, 1930, 2650}},
		{0, {116, 640, 1620, 2200}},
		{0, {118, 650, 1580, 2360}},
		{0, {121, 610, 1100, 2230}},
		{0, {126, 620, 1120, 2330}},
		{1, {118, 700, 1100, 2240}},
		{1, {120, 670, 1100, 2220}},
		{0, {122, 460,  720, 2180}},
		{0, {118, 470,  690, 2200}},
		{0, {129, 320,  770, 1860}},
		{1, {130, 310,  790, 1920}},
		{0, {140, 210,  670, 1900}},
		{0, {148, 240,  730, 1850}},
		{0, {128, 390, 1320, 1550}},
		{0, {124, 420, 1240, 1510}},
		{0, {129, 190, 2650, 3280}},
		{0, {135, 190, 2700, 3170}},
		{0, {132, 370, 1750, 2700}},
		{0, {130, 370, 1800, 2750}},
		{1, {122, 370, 1680, 2560}},
		{1, {125, 375, 1700, 2500}},
		{1, {121, 550, 1570, 2600}},
		{1, {120, 530, 1610, 2650}},
		{1, {118, 570, 1050, 2500}},
		{1, {125, 590, 1100, 2480}},
		{1, {112, 640,  970, 2870}},
		{1, {122, 670,  980, 2900}},
		{1, {113, 560,  860, 2900}},
		{0, {121, 570,  820, 2820}},
		{1, {125, 350, 1000, 2500}},
		{1, {130, 380,  920, 2370}},
		{0, {130, 250, 1000, 2100}},
		{0, {140, 210,  960, 1940}},
		{0, {130, 360, 1300, 1920}},
		{0, {133, 370, 1300, 1760}},
		{0, {127, 250, 2180, 2660}},
		{0, {131, 260, 2210, 2780}},
		{0, {121, 400, 1900, 2440}},
		{0, {122, 350, 1980, 2480}},
		{1, {116, 560, 1670, 2310}},
		{1, {124, 530, 1700, 2380}},
		{0, {120, 680, 1470, 2280}},
		{0, {119, 620, 1580, 2320}},
		{0, {120, 620, 1100, 2390}},
		{1, {125, 640, 1110, 2370}},
		{1, {115, 630,  980, 2330}},
		{1, {121, 670,  940, 2380}},
		{0, {112, 560,  790, 2480}},
		{0, {120, 610,  840, 2420}},
		{0, {121, 360,  860, 2200}},
		{0, {120, 400,  840, 2200}},
		{0, {140, 280,  670, 2140}},
		{0, {126, 250,  720, 2190}},
		{0, {120, 480, 1410, 1760}},
		{0, {121, 470, 1330, 1700}},
		{0, {155, 280, 2400, 2910}},
		{0, {150, 300, 2320, 2960}},
		{1, {142, 410, 2060, 2680}},
		{0, {150, 450, 2050, 2670}},
		{0, {135, 540, 1900, 2530}},
		{0, {135, 540, 1920, 2520}},
		{0, {138, 620, 1800, 2440}},
		{0, {140, 690, 1820, 2480}},
		{1, {150, 630, 1200, 2600}},
		{0, {140, 680, 1290, 2600}},
		{0, {145, 740, 1110, 2500}},
		{1, {143, 700, 1060, 2720}},
		{1, {146, 600,  970, 2570}},
		{0, {138, 650,  880, 2660}},
		{0, {142, 430, 1130, 2440}},
		{0, {143, 430, 1150, 2420}},
		{0, {142, 280,  990, 2330}},
		{0, {145, 290, 1000, 2300}},
		{0, {150, 420, 1350, 1600}},
		{0, {150, 450, 1350, 1600}},
		{0, {135, 300, 2300, 2800}},
		{0, {135, 350, 2240, 2760}},
		{1, {136, 410, 2200, 2680}},
		{1, {138, 440, 2080, 2520}},
		{1, {133, 580, 1870, 2320}},
		{1, {127, 520, 1900, 2400}},
		{0, {130, 760, 1920, 2480}},
		{0, {132, 670, 1850, 2560}},
		{1, {139, 810, 1110, 2100}},
		{1, {131, 770, 1150, 2100}},
		{1, {141, 700, 1040, 2120}},
		{0, {125, 750, 1160, 2080}},
		{1, {133, 670,  920, 2240}},
		{1, {142, 570,  850, 2250}},
		{0, {140, 550,  970, 2200}},
		{1, {141, 490,  870, 2240}},
		{0, {150, 300,  600, 2300}},
		{0, {148, 230,  570, 2100}},
		{0, {140, 560, 1520, 2100}},
		{0, {140, 540, 1570, 2050}},
		{0, {125, 240, 2100, 2900}},
		{0, {119, 240, 2150, 2860}},
		{0, {130, 380, 1870, 2450}},
		{0, {120, 430, 1710, 2350}},
		{1, {119, 580, 1770, 2500}},
		{1, {117, 570, 1750, 2400}},
		{1, {115, 760, 1580, 2440}},
		{0, {110, 715, 1500, 2300}},
		{1, {124, 620,  880, 2500}},
		{1, {124, 650, 1000, 2520}},
		{0, {119, 710,  950, 2520}},
		{1, {120, 690,  960, 2520}},
		{0, {125, 460,  610, 2500}},
		{0, {120, 470,  710, 2500}},
		{0, {125, 390,  900, 2100}},
		{0, {125, 460,  920, 2140}},
		{0, {125, 250,  690, 2080}},
		{0, {130, 270,  650, 2050}},
		{0, {122, 540, 1280, 1720}},
		{0, {118, 510, 1280, 1650}},
		{0, {148, 280, 2450, 2700}},
		{0, {160, 288, 2500, 2880}},
		{0, {160, 400, 2080, 2530}},
		{0, {153, 384, 2110, 2500}},
		{1, {138, 590, 1900, 2200}},
		{1, {153, 583, 1840, 2250}},
		{0, {145, 680, 1850, 2400}},
		{0, {140, 685, 1780, 2160}},
		{0, {143, 660, 1370, 2110}},
		{1, {145, 680, 1300, 2100}},
		{1, {140, 760, 1260, 2120}},
		{0, {135, 770, 1140, 2020}},
		{0, {145, 500,  800, 1850}},
		{0, {132, 600, 1000, 2000}},
		{0, {157, 380, 1060, 1950}},
		{0, {150, 470, 1220, 2150}},
		{0, {162, 324,  800, 2220}},
		{0, {139, 290,  800, 2150}},
		{0, {150, 560, 1350, 1780}},
		{0, {150, 600, 1470, 1820}},
		{0, {110, 250, 2190, 3000}},
		{0, {106, 254, 2085, 2890}},
		{0, {111, 330, 1967, 2670}},
		{0, {108, 430, 1940, 2590}},
		{0, {116, 464, 2100, 2700}},
		{0, {105, 504, 1995, 2780}},
		{0,  {94, 595, 1900, 2700}},
		{0, {100, 670, 1860, 2500}},
		{0,  {96, 620, 1200, 2420}},
		{0, {105, 630, 1127, 2420}},
		{0, {100, 750, 1160, 2360}},
		{0,  {96, 740, 1155, 2330}},
		{0, {101, 460,  740, 2300}},
		{1, {105, 494,  789, 2420}},
		{0, {113, 400, 1020, 2200}},
		{0, {128, 450, 1028, 2160}},
		{0, {140, 392, 1000, 2120}},
		{0, {116, 350,  898, 2140}},
		{0, {117, 547, 1340, 1688}},
		{0, {128, 512, 1280, 1570}},
		{0, {123, 246, 2185, 2730}},
		{0, {133, 267, 2280, 2800}},
		{0, {140, 420, 2300, 2800}},
		{0, {120, 384, 2110, 2620}},
		{0, {120, 480, 1920, 2540}},
		{1, {112, 551, 1788, 2450}},
		{0, {114, 628, 1837, 2570}},
		{1, {111, 622, 1890, 2560}},
		{0, {114, 628, 1254, 2470}},
		{0, {114, 617, 1255, 2480}},
		{1, {117, 690, 1072, 2660}},
		{1, {103, 630, 1000, 2530}},
		{0, {117, 510,  700, 2650}},
		{0, {120, 504,  756, 2540}},
		{0, {122, 465,  990, 2440}},
		{0, {125, 462,  976, 2450}},
		{0, {120, 324,  708, 2440}},
		{0, {157, 387,  786, 2518}},
		{0, {122, 488, 1468, 1712}},
		{0, {118, 472, 1465, 1725}},
		{0, {138, 275, 2060, 2800}},
		{0, {136, 270, 2020, 2790}},
		{0, {133, 349, 2030, 2760}},
		{0, {136, 340, 1940, 2560}},
		{1, {120, 444, 1800, 2500}},
		{1, {127, 380, 1800, 2440}},
		{0, {125, 688, 1600, 2300}},
		{0, {122, 660, 1570, 2380}},
		{1, {128, 565, 1157, 2310}},
		{0, {130, 550, 1150, 2250}},
		{0, {125, 712, 1024, 2250}},
		{0, {125, 670, 1080, 2300}},
		{0, {125, 550,  913, 2360}},
		{0, {126, 550,  890, 2280}},
		{0, {128, 360, 1028, 2160}},
		{1, {140, 390, 1060, 2150}},
		{0, {133, 294,  930, 2050}},
		{0, {140, 280, 1000, 2160}},
		{0, {125, 440, 1250, 1625}},
		{0, {130, 480, 1160, 1520}},
		{0, {125, 320, 2160, 2900}},
		{0, {133, 267, 2230, 3000}},
		{0, {115, 440, 1750, 2400}},
		{0, {116, 390, 1780, 2450}},
		{0, {117, 525, 1800, 2480}},
		{0, {110, 520, 1750, 2390}},
		{0, {111, 660, 1600, 2400}},
		{0, {120, 720, 1680, 2430}},
		{0, {117, 600, 1250, 2300}},
		{1, {125, 575, 1170, 2240}},
		{1, {111, 730, 1160, 2340}},
		{0, {117, 860, 1280, 2470}},
		{0, {114, 560,  810, 2290}},
		{0, {116, 584,  840, 2280}},
		{0, {130, 455,  970, 2140}},
		{0, {120, 456, 1040, 2038}},
		{0, {125, 350,  820, 2130}},
		{0, {128, 366,  772, 2058}},
		{1, {111, 450, 1420, 1870}},
		{0, {118, 472, 1430, 1840}},
		{0, {133, 333, 2305, 3200}},
		{0, {131, 326, 2260, 3030}},
		{1, {125, 375, 2188, 2750}},
		{1, {133, 400, 2150, 2680}},
		{1, {125, 500, 1980, 2480}},
		{1, {150, 480, 1950, 2340}},
		{0, {116, 640, 1710, 2450}},
		{0, {123, 615, 1720, 2220}},
		{0, {116, 583, 1110, 2360}},
		{1, {117, 608, 1120, 2700}},
		{0, {111, 777, 1170, 2600}},
		{0, {114, 750, 1175, 2820}},
		{1, {105, 630,  891, 2519}},
		{1, {114, 572,  924, 2660}},
		{0, {125, 438,  975, 2300}},
		{0, {140, 420,  938, 2300}},
		{0, {133, 333,  800, 2130}},
		{0, {140, 320,  840, 2150}},
		{0, {120, 480, 1320, 1870}},
		{0, {127, 483, 1335, 1844}},
		{0, {166, 267, 2300, 2940}},
		{0, {156, 220, 2300, 2900}},
		{0, {154, 431, 2040, 2460}},
		{0, {155, 360, 2010, 2400}},
		{1, {150, 565, 1950, 2500}},
		{0, {180, 540, 2000, 2450}},
		{1, {143, 600, 2000, 2570}},
		{0, {138, 590, 1950, 2460}},
		{1, {157, 630, 1140, 2200}},
		{1, {186, 630, 1170, 2280}},
		{1, {146, 730, 1048, 2450}},
		{0, {155, 730, 1130, 2320}},
		{1, {150, 600,  900, 2400}},
		{1, {178, 640,  890, 2280}},
		{1, {160, 448,  960, 2200}},
		{0, {196, 450, 1000, 2180}},
		{0, {167, 333,  835, 2170}},
		{0, {198, 280,  750, 2170}},
		{0, {163, 488, 1300, 1600}},
		{0, {163, 490, 1380, 1620}},
		{0, {120, 312, 2380, 2900}},
		{0, {120, 300, 2350, 3000}},
		{0, {140, 490, 2000, 2620}},
		{1, {140, 490, 1960, 2600}},
		{0, {125, 640, 2000, 2620}},
		{0, {111, 555, 1870, 2540}},
		{0, {112, 697, 1610, 2540}},
		{0, {114, 684, 1634, 2510}},
		{0, {115, 633, 1260, 2530}},
		{1, {120, 660, 1213, 2460}},
		{0, {112, 730, 1203, 2700}},
		{0, {107, 752, 1125, 2620}},
		{0, {108, 507,  755, 2420}},
		{0, {116, 538,  816, 2450}},
		{0, {114, 456, 1040, 2300}},
		{0, {120, 480, 1120, 2160}},
		{1, {123, 344,  960, 2150}},
		{0, {125, 350, 1000, 2250}},
		{0, {112, 539, 1370, 1800}},
		{0, {117, 549, 1353, 1728}},
		{0, {146, 292, 2500, 3150}},
		{0, {133, 266, 2370, 3100}},
		{0, {143, 372, 2220, 2640}},
		{0, {131, 350, 2130, 2610}},
		{1, {133, 574, 1840, 2260}},
		{1, {133, 563, 1960, 2450}},
		{0, {125, 650, 1738, 2400}},
		{0, {130, 663, 1820, 2400}},
		{0, {137, 600, 1370, 2180}},
		{1, {125, 625, 1312, 2250}},
		{0, {133, 735, 1070, 2100}},
		{0, {117, 713, 1180, 2200}},
		{0, {125, 625,  875, 2180}},
		{1, {115, 700, 1000, 2250}},
		{0, {150, 420, 1100, 2000}},
		{0, {140, 420, 1120, 2100}},
		{0, {125, 350,  980, 2200}},
		{0, {133, 320,  918, 2100}},
		{0, {143, 554, 1480, 1800}},
		{0, {128, 484, 1505, 1890}},
		{0, {143, 286, 2415, 2860}},
		{0, {150, 300, 2415, 2860}},
		{1, {140, 400, 1980, 2500}},
		{1, {145, 407, 2095, 2620}},
		{1, {125, 525, 1988, 2610}},
		{0, {144, 553, 1935, 2530}},
		{0, {133, 640, 1773, 2490}},
		{0, {133, 640, 1840, 2560}},
		{0, {143, 672, 1272, 2640}},
		{0, {146, 658, 1241, 2560}},
		{0, {130, 780, 1170, 2640}},
		{0, {131, 788, 1115, 2645}},
		{1, {138, 633,  891, 2500}},
		{1, {150, 600,  935, 2550}},
		{0, {175, 490, 1102, 2420}},
		{0, {154, 492, 1077, 2306}},
		{1, {160, 320,  960, 2240}},
		{0, {160, 320,  960, 2290}},
		{0, {143, 543, 1310, 1643}},
		{0, {145, 508, 1309, 1600}},
		{1, {230, 370, 2670, 3100}},
		{0, {234, 390, 2760, 3060}},
		{0, {234, 468, 2330, 2930}},
		{0, {205, 410, 2380, 2950}},
		{0, {190, 550, 2200, 2880}},
		{0, {191, 570, 2100, 3040}},
		{0, {200, 800, 1980, 2810}},
		{0, {192, 860, 1920, 2850}},
		{0, {227, 635, 1200, 3250}},
		{0, {200, 700, 1200, 3100}},
		{1, {210, 880, 1240, 2870}},
		{0, {188, 830, 1200, 2880}},
		{0, {207, 570,  830, 3300}},
		{1, {200, 700, 1000, 3130}},
		{0, {240, 410,  940, 3040}},
		{0, {225, 450,  970, 3190}},
		{0, {238, 480,  955, 2960}},
		{1, {208, 395,  810, 2900}},
		{0, {200, 500, 1850, 2100}},
		{0, {200, 560, 1750, 2100}},
		{0, {225, 270, 2760, 3550}},
		{0, {240, 290, 2700, 3350}},
		{0, {245, 460, 2500, 3220}},
		{0, {220, 410, 2400, 3240}},
		{0, {220, 620, 2300, 3200}},
		{0, {210, 630, 2300, 3170}},
		{0, {220, 820, 2180, 2850}},
		{0, {195, 740, 2120, 3070}},
		{0, {240, 800, 1300, 2900}},
		{0, {225, 760, 1400, 2830}},
		{0, {214, 850, 1120, 2620}},
		{0, {190, 880, 1220, 2850}},
		{0, {228, 460,  900, 2830}},
		{1, {222, 440,  880, 2850}},
		{0, {250, 500, 1040, 2750}},
		{0, {245, 490, 1000, 2720}},
		{0, {250, 400,  940, 2720}},
		{0, {245, 410,  860, 2700}},
		{0, {225, 440, 1560, 1750}},
		{0, {210, 420, 1600, 1750}},
		{0, {210, 290, 2700, 3020}},
		{0, {215, 280, 2630, 3240}},
		{1, {211, 420, 2300, 2950}},
		{0, {211, 420, 2220, 2980}},
		{1, {207, 640, 2120, 2900}},
		{1, {221, 700, 2000, 2900}},
		{0, {212, 1000, 1830, 2820}},
		{0, {204, 980, 1800, 2820}},
		{1, {205, 780, 1410, 2720}},
		{0, {208, 710, 1450, 2750}},
		{0, {205, 950, 1280, 2600}},
		{0, {210, 870, 1260, 2740}},
		{0, {203, 610,  900, 2710}},
		{0, {210, 630,  840, 2700}},
		{0, {211, 440, 1050, 2780}},
		{0, {210, 420, 1050, 2740}},
		{0, {222, 380,  860, 2500}},
		{0, {208, 330,  750, 2740}},
		{0, {208, 580, 1450, 1720}},
		{0, {212, 540, 1560, 1900}},
		{0, {210, 294, 2800, 3100}},
		{0, {222, 270, 2880, 3160}},
		{0, {202, 420, 2430, 3030}},
		{0, {212, 420, 2370, 2930}},
		{1, {200, 580, 2180, 2770}},
		{1, {217, 540, 2160, 2770}},
		{0, {200, 820, 1970, 2620}},
		{0, {210, 840, 2000, 2700}},
		{1, {208, 690, 1200, 2900}},
		{0, {201, 666, 1206, 2900}},
		{1, {200, 800, 1200, 2920}},
		{1, {190, 760, 1140, 2850}},
		{0, {200, 560,  760, 2800}},
		{0, {207, 560,  770, 3000}},
		{1, {215, 430, 1075, 2580}},
		{0, {213, 430, 1000, 2700}},
		{0, {220, 330,  840, 2550}},
		{0, {213, 280,  850, 2500}},
		{0, {205, 430, 1800, 1930}},
		{0, {200, 420, 1740, 1960}},
		{0, {175, 350, 2800, 3160}},
		{0, {187, 338, 2870, 3300}},
		{0, {200, 400, 2540, 3200}},
		{0, {210, 420, 2680, 3000}},
		{0, {180, 518, 2470, 3200}},
		{0, {200, 600, 2400, 3150}},
		{0, {171, 773, 2000, 2870}},
		{0, {175, 875, 2100, 2970}},
		{1, {183, 733, 1468, 2700}},
		{0, {200, 740, 1280, 2900}},
		{1, {178, 730, 1210, 2740}},
		{1, {175, 735, 1220, 2850}},
		{0, {160, 560,  960, 2850}},
		{0, {192, 536,  850, 2850}},
		{0, {212, 424, 1040, 2780}},
		{0, {200, 520, 1060, 2670}},
		{0, {190, 380,  770, 2900}},
		{0, {187, 340,  750, 2780}},
		{1, {177, 490, 2120, 2480}},
		{0, {197, 493, 1930, 2300}},
		{0, {250, 325, 2700, 3100}},
		{0, {225, 310, 2750, 3225}},
		{0, {214, 350, 2580, 3000}},
		{0, {267, 390, 2700, 3200}},
		{0, {233, 560, 2330, 2800}},
		{0, {200, 520, 2500, 3000}},
		{0, {171, 806, 1970, 2600}},
		{0, {150, 825, 1860, 2550}},
		{0, {186, 708, 1485, 2760}},
		{0, {188, 676, 1500, 2590}},
		{0, {200, 800, 1200, 2800}},
		{1, {205, 714, 1154, 2850}},
		{0, {267, 530,  800, 2780}},
		{1, {180, 485,  810, 2750}},
		{0, {214, 450, 1460, 2550}},
		{0, {233, 467, 1400, 2450}},
		{0, {225, 450, 1080, 2350}},
		{0, {200, 400, 1000, 2400}},
		{0, {193, 524, 1700, 2130}},
		{0, {180, 507, 1800, 2380}},
		{0, {200, 300, 3100, 3400}},
		{1, {216, 300, 3100, 3500}},
		{0, {214, 428, 2570, 3000}},
		{0, {220, 440, 2640, 3080}},
		{0, {210, 528, 2540, 3170}},
		{1, {210, 504, 2520, 3200}},
		{0, {187, 940, 2250, 2760}},
		{0, {200, 820, 2200, 2920}},
		{0, {204, 816, 1450, 2700}},
		{1, {214, 858, 1500, 2700}},
		{0, {200, 960, 1280, 3000}},
		{1, {180, 1040, 1300, 3000}},
		{0, {220, 520,  880, 2500}},
		{0, {217, 574,  890, 2510}},
		{1, {233, 466, 1330, 2750}},
		{1, {233, 466, 1165, 2800}},
		{0, {180, 300,  850, 2800}},
		{1, {175, 350,  840, 2750}},
		{0, {216, 432, 1790, 2060}},
		{0, {219, 360, 1900, 2320}},
		{0, {225, 337, 2700, 3300}},
		{0, {233, 340, 2720, 3200}},
		{0, {237, 474, 2370, 3095}},
		{0, {237, 475, 2400, 3090}},
		{0, {229, 526, 2360, 3090}},
		{0, {233, 580, 2360, 3150}},
		{1, {230, 690, 2185, 2990}},
		{1, {220, 660, 2200, 3020}},
		{1, {225, 675, 1551, 2923}},
		{0, {233, 690, 1630, 2900}},
		{0, {222, 845, 1334, 2890}},
		{0, {233, 888, 1290, 2800}},
		{0, {225, 631,  923, 2250}},
		{0, {233, 543,  980, 2300}},
		{0, {233, 537, 1360, 2920}},
		{0, {240, 480, 1345, 2680}},
		{0, {235, 400, 1180, 2760}},
		{0, {233, 396, 1120, 2560}},
		{0, {225, 450, 1640, 2250}},
		{0, {233, 489, 1630, 2090}},
		{0, {225, 225, 2760, 3900}},
		{0, {230, 230, 2850, 3800}},
		{0, {238, 429, 2560, 3200}},
		{1, {230, 430, 2575, 3100}},
		{0, {214, 579, 2570, 3300}},
		{0, {214, 536, 2570, 3100}},
		{0, {205, 823, 2220, 2870}},
		{0, {200, 800, 2100, 2900}},
		{0, {250, 750, 1500, 2750}},
		{0, {217, 738, 1300, 2820}},
		{0, {200, 840, 1300, 3100}},
		{1, {206, 990, 1340, 3100}},
		{0, {214, 579,  856, 2790}},
		{0, {205, 545,  905, 2750}},
		{1, {233, 490, 1220, 2610}},
		{1, {250, 513, 1500, 2650}},
		{0, {250, 400, 1250, 2500}},
		{0, {225, 405, 1080, 2500}},
		{0, {233, 466, 1860, 2260}},
		{0, {225, 540, 1780, 2220}},
		{0, {240, 290, 3000, 3840}},
		{0, {250, 325, 2900, 3500}},
		{0, {250, 500, 2370, 3120}},
		{1, {238, 476, 2380, 3090}},
		{0, {238, 760, 2380, 3205}},
		{0, {233, 746, 2290, 3030}},
		{0, {206, 1008, 1990, 2870}},
		{0, {200, 1040, 2000, 2800}},
		{0, {220, 830, 1540, 2860}},
		{0, {237, 900, 1510, 2840}},
		{0, {206, 970, 1343, 3018}},
		{1, {236, 592, 1230, 2600}},
		{0, {233, 650,  900, 2920}},
		{0, {229, 687, 1060, 2780}},
		{1, {233, 512, 1211, 2630}},
		{0, {233, 467, 1167, 2595}},
		{0, {250, 450,  875, 2750}},
		{0, {233, 420,  935, 2710}},
		{0, {230, 622, 1750, 2070}},
		{0, {225, 652, 1710, 2043}},
		{0, {255, 275, 2800, 3310}},
		{0, {245, 245, 2800, 3300}},
		{0, {267, 534, 2500, 3250}},
		{0, {264, 528, 2640, 3370}},
		{0, {238, 700, 2380, 3250}},
		{0, {250, 750, 2480, 3000}},
		{0, {237, 1020, 1900, 2960}},
		{0, {233, 1005, 2050, 2870}},
		{0, {263, 750, 1500, 2850}},
		{0, {250, 850, 1400, 2750}},
		{0, {258, 978, 1290, 2840}},
		{0, {246, 935, 1230, 2730}},
		{0, {250, 500,  750, 2750}},
		{0, {243, 632,  850, 2850}},
		{0, {250, 350, 1170, 2750}},
		{0, {266, 450, 1000, 2800}},
		{0, {256, 358,  640, 2560}},
		{0, {250, 300,  750, 2500}},
		{0, {260, 520, 1560, 1820}},
		{0, {250, 500, 1500, 1750}},
		{0, {236, 236, 2790, 3760}},
		{0, {242, 242, 2770, 3800}},
		{0, {222, 444, 2555, 3110}},
		{0, {242, 420, 2700, 3120}},
		{0, {226, 634, 2325, 2940}},
		{0, {225, 608, 2475, 3100}},
		{0, {210, 1010, 2060, 2900}},
		{0, {200, 980, 2160, 2920}},
		{0, {217, 818, 1450, 2500}},
		{0, {200, 750, 1280, 2650}},
		{0, {220, 820, 1200, 2640}},
		{0, {210, 900, 1120, 2900}},
		{0, {220, 440,  749, 2640}},
		{0, {210, 567,  752, 2600}},
		{1, {204, 460, 1045, 2504}},
		{0, {240, 480, 1105, 2400}},
		{0, {250, 420, 1000, 2500}},
		{0, {275, 350, 1100, 2400}},
		{0, {217, 487, 1500, 1780}},
		{0, {206, 467, 1420, 1640}},
		{0, {225, 360, 2920, 3400}},
		{0, {233, 340, 2840, 3300}},
		{0, {257, 514, 2570, 3070}},
		{0, {238, 500, 2680, 3260}},
		{0, {238, 650, 2495, 3090}},
		{1, {216, 650, 2380, 3030}},
		{0, {225, 1020, 2030, 2700}},
		{0, {225, 1000, 2200, 2770}},
		{0, {225, 788, 1462, 2920}},
		{0, {217, 736, 1500, 2900}},
		{1, {214, 987, 1330, 2830}},
		{0, {214, 1009, 1415, 3080}},
		{1, {226, 672, 1084, 2495}},
		{0, {209, 627, 1045, 2504}},
		{0, {250, 500, 1200, 2450}},
		{0, {230, 460, 1150, 2880}},
		{0, {267, 420,  990, 2860}},
		{0, {190, 380,  893, 2920}},
		{0, {246, 610, 1630, 2020}},
		{0, {225, 585, 1700, 1850}},
		{0, {285, 285, 2900, 3500}},
		{0, {286, 310, 2900, 3400}},
		{0, {297, 480, 2670, 3260}},
		{1, {220, 440, 2620, 3380}},
		{0, {173, 550, 2370, 3140}},
		{0, {260, 520, 2340, 3040}},
		{0, {167, 790, 2180, 3020}},
		{0, {280, 840, 2160, 3020}},
		{0, {280, 840, 1400, 2750}},
		{0, {270, 760, 1330, 2950}},
		{0, {252, 900, 1290, 2750}},
		{1, {260, 900, 1240, 3110}},
		{0, {175, 700, 1050, 2750}},
		{0, {190, 720, 1080, 3030}},
		{0, {286, 540, 1200, 2860}},
		{0, {205, 570, 1200, 2970}},
		{0, {328, 400,  980, 2630}},
		{0, {290, 440,  990, 2900}},
		{0, {286, 570, 2000, 2480}},
		{0, {260, 510, 1850, 2350}},
		{0, {170, 340, 2750, 3120}},
		{0, {238, 360, 2760, 3120}},
		{0, {167, 480, 2390, 2950}},
		{1, {194, 520, 2450, 3000}},
		{0, {220, 620, 2520, 2920}},
		{0, {222, 620, 2440, 2880}},
		{0, {222, 1110, 2160, 2700}},
		{0, {214, 1070, 1960, 2750}},
		{1, {217, 820, 1240, 2600}},
		{1, {216, 860, 1300, 2670}},
		{0, {150, 840, 1110, 2930}},
		{0, {170, 850, 1120, 2850}},
		{0, {200, 500,  700, 2930}},
		{0, {212, 380,  720, 2700}},
		{0, {235, 400,  940, 2820}},
		{0, {214, 380,  860, 2680}},
		{0, {196, 330,  760, 2870}},
		{0, {188, 350,  710, 2760}},
		{0, {182, 550, 1780, 2080}},
		{0, {201, 600, 1750, 2000}},
		{0, {200, 320, 2360, 2980}},
		{0, {203, 304, 2380, 3050}},
		{1, {211, 444, 2220, 2740}},
		{1, {210, 420, 2090, 2780}},
		{0, {200, 500, 2350, 2830}},
		{0, {200, 600, 2200, 2700}},
		{1, {192, 845, 1700, 2300}},
		{0, {187, 860, 1724, 2530}},
		{0, {200, 720, 1440, 2380}},
		{0, {191, 707, 1470, 2440}},
		{0, {200, 700, 1080, 2420}},
		{1, {192, 767, 1150, 2590}},
		{0, {200, 600,  860, 2410}},
		{0, {200, 600,  900, 2400}},
		{0, {210, 546, 1090, 2400}},
		{0, {210, 462, 1240, 2310}},
		{0, {257, 360,  930, 2260}},
		{0, {220, 440, 1100, 2300}},
		{0, {200, 540, 1400, 1800}},
		{0, {204, 460, 1350, 1560}},
		{0, {203, 406, 2600, 2945}},
		{1, {200, 400, 2600, 3100}},
		{0, {200, 460, 2300, 2800}},
		{0, {210, 420, 2305, 2835}},
		{0, {190, 570, 2100, 2720}},
		{0, {207, 538, 2175, 2880}},
		{0, {189, 850, 1853, 2685}},
		{0, {193, 830, 1800, 2620}},
		{0, {200, 720, 1500, 2560}},
		{0, {200, 800, 1400, 2420}},
		{0, {194, 915, 1280, 2530}},
		{0, {206, 723, 1196, 2600}},
		{0, {192, 575, 1073, 2490}},
		{0, {200, 600, 1100, 2600}},
		{0, {202, 520, 1210, 2420}},
		{0, {212, 468, 1275, 2550}},
		{0, {207, 370, 1000, 2470}},
		{0, {205, 330,  970, 2460}},
		{0, {200, 560, 1600, 1900}},
		{0, {206, 514, 1540, 1955}},
		{0, {240, 380, 2880, 3360}},
		{0, {250, 380, 2820, 3300}},
		{1, {233, 514, 2600, 2930}},
		{0, {237, 473, 2660, 2970}},
		{0, {223, 567, 2460, 3122}},
		{0, {224, 521, 2460, 2920}},
		{0, {218, 808, 2070, 2880}},
		{1, {203, 678, 2420, 3080}},
		{0, {200, 800, 1340, 2700}},
		{1, {214, 772, 1280, 2660}},
		{0, {183, 843, 1190, 2860}},
		{0, {205, 740, 1160, 2780}},
		{1, {222, 623, 1022, 2700}},
		{0, {220, 594,  990, 2640}},
		{1, {240, 480,  960, 2820}},
		{0, {242, 484,  900, 2640}},
		{0, {233, 370,  933, 2520}},
		{0, {250, 325,  750, 2500}},
		{0, {225, 450, 1680, 2050}},
		{1, {233, 466, 1630, 1865}},
		{0, {200, 320, 2750, 3100}},
		{0, {178, 356, 2755, 3200}},
		{0, {194, 388, 2622, 3050}},
		{0, {194, 426, 2460, 3040}},
		{0, {187, 592, 2242, 2765}},
		{1, {191, 535, 2290, 2870}},
		{1, {188, 750, 2060, 2770}},
		{1, {162, 650, 2110, 2618}},
		{0, {187, 618, 1518, 2700}},
		{0, {183, 624, 1430, 2660}},
		{0, {163, 766, 1180, 2340}},
		{0, {167, 750, 1065, 2640}},
		{1, {170, 595,  918, 2600}},
		{1, {176, 630,  985, 2630}},
		{0, {200, 420, 1200, 2600}},
		{0, {200, 460, 1260, 2640}},
		{0, {187, 375, 1124, 2685}},
		{1, {188, 375, 1143, 2700}},
		{0, {180, 504, 1565, 1835}},
		{0, {183, 513, 1578, 1830}},
		{0, {280, 357, 2800, 3360}},
		{0, {275, 340, 2860, 3350}},
		{0, {290, 480, 2600, 3060}},
		{0, {292, 465, 2598, 3060}},
		{1, {250, 700, 2350, 2980}},
		{0, {240, 737, 2325, 3100}},
		{0, {200, 960, 2100, 3000}},
		{0, {217, 1030, 2200, 3260}},
		{0, {275, 920, 1512, 2950}},
		{0, {260, 910, 1688, 3050}},
		{0, {275, 990, 1237, 2360}},
		{0, {267, 987, 1172, 3180}},
		{0, {267, 587, 1068, 3270}},
		{0, {293, 560,  990, 3150}},
		{1, {275, 520, 1350, 3190}},
		{1, {280, 510, 1415, 3130}},
		{0, {300, 420, 1045, 3060}},
		{0, {300, 390,  960, 3030}},
		{0, {230, 460, 1860, 2250}},
		{0, {214, 504, 1820, 2290}},
		{0, {200, 240, 2760, 3700}},
		{0, {220, 220, 2850, 3800}},
		{0, {228, 319, 2500, 3020}},
		{0, {216, 324, 2500, 3010}},
		{0, {220, 616, 2380, 2900}},
		{0, {212, 615, 2300, 2800}},
		{1, {212, 710, 2120, 2600}},
		{1, {210, 690, 2250, 2680}},
		{0, {221, 800, 1520, 2380}},
		{0, {210, 780, 1470, 2400}},
		{0, {199, 995, 1392, 2290}},
		{0, {200, 1000, 1400, 2440}},
		{0, {205, 656,  944, 2250}},
		{0, {200, 720,  960, 2380}},
		{0, {223, 335, 1049, 2470}},
		{0, {210, 420, 1009, 2300}},
		{0, {219, 329,  877, 2550}},
		{0, {230, 340,  900, 2530}},
		{0, {206, 400, 1380, 1560}},
		{0, {201, 400, 1240, 1480}},
		{0, {220, 286, 2800, 3550}},
		{0, {241, 289, 2800, 3400}},
		{0, {225, 383, 2420, 3080}},
		{0, {240, 384, 2400, 3050}},
		{1, {209, 418, 2430, 3110}},
		{0, {230, 460, 2300, 3050}},
		{0, {187, 861, 2100, 2800}},
		{0, {224, 896, 2040, 3000}},
		{0, {218, 654, 1160, 2800}},
		{0, {230, 690, 1195, 2770}},
		{0, {208, 860, 1103, 2700}},
		{0, {212, 806, 1060, 2850}},
		{0, {202, 606,  910, 2900}},
		{0, {201, 583,  860, 2840}},
		{0, {225, 340,  900, 2650}},
		{0, {235, 470, 1100, 2560}},
		{0, {205, 308, 1025, 2650}},
		{0, {235, 329, 1151, 2560}},
		{0, {213, 533, 1425, 1830}},
		{0, {214, 535, 1412, 1800}},
		{0, {236, 307, 2670, 3150}},
		{0, {245, 340, 2700, 3250}},
		{0, {231, 417, 2300, 3000}},
		{1, {239, 410, 2200, 2910}},
		{1, {222, 644, 2250, 3000}},
		{0, {224, 670, 2300, 2880}},
		{1, {224, 784, 1800, 2750}},
		{1, {234, 820, 1750, 2890}},
		{0, {225, 765, 1300, 2700}},
		{0, {221, 730, 1390, 2790}},
		{1, {225, 834, 1282, 2800}},
		{0, {212, 850, 1270, 2760}},
		{1, {229, 688, 1029, 2750}},
		{0, {222, 670, 1040, 2640}},
		{0, {251, 427, 1506, 2640}},
		{0, {240, 460, 1370, 2610}},
		{0, {236, 378, 1416, 2580}},
		{0, {239, 380, 1430, 2610}},
		{0, {230, 460, 1200, 1909}},
		{0, {225, 410, 1580, 1800}},
		{0, {256, 384, 2860, 3210}},
		{0, {250, 375, 3000, 3400}},
		{0, {230, 460, 2665, 3140}},
		{0, {233, 467, 2680, 3150}},
		{0, {229, 640, 2400, 2860}},
		{0, {233, 630, 2530, 3030}},
		{1, {233, 700, 2560, 3150}},
		{1, {225, 675, 2510, 3145}},
		{1, {240, 768, 1440, 2855}},
		{0, {234, 794, 1447, 2920}},
		{0, {227, 978, 1362, 2724}},
		{0, {233, 933, 1350, 2610}},
		{0, {240, 700, 1080, 2810}},
		{0, {240, 720, 1090, 2840}},
		{0, {243, 500, 1215, 2870}},
		{0, {239, 500, 1240, 2860}},
		{0, {263, 470, 1000, 2820}},
		{0, {272, 378,  950, 2990}},
		{0, {243, 480, 1410, 1700}},
		{0, {243, 493, 1580, 1775}},
		{0, {268, 320, 2900, 3200}},
		{0, {263, 290, 2750, 3050}},
		{1, {258, 460, 2380, 2940}},
		{1, {251, 480, 2260, 2980}},
		{1, {246, 640, 2220, 2900}},
		{1, {250, 670, 2250, 2960}},
		{0, {243, 950, 1970, 2890}},
		{0, {244, 980, 1950, 2920}},
		{1, {251, 750, 1280, 2760}},
		{0, {258, 770, 1340, 2800}},
		{1, {250, 950, 1130, 3160}},
		{1, {256, 850, 1150, 2940}},
		{0, {242, 530,  870, 2680}},
		{0, {250, 600,  900, 2770}},
		{1, {250, 600, 1225, 2500}},
		{1, {264, 630, 1320, 2560}},
		{0, {258, 440, 1290, 2530}},
		{0, {269, 460, 1080, 2640}},
		{0, {250, 600, 1500, 2000}},
		{0, {254, 610, 1520, 1950}},
		{0, {234, 280, 2690, 3040}},
		{0, {261, 280, 2740, 2980}},
		{0, {260, 470, 2500, 3400}},
		{0, {262, 440, 2480, 3240}},
		{0, {242, 730, 2300, 3100}},
		{0, {260, 750, 2340, 3120}},
		{0, {233, 860, 2070, 2880}},
		{0, {240, 890, 1920, 2710}},
		{0, {257, 770, 1540, 2840}},
		{0, {257, 800, 1410, 2860}},
		{0, {240, 790, 1250, 3080}},
		{1, {241, 820, 1210, 2960}},
		{0, {234, 408,  695, 3040}},
		{0, {246, 420,  590, 3100}},
		{0, {251, 500, 1230, 2520}},
		{0, {256, 480, 1230, 2750}},
		{0, {263, 419, 1050, 2850}},
		{0, {278, 390, 1060, 2800}},
		{0, {220, 420, 1720, 1900}},
		{0, {255, 510, 1680, 1890}},
		{0, {208, 270, 2820, 3450}},
		{0, {225, 250, 2880, 3350}},
		{0, {220, 370, 2530, 3060}},
		{0, {250, 400, 2600, 3120}},
		{0, {214, 640, 2360, 3020}},
		{0, {219, 650, 2430, 3040}},
		{0, {205, 900, 2090, 3000}},
		{0, {200, 860, 2160, 2870}},
		{0, {214, 750, 1540, 2800}},
		{0, {214, 770, 1530, 2780}},
		{0, {195, 920, 1350, 2550}},
		{1, {210, 920, 1470, 2690}},
		{0, {194, 720, 1110, 2420}},
		{0, {200, 700, 1100, 2780}},
		{0, {222, 470, 1200, 2900}},
		{0, {237, 470, 1190, 2800}},
		{0, {240, 380,  980, 3100}},
		{0, {188, 340,  920, 3050}},
		{0, {222, 530, 1670, 2050}},
		{0, {200, 500, 1720, 1900}},
		{0, {258, 310, 2740, 3200}},
		{0, {262, 262, 2680, 3170}},
		{0, {262, 450, 2310, 3020}},
		{0, {263, 472, 2270, 2950}},
		{1, {245, 640, 1980, 2920}},
		{1, {235, 700, 2110, 2940}},
		{0, {194, 810, 1860, 2620}},
		{0, {234, 890, 1800, 2700}},
		{1, {230, 710, 1340, 2780}},
		{1, {245, 740, 1470, 2940}},
		{1, {225, 830, 1020, 2650}},
		{1, {219, 830, 1095, 2610}},
		{0, {240, 600,  850, 2760}},
		{0, {253, 455,  810, 2750}},
		{0, {282, 400, 1070, 2530}},
		{0, {250, 450, 1050, 2450}},
		{0, {260, 290,  670, 2380}},
		{0, {275, 330,  630, 2460}},
		{0, {240, 500, 1630, 2040}},
		{0, {243, 490, 1580, 2190}},
		{0, {228, 460, 3300, 3950}},
		{0, {200, 400, 3400, 3850}},
		{1, {205, 600, 2550, 4000}},
		{1, {205, 610, 2500, 4100}},
		{1, {225, 600, 2750, 3600}},
		{1, {210, 760, 2500, 3850}},
		{0, {200, 1000, 2300, 3900}},
		{1, {200, 800, 2500, 4050}},
		{1, {200, 1000, 1750, 3550}},
		{0, {223, 1110, 1690, 4040}},
		{1, {205, 1220, 1560, 3650}},
		{1, {200, 1300, 1800, 3450}},
		{0, {219, 660, 1100, 3850}},
		{1, {217, 690, 1090, 3900}},
		{1, {206, 620, 1420, 3700}},
		{0, {220, 620, 1410, 3520}},
		{0, {233, 440,  900, 3900}},
		{0, {200, 400,  650, 3800}},
		{0, {210, 610, 2300, 2900}},
		{0, {200, 450, 2150, 2550}},
		{0, {290, 320, 3500, 4260}},
		{0, {305, 350, 3400, 4100}},
		{0, {322, 640, 3200, 3660}},
		{0, {325, 650, 3000, 3800}},
		{0, {270, 850, 2900, 3680}},
		{1, {285, 700, 3120, 3750}},
		{0, {256, 1130, 2560, 3500}},
		{0, {285, 1140, 2000, 3560}},
		{0, {310, 1130, 1740, 3670}},
		{0, {300, 1000, 1800, 3450}},
		{1, {265, 1170, 1500, 3440}},
		{1, {283, 980, 1300, 3100}},
		{1, {265, 530, 1060, 3450}},
		{1, {272, 540, 1080, 3000}},
		{0, {285, 560, 1440, 3500}},
		{0, {294, 570, 1450, 3500}},
		{0, {333, 350, 1280, 3650}},
		{0, {290, 340, 1160, 2950}},
		{1, {275, 560, 1740, 2460}},
		{0, {302, 600, 1800, 2200}},
		{0, {240, 380, 3140, 3700}},
		{0, {258, 310, 3350, 3650}},
		{1, {290, 580, 2760, 3400}},
		{0, {250, 500, 2660, 3500}},
		{1, {250, 780, 2450, 3400}},
		{1, {240, 672, 2550, 3400}},
		{1, {240, 660, 2900, 3370}},
		{1, {215, 760, 2850, 3300}},
		{0, {250, 880, 1500, 3200}},
		{0, {243, 850, 1700, 3250}},
		{0, {250, 940, 1380, 2400}},
		{0, {276, 1200, 1500, 3160}},
		{0, {250, 750, 1250, 3450}},
		{0, {225, 675,  950, 3240}},
		{0, {300, 610, 1500, 3300}},
		{0, {275, 500, 1370, 3500}},
		{0, {256, 300, 1280, 3150}},
		{0, {250, 400, 1300, 3700}},
		{0, {250, 500, 1540, 1700}},
		{0, {242, 580, 1620, 1790}},
		{0, {291, 410, 3200, 3800}},
		{0, {264, 420, 3400, 3900}},
		{1, {291, 580, 2900, 3820}},
		{1, {280, 560, 2840, 3900}},
		{1, {292, 810, 2640, 4200}},
		{1, {270, 780, 2720, 4100}},
		{0, {270, 1080, 2480, 3950}},
		{0, {245, 1050, 2420, 4000}},
		{1, {286, 970, 1600, 3950}},
		{1, {250, 800, 1680, 3800}},
		{1, {275, 1040, 1350, 3850}},
		{0, {250, 1100, 1460, 4250}},
		{1, {286, 770, 1150, 3950}},
		{1, {273, 710, 1200, 3900}},
		{1, {285, 680, 1420, 3800}},
		{0, {278, 640, 1350, 3950}},
		{0, {300, 420, 1110, 3640}},
		{0, {280, 505, 1050, 3400}},
		{0, {320, 640, 1940, 2820}},
		{0, {265, 610, 2100, 2600}},
		{0, {330, 460, 2800, 3550}},
		{0, {333, 490, 2730, 3550}},
		{1, {310, 560, 2500, 3450}},
		{1, {310, 580, 2500, 3450}},
		{1, {286, 800, 2300, 3750}},
		{1, {310, 835, 2420, 3740}},
		{0, {282, 950, 2150, 3650}},
		{0, {310, 1000, 2150, 3700}},
		{0, {293, 880, 1700, 3750}},
		{0, {340, 900, 1600, 3650}},
		{1, {299, 990, 1410, 3750}},
		{0, {280, 1050, 1320, 3730}},
		{1, {285, 770,  940, 3750}},
		{1, {333, 680, 1020, 3700}},
		{0, {322, 550, 1195, 3750}},
		{0, {350, 550, 1340, 3500}},
		{1, {316, 600, 1200, 3600}},
		{0, {345, 550, 1100, 3470}},
		{0, {310, 805, 1705, 2420}},
		{0, {310, 710, 1700, 2400}},
		{0, {210, 340, 3400, 4320}},
		{0, {227, 590, 3610, 4220}},
		{0, {235, 680, 3250, 4380}},
		{1, {220, 440, 3000, 3790}},
		{0, {212, 660, 2900, 3610}},
		{1, {216, 610, 2760, 3650}},
		{0, {214, 1240, 2700, 3640}},
		{0, {215, 1050, 2550, 3550}},
		{0, {216, 820, 1470, 3500}},
		{1, {211, 970, 1410, 3200}},
		{0, {218, 1090, 1380, 3050}},
		{1, {212, 860, 1250, 2800}},
		{0, {211, 800, 1220, 3700}},
		{0, {214, 640, 1070, 3000}},
		{0, {219, 660, 1360, 3700}},
		{0, {214, 730, 1500, 3600}},
		{0, {220, 620, 1100, 3250}},
		{0, {216, 600, 1280, 3650}},
		{0, {222, 670, 2130, 2360}},
		{0, {205, 760, 2240, 2460}},
		{0, {253, 330, 3250, 3720}},
		{0, {262, 340, 3100, 3400}},
		{1, {250, 500, 2500, 3640}},
		{1, {278, 530, 2630, 3640}},
		{1, {255, 710, 2550, 3560}},
		{1, {250, 750, 2480, 3470}},
		{0, {233, 1140, 2260, 3640}},
		{0, {245, 1110, 2230, 3380}},
		{0, {256, 770, 1540, 3500}},
		{0, {257, 800, 1490, 3300}},
		{0, {240, 940, 1400, 3400}},
		{1, {245, 930, 1370, 3120}},
		{0, {240, 530,  860, 3400}},
		{0, {240, 520,  910, 3420}},
		{0, {255, 510, 1250, 3320}},
		{0, {260, 520, 1140, 3320}},
		{0, {274, 360,  660, 3050}},
		{0, {260, 310,  730, 3500}},
		{0, {250, 550, 1500, 1800}},
		{0, {239, 480, 1650, 1960}},
		{0, {250, 300, 2950, 3600}},
		{0, {270, 320, 3210, 3600}},
		{0, {290, 550, 2610, 3560}},
		{1, {286, 540, 2570, 3600}},
		{0, {280, 700, 2500, 3580}},
		{0, {263, 600, 2360, 3400}},
		{0, {260, 970, 2400, 3200}},
		{0, {250, 950, 2270, 3200}},
		{0, {270, 780, 1650, 3350}},
		{1, {250, 720, 1500, 3240}},
		{1, {278, 950, 1200, 2950}},
		{1, {250, 920, 1080, 2770}},
		{1, {262, 790, 1050, 2900}},
		{0, {250, 750, 1000, 2500}},
		{0, {275, 540, 1430, 3320}},
		{0, {263, 530, 1580, 3200}},
		{0, {295, 420, 1500, 3010}},
		{0, {260, 450, 1330, 2840}},
		{0, {272, 570, 1880, 2400}},
		{0, {255, 510, 1610, 1910}},
		{0, {235, 280, 2820, 3400}},
		{0, {244, 317, 3125, 3500}},
		{0, {230, 460, 2520, 3300}},
		{0, {212, 420, 2480, 3140}},
		{0, {235, 657, 2300, 3300}},
		{0, {232, 672, 2275, 3300}},
		{0, {231, 808, 1950, 3300}},
		{0, {225, 870, 2000, 3200}},
		{0, {236, 706, 1410, 3200}},
		{0, {211, 720, 1480, 2880}},
		{1, {250, 950, 1350, 3100}},
		{1, {227, 910, 1360, 2950}},
		{0, {203, 700, 1120, 3070}},
		{0, {230, 690,  920, 2760}},
		{0, {250, 475, 1250, 3150}},
		{0, {212, 460, 1210, 2750}},
		{0, {244, 403, 1100, 2950}},
		{0, {242, 363,  920, 2900}},
		{0, {226, 452, 1580, 1810}},
		{0, {232, 510, 1550, 1740}},
		{0, {230, 280, 3140, 3830}},
		{0, {250, 300, 3400, 3950}},
		{1, {225, 450, 2700, 3650}},
		{1, {250, 400, 2840, 3700}},
		{0, {215, 580, 2650, 3550}},
		{1, {220, 620, 2660, 3770}},
		{0, {240, 910, 2370, 3160}},
		{0, {233, 930, 2350, 3450}},
		{0, {250, 770, 1650, 3420}},
		{0, {230, 690, 1600, 3350}},
		{1, {242, 970, 1450, 3260}},
		{1, {225, 1010, 1650, 3150}},
		{0, {232, 670, 1160, 3550}},
		{0, {225, 720, 1260, 3400}},
		{0, {216, 500, 1640, 3580}},
		{0, {250, 450, 1440, 3500}},
		{0, {290, 350, 1160, 3260}},
		{0, {273, 330, 1090, 3350}},
		{0, {240, 430, 1800, 2400}},
		{0, {233, 470, 1840, 2400}},
		{0, {275, 330, 3050, 3800}},
		{0, {286, 340, 2860, 3610}},
		{0, {280, 500, 2720, 3360}},
		{0, {230, 600, 2750, 3550}},
		{0, {245, 735, 2450, 3300}},
		{0, {258, 780, 2560, 3300}},
		{0, {235, 940, 2020, 2580}},
		{0, {232, 1070, 2320, 2900}},
		{0, {268, 860, 1530, 3100}},
		{0, {256, 970, 1500, 3050}},
		{1, {245, 780, 1250, 3180}},
		{1, {236, 970,  970, 3120}},
		{1, {258, 825, 1210, 3100}},
		{1, {300, 930,  930, 2900}},
		{0, {260, 490, 1460, 2860}},
		{0, {286, 570, 1320, 2840}},
		{0, {275, 470, 1400, 2800}},
		{0, {286, 370, 1160, 2800}},
		{0, {268, 510, 1660, 2100}},
		{0, {250, 480, 1700, 1830}},
		{0, {295, 380, 3200, 4000}},
		{0, {267, 350, 3250, 3700}},
		{0, {294, 380, 2960, 3800}},
		{0, {300, 520, 2900, 3600}},
		{0, {280, 670, 2790, 3600}},
		{1, {275, 620, 2750, 3500}},
		{0, {262, 1070, 2380, 3100}},
		{0, {275, 1130, 2320, 3110}},
		{1, {290, 700, 1730, 2960}},
		{0, {270, 725, 1570, 2900}},
		{0, {278, 1110, 1630, 2780}},
		{0, {280, 1130, 1400, 3000}},
		{1, {292, 580,  930, 2950}},
		{0, {270, 540, 1070, 3000}},
		{0, {300, 450, 1350, 3000}},
		{0, {320, 520, 1600, 3150}},
		{0, {307, 460, 1460, 3070}},
		{0, {300, 400, 1700, 3000}},
		{0, {300, 540, 1770, 2040}},
		{0, {286, 540, 2050, 2300}},
		{0, {300, 300, 3250, 3850}},
		{0, {275, 275, 3280, 3800}},
		{0, {286, 570, 2850, 3400}},
		{0, {267, 485, 2630, 3450}},
		{0, {264, 650, 2880, 3500}},
		{0, {284, 570, 2900, 3600}},
		{0, {260, 1300, 2280, 3130}},
		{0, {260, 1300, 2160, 3300}},
		{0, {275, 850, 1540, 3020}},
		{0, {262, 840, 1580, 2880}},
		{0, {250, 1230, 1300, 3200}},
		{0, {286, 1090, 1230, 2980}},
		{0, {265, 850,  850, 2920}},
		{1, {285, 685, 1030, 2900}},
		{0, {283, 540, 1420, 3050}},
		{0, {300, 600, 1440, 2900}},
		{0, {280, 390, 1340, 2830}},
		{0, {284, 340, 1110, 3080}},
		{0, {280, 530, 1650, 1740}},
		{0, {286, 550, 1660, 1770}},
		{0, {265, 370, 2950, 3400}},
		{0, {290, 370, 2910, 3480}},
		{0, {271, 515, 2740, 3280}},
		{0, {290, 485, 2600, 3200}},
		{0, {262, 630, 2520, 3150}},
		{0, {272, 565, 2440, 3120}},
		{0, {262, 970, 2030, 2880}},
		{0, {275, 915, 2130, 2900}},
		{0, {270, 810, 1600, 3230}},
		{1, {280, 760, 1530, 3180}},
		{1, {270, 810, 1350, 2940}},
		{0, {275, 1000, 1360, 3000}},
		{1, {270, 535,  970, 2960}},
		{1, {275, 550, 1080, 2850}},
		{1, {275, 550, 1420, 3040}},
		{0, {295, 570, 1500, 3000}},
		{0, {283, 510, 1700, 3020}},
		{1, {278, 500, 1640, 3050}},
		{0, {261, 522, 1830, 2350}},
		{0, {282, 530, 1800, 2250}},
		{0, {320, 350, 3240, 3760}},
		{0, {344, 344, 3120, 3640}},
		{0, {308, 590, 2760, 3500}},
		{0, {320, 540, 2900, 3500}},
		{0, {307, 830, 2750, 3650}},
		{0, {308, 800, 2640, 3540}},
		{0, {294, 1140, 2450, 3230}},
		{0, {239, 1130, 2550, 3150}},
		{0, {310, 930, 1540, 3120}},
		{0, {315, 950, 1670, 3150}},
		{0, {350, 1190, 1470, 3150}},
		{1, {314, 1070, 1460, 2950}},
		{0, {300, 910, 1200, 3180}},
		{0, {330, 830, 1250, 3250}},
		{0, {327, 630, 1310, 3270}},
		{0, {322, 610, 1550, 3400}},
		{0, {345, 520, 1250, 3460}},
		{0, {334, 500, 1140, 3380}},
		{0, {308, 740, 1850, 2160}},
		{0, {328, 660, 1830, 2200}}
	};

	try {
		autoTable me = Table_create (nrows, ncols);

		for (integer i = 1; i <= nrows; i ++) {
			const TableRow row = my rows.at [i];
			const integer vowel_id = ( (i - 1) % 20) / 2 + 1;	/* 1 - 10 */
			const integer speaker_id = (i - 1) / 20 + 1;		/* 1 - 76 */
			integer speaker_type, speaker_sex;

			if (speaker_id <= 33) { /* 33 men */
				speaker_type = 0;
				speaker_sex = 0;
			} else if (speaker_id <= (33 + 28)) { /* 28 women */
				speaker_type = 1;
				speaker_sex = 1;
			} else { /*15  children */
				speaker_type = 2;
				speaker_sex = 0;
				if (speaker_id == 62 || speaker_id == 63 ||
					(speaker_id >= 65 && speaker_id <= 68) ||
					speaker_id == 73 || speaker_id == 76)
					speaker_sex = 1;
			}

			row -> cells [1]. string = Melder_dup (type [speaker_type]);
			row -> cells [2]. string = Melder_dup (sex [speaker_sex]);
			row -> cells [3]. string = Melder_dup (Melder_integer (speaker_id));
			row -> cells [4]. string = Melder_dup (vowel [vowel_id - 1]);
			row -> cells [5]. string = Melder_dup (ipa [vowel_id - 1]);
			for (integer j = 0; j <= 3; j ++)
				row -> cells [j + 6]. string = Melder_dup (Melder_integer (pbdata [i - 1].f [j]));
		}
		for (integer j = 1; j <= ncols; j ++) {
			Table_renameColumn_e (me.get(), j, columnLabels [j - 1]);
			my columnHeaders [j]. numericized = false;
		}
		return me;
	} catch (MelderError) {
		Melder_throw (U"Table not created from Peterson & Barney data.");
	}
}

autoTable Table_create_polsVanNierop1973 () {
	const integer nrows = 900, ncols = 10;
	const conststring32 columnLabels [10] = {U"Sex", U"Speaker", U"Vowel", U"IPA", U"F1", U"F2", U"F3", U"L1", U"L2", U"L3"};
	const conststring32 vowel [12] = {U"oe", U"aa", U"oo", U"a", U"eu", U"ie", U"uu", U"ee", U"u", U"e", U"o", U"i"};
	const conststring32 ipa [12] = {U"u", U"a", U"o", U"\\as", U"\\o/", U"i", U"y", U"e", U"\\yc", U"\\ep", U"\\ct", U"\\ic"};
	const conststring32 sex [2] = {U"m", U"f"};
	const struct polsdatum {
		short f [3]; // frequency F1, F2, F3
		short l [3];	// level f1, f2, f3
	} polsdata [] = {
		/* 50*12 males */
		/* male 1 */
		{{320,  630,  2560},  {6,  13,  48}}, 	// poet
		{{780, 1300,  2460},  {6,   8,  30}},	// paat
		{{500,  940,  2420},  {3,  12,  35}},	// poot
		{{720, 1060,  2420},  {3,   8,  27}},	// pat
		{{430, 1580,  2260},  {2,  24,  36}},	// peut
		{{280, 2300,  2780}, {14,  22,  27}},	// piet
		{{320, 1680,  2140},  {6,  23,  30}},	// puut
		{{420, 2000,  2620},  {5,  20,  23}},	// peet
		{{420, 1540,  2380},  {4,  19,  24}},	// put
		{{600, 1720,  2700},  {3,  17,  29}},	// pet
		{{520, 1000,  2520},  {4,  13,  31}},	// pot
		{{350, 2000,  2520},  {7,  19,  18}},	// pit
		/* male 2 */
		{{440,  780,  2600},  {7,  20,  35}},
		{{940, 1300,  2780},  {5,  13,  26}},
		{{500,  740,  2700},  {7,  11,  32}},
		{{800, 1000,  2480}, {11,  11,  31}},
		{{460, 1500,  2300}, {10,  20,  20}},
		{{320, 2400,  3040},  {9,  28,  27}},
		{{340, 1600,  2050},  {7,  30,  32}},
		{{420, 2200,  2650},  {4,  10,  12}},
		{{540, 1370,  2320},  {6,  23,  31}},
		{{760, 1660,  2600},  {8,  13,  18}},
		{{560,  800,  2800},  {5,  17,  37}},
		{{430, 2030,  2660},  {7,  18,  17}},
		/* male 3 */
		{{280,  740,  2160},  {8,  32,  46}},
		{{860, 1500,  2580},  {7,  13,  19}},
		{{480,  820,  2280},  {4,  11,  33}},
		{{680, 1020,  2460},  {9,  14,  25}},
		{{360, 1520,  2080},  {5,  16,  21}},
		{{270, 2040,  2860},  {5,  18,  18}},
		{{280, 1600,  1900},  {5,  16,  21}},
		{{380, 1940,  2580},  {5,  15,  17}},
		{{400, 1520,  2120},  {4,  21,  24}},
		{{560, 1840,  2520},  {7,  18,  22}},
		{{500,  820,  2520},  {3,  11,  30}},
		{{350, 2000,  2660},  {4,  15,  19}},
		/* male 4 */
		{{360,  820,  2220},  {6,  11,  30}},
		{{840, 1300,  2280},  {9,   7,  22}},
		{{500,  900,  2320},  {7,  13,  35}},
		{{680, 1000,  2480}, {12,  12,  27}},
		{{440, 1320,  2060},  {5,  12,  27}},
		{{240, 2060,  2580},  {5,  20,  23}},
		{{300, 1540,  2020},  {4,  18,  24}},
		{{460, 1920,  2460},  {7,  18,  17}},
		{{480, 1320,  2200},  {6,  15,  22}},
		{{660, 1660,  2340},  {7,  16,  21}},
		{{500,  800,  2520},  {7,  14,  34}},
		{{440, 1920,  2380},  {3,  19,  18}},
		/* male 5 */
		{{440,  880,  2300},  {6,  22,  41}},
		{{820, 1420,  2180},  {6,   7,  28}},
		{{540,  960,  2460},  {4,  10,  28}},
		{{780, 1040,  2620},  {8,  10,  32}},
		{{540, 1540,  2160},  {2,  17,  19}},
		{{300, 2300,  2900},  {5,  23,  30}},
		{{360, 1860,  2200},  {2,  15,  15}},
		{{520, 1960,  2400},  {4,  21,  21}},
		{{560, 1440,  2280},  {2,  17,  27}},
		{{700, 1720,  2300},  {7,  13,  22}},
		{{600,  880,  3000},  {5,   5,  25}},
		{{560, 1920,  2400},  {5,  22,  25}},
		/* male 6 */
		{{260,  700,  2550},  {3,  24,  45}},
		{{820, 1460,  2760}, {10,  15,  27}},
		{{450,  900,  2460}, {19,  20,  45}},
		{{700, 1080,  2660}, {13,  22,  32}},
		{{460, 1750,  2300},  {7,  32,  40}},
		{{240, 2500,  3000},  {3,  33,  37}},
		{{260, 2100,  2500}, {10,  37,  43}},
		{{300, 2320,  2860},  {8,  26,  26}},
		{{440, 1700,  2660},  {7,  27,  32}},
		{{560, 2080,  2840}, {12,  15,  16}},
		{{550,  900,  2740},  {7,  25,  30}},
		{{340, 2340,  3000},  {8,  31,  31}},
		/* male 7 */
		{{280,  860,  2340},  {5,  15,  33}},
		{{800, 1320,  2540},  {7,  14,  26}},
		{{520,  920,  2600},  {8,  15,  33}},
		{{600, 1000,  2760},  {7,   7,  25}},
		{{450, 1660,  2260},  {7,  20,  24}},
		{{260, 2340,  2640},  {3,  17,  20}},
		{{280, 1780,  2160},  {3,  25,  29}},
		{{400, 2040,  2400},  {9,  19,  21}},
		{{460, 1560,  2400},  {4,  19,  22}},
		{{620, 1760,  2560},  {6,  18,  25}},
		{{560,  960,  2760},  {9,  16,  33}},
		{{340, 2000,  2600},  {6,  14,  18}},
		/* male 8 */
		{{320,  880,  2200},  {6,  16,  40}},
		{{800, 1160,  2600},  {6,  13,  26}},
		{{560,  980,  2360},  {5,   8,  35}},
		{{700, 1080,  2540}, {10,  14,  34}},
		{{500, 1480,  2300},  {8,  17,  29}},
		{{280, 2080,  2620},  {3,  27,  27}},
		{{320, 1760,  2060},  {4,  25,  32}},
		{{400, 1940,  2540},  {9,  17,  22}},
		{{400, 1560,  2280},  {6,  18,  28}},
		{{540, 1860,  2540},  {4,  14,  19}},
		{{560,  920,  2320},  {8,  13,  32}},
		{{340, 1960,  2480},  {6,  13,  17}},
		/* male 9 */
		{{300,  680,  2400},  {3,  19,  32}},
		{{860, 1300,  2660}, {12,  18,  26}},
		{{500,  940,  2500},  {3,  13,  33}},
		{{700, 1120,  2620},  {9,  17,  24}},
		{{500, 1500,  2280},  {3,  17,  22}},
		{{300, 2380,  2960},  {2,  20,  22}},
		{{300, 1760,  2160},  {1,  19,  23}},
		{{480, 2100,  2580},  {4,  15,  15}},
		{{500, 1580,  2400},  {5,  12,  22}},
		{{640, 1700,  2620},  {3,  15,  19}},
		{{560,  900,  2940},  {4,   8,  31}},
		{{400, 2040,  2600},  {7,  17,  19}},
		/* male 10 */
		{{360,  900,  2220}, {11,  21,  46}},
		{{880, 1400,  2660},  {9,  17,  29}},
		{{460,  940,  2400},  {3,  13,  37}},
		{{660, 1040,  2660},  {5,   5,  31}},
		{{460, 1580,  2360},  {4,  22,  26}},
		{{340, 2200,  2920}, {14,  30,  26}},
		{{400, 1880,  2800}, {12,  19,  33}},
		{{460, 2080,  2800},  {4,  18,  17}},
		{{460, 1480,  2260},  {5,  27,  31}},
		{{600, 1860,  2640},  {7,  18,  23}},
		{{520,  860,  2880},  {2,  24,  32}},
		{{460, 2100,  2800},  {6,  19,  22}},
		/* male 11 */
		{{320,  830,  2060},  {5,  16,  45}},
		{{820, 1340,  2200},  {7,   9,  26}},
		{{520,  840,  2040},  {7,  13,  31}},
		{{660, 1060,  2300},  {8,   9,  28}},
		{{440, 1520,  2040},  {7,  16,  22}},
		{{300, 2100,  2600},  {5,  23,  30}},
		{{300, 1740,  2040},  {2,  17,  19}},
		{{340, 2040,  2460},  {5,  20,  24}},
		{{500, 1440,  2200},  {8,  17,  21}},
		{{600, 1760,  2380}, {10,  15,  19}},
		{{560,  900,  2300},  {9,  10,  28}},
		{{300, 1960,  2400},  {7,  22,  24}},
		/* male 12 */
		{{400,  860,  2700},  {7,  15,  46}},
		{{940, 1520,  3040}, {10,  19,  33}},
		{{580, 1040,  2960},  {5,  17,  42}},
		{{860, 1280,  3000},  {8,  19,  34}},
		{{480, 1600,  2620},  {7,  15,  32}},
		{{300, 2500,  2880},  {3,  29,  29}},
		{{300, 1670,  2350},  {3,  31,  39}},
		{{480, 2220,  2640},  {7,  30,  31}},
		{{380, 1600,  2540},  {4,  22,  33}},
		{{630, 2140,  2880},  {7,  27,  27}},
		{{640,  900,  3000},  {8,  13,  39}},
		{{360, 2220,  2780},  {5,  23,  31}},
		/* male 13 */
		{{260,  780,  2460},  {5,  20,  44}},
		{{900, 1560,  2860},  {9,  13,  26}},
		{{440,  850,  2600},  {3,  17,  42}},
		{{860, 1140,  2820}, {10,  12,  30}},
		{{460, 1580,  2400},  {3,  19,  27}},
		{{260, 2560,  3240},  {3,  28,  34}},
		{{300, 1960,  2500},  {2,  31,  29}},
		{{460, 2320,  2960},  {6,  25,  27}},
		{{460, 1600,  2460},  {6,  22,  30}},
		{{680, 2100,  2940}, {14,  19,  25}},
		{{540,  800,  2740},  {7,  13,  36}},
		{{380, 2500,  2980},  {7,  20,  23}},
		/* male 14 */
		{{340,  720,  2500},  {6,  16,  47}},
		{{900, 1500,  3020}, {10,  13,  33}},
		{{450,  900,  2700},  {4,  16,  42}},
		{{600, 1000,  2720}, {12,  10,  31}},
		{{420, 1740,  2560},  {8,  21,  25}},
		{{360, 2500,  3000},  {8,  10,  13}},
		{{360, 1900,  2420},  {6,  16,  12}},
		{{380, 1780,  2420},  {4,  36,  18}},
		{{500, 1640,  2620},  {4,  23,  29}},
		{{600, 1940,  2700},  {3,  16,  19}},
		{{500,  800,  2800},  {3,  10,  38}},
		{{400, 2360,  2740},  {7,  13,  27}},
		/* male 15 */
		{{360,  780,  2320},  {5,  23,  50}},
		{{860, 1420,  2420},  {8,  15,  35}},
		{{440,  840,  2480},  {9,  20,  38}},
		{{660,  980,  2500},  {6,  13,  33}},
		{{460, 1660,  2200},  {2,  21,  31}},
		{{300, 2360,  2960},  {7,  29,  27}},
		{{320, 1740,  2220},  {2,  28,  30}},
		{{400, 2240,  2560},  {5,  23,  24}},
		{{480, 1420,  2220},  {6,  23,  37}},
		{{680, 1640,  2340},  {9,  21,  28}},
		{{560,  860,  2780},  {4,  15,  44}},
		{{440, 2120,  2500},  {2,  22,  23}},
		/* male 16 */
		{{360,  760,  2300},  {7,  23,  52}},
		{{660, 1000,  2500},  {9,  15,  35}},
		{{500,  920,  2520},  {7,  15,  37}},
		{{780, 1060,  2380},  {5,   9,  38}},
		{{440, 1560,  2260},  {7,  25,  31}},
		{{280, 2200,  2880},  {7,  43,  38}},
		{{380, 1720,  2200},  {7,  29,  39}},
		{{360, 2140,  2620},  {3,  26,  28}},
		{{360, 1600,  2400},  {4,  26,  33}},
		{{520, 1800,  2480},  {8,  32,  35}},
		{{540,  920,  2640},  {6,  20,  44}},
		{{340, 2080,  2680},  {3,  28,  27}},
		/* male 17 */
		{{400,  820,  2200},  {5,  15,  48}},
		{{1100, 1480,  2260}, {10,  14,  30}},
		{{520,  940,  2560},  {5,  15,  43}},
		{{660,  940,  2820}, {11,  11,  35}},
		{{500, 1720,  2400},  {5,  19,  23}},
		{{360, 2300,  3260}, {11,  25,  17}},
		{{360, 2100,  2420}, {10,  19,  20}},
		{{440, 2360,  2860},  {6,  20,  25}},
		{{500, 1760,  2600},  {6,  15,  25}},
		{{660, 1840,  2620},  {7,  17,  24}},
		{{540,  860,  2860},  {3,   9,  41}},
		{{400, 2440,  3000},  {5,  28,  30}},
		/* male 18 */
		{{360,  860,  2520},  {6,  15,  35}},
		{{740, 1300,  2660},  {7,   9,  22}},
		{{460,  800,  2620},  {3,  20,  35}},
		{{740, 1040,  2800}, {15,  13,  30}},
		{{440, 1400,  2200},  {4,  21,  25}},
		{{340, 2040,  2500},  {6,  18,  20}},
		{{340, 1340,  2040},  {3,  15,  25}},
		{{420, 1760,  2420},  {6,  18,  21}},
		{{460, 1380,  2200},  {6,  20,  22}},
		{{560, 1640,  2400},  {9,  13,  19}},
		{{540,  920,  2520},  {9,  10,  23}},
		{{400, 1800,  2400},  {8,  19,  21}},
		/* male 19 */
		{{320,  840,  2360},  {5,  20,  45}},
		{{780, 1140,  2740},  {8,  15,  36}},
		{{460, 1020,  2700},  {6,  11,  43}},
		{{800, 1100,  2720}, {15,  17,  42}},
		{{440, 1500,  2500},  {5,  23,  37}},
		{{260, 2000,  2680},  {4,  39,  40}},
		{{300, 1540,  2100},  {5,  33,  42}},
		{{400, 1900,  2680},  {6,  25,  34}},
		{{440, 1500,  2400},  {4,  26,  42}},
		{{600, 1700,  2640}, {18,  33,  40}},
		{{500,  800,  3000},  {5,  20,  47}},
		{{400, 2000,  2500},  {5,  27,  37}},
		/* male 20 */
		{{400,  960,  2400}, {10,  16,  37}},
		{{800, 1220,  2380},  {7,  13,  25}},
		{{500, 1080,  2500}, {10,  14,  31}},
		{{780, 1100,  2600}, {10,  11,  30}},
		{{400, 1480,  2380},  {8,  20,  25}},
		{{280, 2380,  2720},  {6,  23,  24}},
		{{300, 1760,  2220},  {5,  14,  22}},
		{{400, 2000,  2600},  {4,  20,  24}},
		{{440, 1500,  2440},  {4,  15,  20}},
		{{440, 1800,  2620}, {12,  18,  22}},
		{{460,  860,  2600}, {12,  13,  33}},
		{{400, 2040,  2640},  {3,  18,  23}},
		/* male 21 */
		{{300,  700,  2100},  {3,  15,  37}},
		{{800, 1100,  2300}, {11,  15,  30}},
		{{420,  700,  2440},  {5,  12,  33}},
		{{660,  920,  2520},  {6,   9,  32}},
		{{400, 1300,  2000},  {2,  22,  30}},
		{{300, 1940,  2620},  {5,  25,  24}},
		{{260, 1920,  2900},  {4,  21,  31}},
		{{300, 1900,  2340},  {4,  20,  24}},
		{{400, 1200,  2000},  {4,  20,  29}},
		{{540, 1500,  2280},  {4,  15,  22}},
		{{400,  740,  2580},  {3,  11,  34}},
		{{300, 1900,  2380},  {2,  20,  20}},
		/* male 22 */
		{{400,  780,  2500},  {6,  14,  40}},
		{{760, 1260,  2620},  {6,  14,  30}},
		{{540,  860,  2600},  {4,   7,  34}},
		{{660, 1100,  2460},  {9,  12,  30}},
		{{540, 1460,  2260},  {5,  12,  20}},
		{{300, 2300,  2800},  {7,  20,  17}},
		{{300, 1980,  2900},  {6,  14,  32}},
		{{420, 2100,  2600},  {4,  17,  21}},
		{{500, 1440,  2300},  {4,  18,  26}},
		{{540, 1900,  2600}, {10,  20,  22}},
		{{550,  900,  3000},  {5,  11,  34}},
		{{300, 2200,  2700},  {6,  14,  20}},
		/* male 23 */
		{{360,  900,  2140},  {3,  14,  37}},
		{{800, 1250,  2650}, {12,  13,  26}},
		{{520,  960,  2200},  {5,   7,  28}},
		{{760, 1120,  2700}, {12,  10,  29}},
		{{400, 1500,  2160},  {7,  20,  30}},
		{{300, 2260,  3000},  {4,  18,  17}},
		{{320, 1800,  2500},  {6,  17,  32}},
		{{460, 2020,  2800},  {3,  10,  16}},
		{{500, 1500,  2300},  {7,  15,  23}},
		{{640, 1600,  2500},  {8,  22,  32}},
		{{550,  940,  2420},  {7,  12,  40}},
		{{460, 2100,  2880},  {2,  21,  24}},
		/* male 24 */
		{{360,  860,  2460},  {4,  15,  33}},
		{{840, 1400,  2500},  {5,   8,  20}},
		{{460,  900,  2520},  {1,   8,  31}},
		{{620, 1020,  2770},  {7,   6,  26}},
		{{410, 1460,  2360},  {2,  11,  18}},
		{{270, 2140,  2580},  {2,  15,  21}},
		{{300, 1870,  2300},  {1,  11,  16}},
		{{360, 2000,  2520},  {2,  17,  19}},
		{{400, 1520,  2400},  {2,  12,  17}},
		{{600, 1600,  2580},  {3,  12,  20}},
		{{500,  900,  2700},  {2,  11,  33}},
		{{360, 1940,  2550},  {1,  19,  21}},
		/* male 25 */
		{{360,  860,  2200},  {8,  22,  46}},
		{{880, 1240,  2400},  {7,  14,  25}},
		{{460,  920,  3300},  {2,  11,  40}},
		{{600, 1000,  2600},  {4,   7,  32}},
		{{400, 1440,  2160},  {2,  31,  37}},
		{{360, 2240,  2760},  {6,  23,  21}},
		{{380, 1660,  2000}, {13,  16,  26}},
		{{460, 2000,  2520},  {5,  23,  32}},
		{{460, 1500,  2300},  {7,  17,  25}},
		{{540, 1700,  2460}, {10,  22,  35}},
		{{540, 1000,  2600},  {8,  13,  40}},
		{{340, 2040,  2580},  {9,  27,  30}},
		/* male 26 */
		{{400,  800,  2500},  {6,  23,  42}},
		{{960, 1300,  2640},  {9,   8,  29}},
		{{460,  860,  2460},  {7,  16,  39}},
		{{740, 1140,  2400},  {9,  10,  30}},
		{{400, 1600,  2400},  {6,  21,  25}},
		{{360, 2500,  2840}, {10,  19,  20}},
		{{360, 1800,  2400},  {8,  21,  27}},
		{{360, 2080,  2680},  {5,  14,  16}},
		{{400, 1620,  2440},  {5,  15,  21}},
		{{600, 1940,  2600},  {5,  13,  23}},
		{{560,  980,  2900},  {5,  13,  32}},
		{{400, 2060,  2540},  {5,  21,  21}},
		/* male 27 */
		{{300,  900,  2300},  {3,  15,  38}},
		{{780, 1300,  2400},  {9,  18,  32}},
		{{550, 1000,  2480},  {5,  10,  35}},
		{{680, 1050,  2550},  {5,  10,  35}},
		{{520, 1480,  2400},  {5,  16,  27}},
		{{260, 2180,  2560},  {1,  30,  30}},
		{{250, 1720,  2220},  {1,  26,  32}},
		{{360, 2100,  2650},  {4,  31,  25}},
		{{440, 1440,  2440},  {4,  21,  26}},
		{{600, 1600,  2500},  {5,  15,  27}},
		{{560,  950,  2700},  {3,  10,  40}},
		{{360, 1900,  2600},  {3,  26,  31}},
		/* male 28 */
		{{280,  740,  2500},  {3,  20,  45}},
		{{780, 1300,  2840},  {4,  12,  25}},
		{{440,  860,  2860},  {5,  13,  41}},
		{{440,  700,  3040},  {9,  10,  37}},
		{{450, 1520,  2320},  {2,  25,  30}},
		{{220, 2340,  2960},  {1,  35,  36}},
		{{240, 1800,  2140},  {3,  36,  37}},
		{{300, 2200,  2600},  {4,  28,  33}},
		{{440, 1500,  2480},  {3,  26,  37}},
		{{500, 1660,  2620}, {10,  30,  38}},
		{{420,  700,  3000},  {4,  12,  35}},
		{{300, 2140,  2760},  {6,  32,  31}},
		/* male 29 */
		{{340,  660,  2320},  {4,  13,  37}},
		{{640, 1250,  2480}, {10,  16,  29}},
		{{560, 1000,  2480}, {10,  14,  31}},
		{{720, 1150,  2600}, {10,  13,  26}},
		{{480, 1400,  2160},  {9,  22,  28}},
		{{300, 2040,  2640},  {5,  20,  22}},
		{{280, 1540,  1960},  {3,  21,  23}},
		{{460, 1760,  2320},  {7,  19,  20}},
		{{440, 1550,  2200},  {8,  16,  23}},
		{{480, 1660,  1960}, {10,  16,  23}},
		{{480,  840,  2840},  {9,  12,  28}},
		{{400, 1780,  2360},  {7,  20,  23}},
		/* male 30 */
		{{360,  800,  2540},  {1,  11,  40}},
		{{600, 1300,  2600},  {6,   8,  27}},
		{{500,  860,  2440},  {2,   7,  36}},
		{{750, 1140,  2640},  {4,   9,  30}},
		{{460, 1400,  2340},  {1,  23,  28}},
		{{340, 2300,  2620},  {2,  26,  25}},
		{{300, 1540,  2300},  {5,  25,  35}},
		{{440, 2000,  2540},  {1,  14,  19}},
		{{440, 1360,  2360},  {1,  19,  26}},
		{{620, 1840,  2560},  {3,  18,  23}},
		{{520,  820,  2680},  {2,   7,  34}},
		{{420, 2000,  2640},  {1,  20,  25}}, /*L2 (10) corrected 20021211 */
		/* male 31 */
		{{340,  740,  2240},  {5,  15,  44}},
		{{820, 1200,  2250},  {5,  17,  40}},
		{{440,  820,  2540},  {3,  13,  37}},
		{{760, 1060,  2340},  {8,  15,  42}},
		{{460, 1540,  2380},  {3,  21,  25}},
		{{280, 2260,  2620},  {8,  31,  32}},
		{{300, 1800,  2220},  {6,  36,  37}},
		{{460, 1900,  2260},  {3,  34,  31}},
		{{380, 1540,  2400}, {15,  36,  40}},
		{{500, 1740,  2400},  {4,  26,  39}},
		{{460,  840,  2580},  {7,  28,  35}},
		{{320, 2100,  2460},  {7,  30,  27}},
		/* male 32 */
		{{360,  900,  2200},  {5,  18,  39}},
		{{640, 1280,  2340},  {7,  15,  26}},
		{{460,  920,  2360},  {7,  14,  33}},
		{{720, 1200,  2580}, {11,  14,  26}},
		{{420, 1520,  2260},  {6,  17,  23}},
		{{400, 2000,  2560},  {5,  15,  20}},
		{{380, 1700,  2100},  {6,  18,  21}},
		{{440, 1740,  2420},  {4,  14,  17}},
		{{500, 1520,  2440},  {5,  14,  21}},
		{{580, 1540,  2460},  {5,  12,  24}},
		{{580, 1020,  2700},  {9,  13,  30}},
		{{460, 1720,  2400},  {7,  22,  20}},
		/* male 33 */
		{{400,  700,  2600},  {4,  17,  45}},
		{{900, 1440,  2600}, {10,  17,  36}},
		{{460,  860,  2600},  {7,  18,  45}},
		{{680, 1000,  2200},  {7,  13,  39}},
		{{460, 1600,  2540},  {5,  28,  37}},
		{{300, 2260,  2880},  {7,  28,  26}},
		{{320, 1860,  2200},  {7,  22,  28}},
		{{440, 2180,  2660},  {3,  24,  28}},
		{{380, 1560,  2360},  {8,  26,  33}},
		{{620, 1720,  2060},  {6,  21,  28}},
		{{600,  860,  2900}, {12,  18,  37}},
		{{440, 2040,  2600},  {4,  26,  30}},
		/* male 34 */
		{{370,  900,  2230},  {3,  17,  35}},
		{{700, 1200,  2580}, {12,  17,  30}},
		{{500,  840,  2460},  {4,  13,  37}},
		{{720, 1080,  2640},  {7,  13,  37}},
		{{440, 1300,  2220},  {4,  20,  31}},
		{{300, 2040,  2580},  {7,  25,  25}},
		{{320, 1540,  2080},  {8,  21,  23}},
		{{380, 1860,  2450},  {7,  24,  33}},
		{{460, 1200,  2360},  {3,  20,  30}},
		{{580, 1500,  2380},  {9,  22,  25}},
		{{480,  820,  2580},  {8,  15,  32}},
		{{400, 1800,  2360},  {6,  23,  26}},
		/* male 35 */
		{{280, 1040,  2340},  {7,  26,  41}},
		{{820, 1300,  2760}, {10,  15,  32}},
		{{440, 1220,  2580},  {6,  18,  29}},
		{{600, 1040,  2540},  {8,  13,  27}},
		{{420, 1560,  2480},  {6,  22,  26}},
		{{300, 2160,  2700},  {5,  29,  30}},
		{{250, 1760,  2320},  {9,  30,  38}},
		{{440, 1940,  2550},  {5,  25,  28}},
		{{400, 1600,  2460},  {8,  26,  29}},
		{{580, 1820,  2460},  {5,  23,  30}},
		{{460,  860,  2660},  {5,  21,  37}},
		{{400, 2100,  2640},  {8,  27,  27}},
		/* male 36 */
		{{360,  740,  2160},  {2,  21,  40}},
		{{660, 1260,  2540}, {10,  18,  21}},
		{{500,  900,  2600},  {9,  20,  30}},
		{{640, 1000,  2880}, {11,  17,  29}},
		{{460, 1300,  2140},  {8,  19,  25}},
		{{300, 1900,  2580}, {11,  18,  22}},
		{{320, 1660,  2060},  {9,  17,  20}},
		{{400, 1780,  2320},  {8,  20,  21}},
		{{380, 1360,  2200},  {6,  20,  25}},
		{{540, 1600,  2260},  {8,  21,  22}},
		{{540,  860,  2720},  {7,  20,  32}},
		{{400, 1740,  2340},  {5,  22,  23}},
		/* male 37 */
		{{300,  900,  2140},  {5,  21,  39}},
		{{700, 1240,  2460}, {10,  18,  28}},
		{{480,  960,  2140},  {4,  12,  32}},
		{{640, 1120,  2480},  {9,  14,  32}},
		{{460, 1520,  2160},  {4,  18,  25}},
		{{320, 2120,  2600}, {10,  20,  26}},
		{{320, 1800,  2200},  {8,  25,  27}},
		{{320, 1920,  2460},  {8,  21,  27}},
		{{480, 1460,  2260},  {5,  22,  27}},
		{{600, 1600,  2480},  {7,  17,  24}},
		{{500,  950,  2450},  {4,  14,  38}},
		{{460, 1820,  2480},  {6,  18,  26}},
		/* male 38 */
		{{320,  760,  2080}, {11,  23,  41}},
		{{840, 1180,  2700}, {13,  20,  37}},
		{{500,  920,  2400}, {13,  17,  37}},
		{{660, 1060,  2700}, {13,  17,  36}},
		{{440, 1400,  2220},  {5,  32,  37}},
		{{280, 2240,  2700}, {14,  29,  35}},
		{{300, 1640,  2080}, {12,  31,  31}},
		{{440, 2040,  2600}, {11,  19,  24}},
		{{400, 1460,  2160},  {8,  32,  38}},
		{{580, 1700,  1900}, {13,  26,  26}},
		{{500,  840,  2920}, {10,  18,  40}},
		{{360, 2060,  2440},  {7,  21,  27}},
		/* male 39 */
		{{320,  760,  2480},  {9,  21,  46}},
		{{700, 1420,  2680},  {9,  16,  31}},
		{{500,  940,  2500},  {4,  16,  41}},
		{{700, 1060,  2720},  {8,  10,  38}},
		{{440, 1580,  2260}, {11,  34,  39}},
		{{260, 2200,  2700},  {5,  29,  32}},
		{{200, 1600,  2060},  {7,  33,  34}},
		{{400, 2200,  2600}, {13,  29,  31}},
		{{380, 1500,  2220},  {8,  25,  37}},
		{{540, 1750,  2420},  {8,  23,  32}},
		{{520,  820,  2560}, {10,  19,  43}},
		{{400, 1700,  2320},  {3,  38,  23}},
		/* male 40 */
		{{300,  680,  1920},  {7,  28,  48}},
		{{740, 1200,  2550},  {8,  10,  24}},
		{{420,  860,  2420},  {7,  17,  37}},
		{{640, 1120,  2500}, {12,  17,  37}},
		{{360, 1500,  2180},  {3,  27,  35}},
		{{280, 2160,  2920},  {4,  27,  31}},
		{{260, 1560,  2050},  {2,  26,  27}},
		{{360, 2020,  2500},  {4,  26,  28}},
		{{440, 1400,  2320},  {4,  21,  32}},
		{{460, 1660,  2460},  {5,  19,  27}},
		{{500,  840,  2580},  {6,  14,  35}},
		{{360, 1920,  2560},  {3,  31,  31}},
		/* male 41 */
		{{360,  880,  2320},  {2,  12,  43}},
		{{840, 1200,  2500}, {12,  17,  37}},
		{{580, 1060,  2300}, {11,  10,  32}},
		{{580, 1100,  2680},  {8,  12,  33}},
		{{560, 1600,  2200},  {9,  22,  27}},
		{{300, 2260,  2800},  {5,  25,  26}},
		{{320, 1760,  2100},  {5,  23,  25}},
		{{500, 2020,  2660},  {5,  17,  21}},
		{{420, 1520,  2320},  {1,  20,  28}},
		{{700, 1800,  2620},  {8,  17,  22}},
		{{540,  860,  2720},  {7,  13,  35}},
		{{420, 2080,  2600},  {4,  21,  25}},
		/* male 42 */
		{{420,  800,  2400},  {5,  15,  40}},
		{{800, 1400,  2900},  {9,  16,  41}},
		{{420,  820,  2480},  {5,   8,  40}},
		{{600, 1200,  2760},  {6,  12,  34}},
		{{400, 1560,  2120},  {2,  20,  31}},
		{{320, 2360,  2820},  {8,  25,  27}},
		{{340, 1680,  2240},  {9,  19,  35}},
		{{400, 2180,  2760},  {2,  19,  20}},
		{{400, 1440,  2360},  {3,  15,  26}},
		{{700, 1700,  2340}, {11,  18,  29}},
		{{500,  780,  2840},  {7,  14,  38}},
		{{380, 2120,  2720},  {2,  21,  25}},
		/* male 43 */
		{{300,  760,  2020},  {3,  16,  38}},
		{{740, 1200,  2360},  {8,  15,  29}},
		{{460,  860,  2200},  {8,  12,  39}},
		{{620,  900,  2500},  {8,  12,  27}},
		{{400, 1340,  2100},  {7,  20,  31}},
		{{240, 2000,  2340},  {2,  22,  28}},
		{{240, 1580,  1860},  {3,  16,  24}},
		{{360, 1640,  2080},  {5,  19,  26}},
		{{400, 1340,  2060},  {3,  16,  28}},
		{{580, 1400,  2120},  {6,  13,  24}},
		{{500,  800,  2460},  {6,   6,  31}},
		{{440, 1720,  2100},  {7,  19,  24}},
		/* male 44 */
		{{260,  800,  2400},  {3,  16,  48}},
		{{780, 1300,  2700},  {6,  14,  28}},
		{{480,  900,  2500},  {5,   8,  35}},
		{{620, 1000,  2820},  {5,   9,  28}},
		{{420, 1400,  2300},  {3,  18,  29}},
		{{240, 2040,  2680},  {1,  31,  28}},
		{{260, 1580,  2260},  {3,  31,  28}},
		{{380, 2000,  2600},  {5,  29,  26}},
		{{420, 1420,  2400},  {2,  21,  26}},
		{{540, 1640,  2440},  {5,  19,  26}},
		{{480,  840,  2800},  {6,  13,  32}},
		{{280, 1960,  2560},  {5,  27,  28}},
		/* male 45 */
		{{300,  840,  3060},  {3,  10,  38}},
		{{800, 1220,  2280},  {6,  10,  26}},
		{{500,  920,  2120},  {6,   8,  31}},
		{{700, 1020,  2600},  {3,  11,  28}},
		{{400, 1260,  2020},  {6,  17,  24}},
		{{260, 1960,  2440},  {1,  22,  22}},
		{{300, 1480,  1940},  {2,  18,  22}},
		{{440, 1880,  2380},  {6,  17,  17}},
		{{320, 1400,  2140},  {5,  18,  27}},
		{{500, 1560,  2300},  {7,  18,  22}},
		{{540,  780,  2400},  {8,  13,  34}},
		{{360, 1860,  2300},  {4,  20,  21}},
		/* male 46 */
		{{320,  860,  2380},  {3,  19,  41}},
		{{660, 1400,  2540}, {11,  20,  27}},
		{{520,  940,  2580},  {7,  11,  34}},
		{{700, 1040,  2720},  {4,   8,  23}},
		{{400, 1600,  2280},  {2,  27,  29}},
		{{320, 2340,  3140},  {3,  33,  29}},
		{{300, 1860,  2160},  {2,  25,  25}},
		{{420, 2200,  2760},  {1,  17,  23}},
		{{460, 2320,  3360},  {2,  33,  37}},
		{{500, 2100,  2760},  {6,  23,  38}},
		{{600,  920,  2700},  {8,  17,  29}},
		{{420, 2200,  2740},  {3,  30,  32}},
		/* male 47 */
		{{360,  800,  2120},  {3,  18,  33}},
		{{700, 1220,  2760},  {6,  13,  27}},
		{{540,  940,  2640},  {2,   8,  33}},
		{{620, 1080,  2800},  {4,  10,  33}},
		{{500, 1400,  2200},  {6,  18,  25}},
		{{320, 2240,  2940},  {2,  22,  27}},
		{{320, 1800,  2100},  {1,  26,  27}},
		{{420, 2040,  2400},  {3,  19,  24}},
		{{460, 1440,  2140},  {1,  13,  25}},
		{{600, 1600,  2520},  {6,  13,  27}},
		{{560,  700,  2780},  {5,  11,  31}},
		{{440, 1920,  2560},  {3,  22,  24}},
		/* male 48 */
		{{300,  760,  1900},  {3,  17,  42}},
		{{800, 1260,  2740},  {7,  10,  28}},
		{{460,  840,  1840},  {4,  13,  38}},
		{{540,  900,  2400}, {10,  14,  28}},
		{{420, 1380,  2100},  {2,  16,  29}},
		{{220, 2080,  2900},  {2,  28,  21}},
		{{220, 1760,  2120},  {1,  22,  25}},
		{{440, 2060,  2780},  {1,  19,  21}},
		{{440, 1440,  2560},  {3,  19,  31}},
		{{580, 1400,  2100},  {5,  15,  22}},
		{{520,  900,  2300},  {3,  10,  32}},
		{{420, 1720,  2720},  {6,  22,  21}},
		/* male 49 */
		{{320, 1000,  2220},  {3,  24,  43}},
		{{700, 1280,  2500},  {3,  13,  30}},
		{{460, 1060,  2380},  {4,  13,  31}},
		{{620, 1100,  2840}, {10,  18,  33}},
		{{340, 1440,  2260},  {4,  21,  30}},
		{{280, 2140,  2580},  {3,  31,  32}},
		{{280, 1820,  2220},  {2,  36,  35}},
		{{340, 2100,  2500},  {2,  29,  31}},
		{{380, 1460,  2400},  {2,  26,  38}},
		{{500, 1640,  2500},  {7,  27,  31}},
		{{500,  960,  2720},  {4,  17,  31}},
		{{420, 1960,  2700},  {1,  32,  32}},
		/* male 50 */
		{{340,  780,  2020}, {11,  22,  36}},
		{{660, 1220,  2500},  {9,  14,  22}},
		{{420,  760,  2440},  {2,  17,  33}},
		{{560, 1000,  2600},  {6,  13,  25}},
		{{400, 1320,  2120},  {8,  18,  21}},
		{{300, 1860,  2440},  {6,  22,  22}},
		{{280, 1600,  1900},  {6,  16,  19}},
		{{340, 1740,  2260},  {3,  12,  17}},
		{{400, 1360,  2160},  {4,  16,  22}},
		{{520, 1580,  2240},  {2,  12,  16}},
		{{380,  800,  2560},  {7,  11,  25}},
		{{360, 1740,  2260},  {5,  14,  17}},
		/* 25*12 females */
		{{250,  800, 2450},  {0,  8, 45}},	// poet
		{{950, 1500, 2650},  {5, 14, 30}},	// paat
		{{500, 1050, 2600},  {3,  5, 38}},	// poot
		{{720, 1100, 2950},  {8,  2, 24}},	// pat
		{{500, 1800, 2500},  {6, 14, 30}},	// peut
		{{280, 2500, 3100},  {0, 32, 26}},	// piet
		{{250, 1700, 2200},  {0, 18, 21}},	// puut
		{{500, 2350, 2750},  {2, 12, 12}},	// peet
		{{520, 1550, 2400},  {4, 15, 27}},	// put
		{{750, 2000, 2600},  {4, 20, 20}},	// pet
		{{550,  900, 2800},  {6,  3, 34}},	// pot
		{{480, 2150, 2650},  {5, 20, 22}},	// pit
		/* female 2 */
		{{300,  750, 2700},  {0, 10, 50}},
		{{1100, 1500, 3000},  {6,  9, 28}}, // djmw 20021212 L3 (was 20)
		{{520,  900, 2800},  {2,  8, 30}},
		{{800, 1150, 3000},  {2, 12, 34}}, // djmw 20021212 F3 (was 300 in Van Nierop data!)
		{{450, 1600, 2950},  {2, 22, 31}},
		{{250, 2700, 3300},  {0, 27, 31}},
		{{300, 1900, 2650},  {0, 23, 33}},
		{{550, 2300, 3000},  {0, 17, 22}},
		{{450, 1900, 2800},  {0, 20, 32}},
		{{700, 2150, 3100},  {0, 22, 30}},
		{{600,  950, 2800},  {0, 10, 40}},
		{{350, 2450, 2900},  {0, 27, 28}},
		/* female 3 */
		{{350,  800, 2700},  {0, 19, 50}},
		{{1000, 1450, 2800},  {7,  7, 34}},
		{{500, 1000, 2600},  {4, 11, 45}},
		{{650, 1000, 2700},  {7,  1, 29}},
		{{460, 1650, 2400},  {2, 21, 35}},
		{{250, 2450, 2900},  {0, 30, 26}},
		{{250, 1750, 2500},  {0, 22, 31}},
		{{480, 2600, 3600},  {4, 22, 33}},
		{{450, 1650, 2600},  {2, 16, 32}},
		{{650, 2450, 2800},  {6, 19, 20}},
		{{650, 1050, 2800},  {1, 10, 42}},
		{{350, 2600, 3200},  {0, 24, 35}},
		/* female 4 */
		{{350,  750, 2500},  {1, 10, 36}},
		{{1000, 1550, 2900},  {6, 14, 18}},
		{{450,  950, 2450},  {2,  6, 30}},
		{{800, 1100, 3200},  {6,  6, 26}},
		{{450, 1900, 2400},  {3, 14, 14}},
		{{250, 2450, 3050},  {0, 15, 22}},
		{{300, 1900, 2350},  {0, 12, 15}},
		{{450, 2250, 2850},  {3, 16, 12}},
		{{430, 1700, 2650},  {2, 12, 17}},
		{{600, 2250, 2950},  {4, 12, 15}},
		{{600,  850, 3200},  {4,  3, 30}},
		{{430, 2500, 3000},  {2, 16, 24}},
		/* female 5 */
		{{300,  900, 2700},  {0, 12, 42}},
		{{750, 1600, 2800},  {2, 15, 24}},
		{{480,  950, 2600},  {2, 13, 32}},
		{{800, 1200, 2900},  {3, 13, 30}},
		{{430, 1500, 2600},  {1, 20, 25}},
		{{250, 2800, 3150}, { -1, 22, 32}},
		{{300, 1400, 2500},  {0, 24, 28}},
		{{420, 2400, 2900},  {2, 27, 27}},
		{{500, 1650, 2850},  {1, 16, 25}},
		{{750, 1900, 3000},  {4, 25, 30}},
		{{600, 1200, 2850},  {3, 17, 23}},
		{{490, 1950, 2900},  {2, 29, 28}},
		/* female 6 */
		{{300,  750, 2350},  {2,  8, 37}}, // djmw 20021212 L2 (was  0)
		{{950, 1400, 2400},  {2, 12, 23}},
		{{650, 1100, 2200},  {4,  8, 34}}, // djmw 20021212 L2 (was  0)
		{{900, 1100, 2600},  {2,  6, 26}}, // djmw 20021212 L3 (was 20)
		{{490, 1700, 2400},  {1, 18, 24}}, // djmw 20021212 L2 (was 14)
		{{300, 2500, 2800},  {0, 16, 21}},
		{{300, 1800, 2400},  {0, 10, 14}},
		{{470, 2400, 2750},  {2, 17, 19}},
		{{570, 1750, 2550},  {0, 17, 18}}, // djmw 20021212 L3 (was 19)
		{{700, 1750, 2400},  {0, 10, 16}},
		{{600, 1200, 2500},  {0, 18, 28}}, // djmw 20021212 L2,3 (was 10,20)
		{{440, 2250, 2700},  {0, 17, 18}},
		/* female 7 */
		{{350,  850, 2600},  { -1, 16, 38}},
		{{950, 1200, 2900},  {7,  9, 40}},
		{{550, 1050, 2500},  {2,  6, 38}},
		{{850, 1200, 2800},  { -1, 12, 39}},
		{{440, 1800, 2500},  {0, 24, 30}},
		{{300, 2300, 3000},  { -1, 22, 30}},
		{{350, 1850, 2400},  { -1, 23, 27}},
		{{460, 2400, 2900},  {1, 28, 30}}, // djmw 20021212 L2 (was 20)
		{{490, 1650, 2700},  {1, 22, 28}},
		{{650, 1700, 2750},  {2, 28, 28}}, // djmw 20021212 L2 (was 20)
		{{450,  700, 3000},  {6,  2, 33}}, // djmw 20021212 L2 (was  8)
		{{440, 2550, 3000},  {1, 40, 41}},
		/* female 8 */
		{{350, 1000, 2500},  {2, 19, 40}},
		{{950, 1400, 2700},  {8, 14, 29}},
		{{500,  950, 2450},  {6, 10, 34}},
		{{850, 1230, 2950},  {8, 19, 42}},
		{{520, 1650, 2600},  {6, 22, 28}},
		{{250, 2500, 3000},  {0, 25, 30}},
		{{250, 1750, 2300},  {0, 23, 29}},
		{{510, 2250, 2850},  {4, 30, 24}},
		{{650, 1600, 2600},  {6, 18, 26}},
		{{720, 1750, 2650},  {4, 19, 27}},
		{{750, 1150, 2650},  {2, 15, 37}},
		{{510, 2050, 2950},  {2, 22, 23}},
		/* female 9 */
		{{300,  850, 2650},  {0, 14, 43}},
		{{950, 1350, 2700},  {4,  6, 25}},
		{{550,  900, 2700},  {5,  4, 37}},
		{{930, 1300, 2800},  {0, 15, 35}},
		{{550, 1500, 2400},  {0, 10, 22}},
		{{280, 2450, 3000},  {0, 13, 21}},
		{{300, 1650, 2400},  {0, 26, 32}},
		{{530, 2400, 3000},  {0,  4, 14}},
		{{600, 1600, 2000},  {0, 14, 16}},
		{{850, 1950, 2800},  {1, 12, 19}},
		{{750, 1050, 2900},  {2,  4, 32}},
		{{550, 2300, 2800},  {1, 14, 22}},
		/* female 10 */
		{{350,  800, 2500},  {0, 21, 44}},
		{{1100, 1450, 2900},  {5,  9, 31}},
		{{400,  850, 2600},  {0,  5, 46}},
		{{750, 1050, 2950},  {4,  9, 25}},
		{{420, 1650, 2550},  {0, 17, 38}},
		{{300, 2450, 3050},  { -1, 28, 35}},
		{{280, 1850, 2500},  { -1, 20, 30}},
		{{420, 2400, 2800},  {1, 25, 29}},
		{{420, 1500, 2350},  {0, 22, 34}},
		{{650, 1850, 2800},  {2, 28, 31}},
		{{550,  800, 2800},  {4,  7, 44}},
		{{420, 2350, 2750},  {2, 27, 28}},
		/* female 11 */
		{{300,  800, 2850},  { -1, 22, 54}},
		{{1150, 1550, 2850},  {6,  9, 26}},
		{{600, 1000, 2700},  {3,  9, 30}},
		{{750, 1100, 2700},  {2, 10, 34}},
		{{500, 1800, 2550},  {3, 17, 24}},
		{{250, 2400, 3100},  {0, 27, 35}},
		{{250, 2000, 3650},  {0, 30, 36}},
		{{420, 2400, 2900},  {5, 28, 31}}, // djmw 20021212 L2,3 (was 20,34)
		{{470, 1700, 2500},  {4, 26, 37}}, // djmw 20021212 L2 (was 20)
		{{700, 1880, 2650},  {2, 21, 26}},
		{{650,  900, 2650},  {5,  4, 33}},
		{{490, 2250, 2650},  {3, 22, 26}},
		/* female 12 */
		{{300,  900, 2500},  { -1, 20, 39}},
		{{950, 1500, 2700},  {3, 16, 20}},
		{{450, 1000, 2650},  {2,  8, 35}},
		{{600, 1050, 2800},  {4,  4, 33}},
		{{420, 1650, 2500},  {1, 14, 22}},
		{{250, 2300, 2850},  {0, 30, 24}},
		{{300, 1800, 2300},  { -1, 21, 28}},
		{{400, 2100, 2700},  {0, 29, 20}},
		{{420, 1650, 2450},  {0, 17, 28}},
		{{550, 1800, 2650},  {2, 19, 20}},
		{{450, 1050, 2800},  {2, 10, 26}},
		{{420, 1850, 2700},  {0, 27, 26}},
		/* female 13 */
		{{300,  850, 2800},  {0, 19, 47}},
		{{1100, 1400, 2800},  {6, 10, 25}},
		{{550,  900, 2900},  {2, 16, 44}},
		{{700, 1050, 2750},  {4,  6, 33}},
		{{490, 1700, 2500},  {0, 33, 35}},
		{{300, 2500, 2950},  {0, 40, 35}},
		{{300, 1900, 2450},  {0, 24, 36}},
		{{500, 2500, 3000},  {4, 31, 32}},
		{{460, 1450, 2500},  {6, 33, 38}},
		{{700, 1700, 2700},  {1, 18, 30}},
		{{500,  800, 2700},  {2,  6, 38}},
		{{550, 2250, 2750},  {3, 28, 26}},
		/* female 14 */
		{{350,  900, 3600},  {0, 16, 44}},
		{{950, 1420, 2950},  {5,  7, 26}},
		{{400, 1000, 2400},  {0, 12, 36}},
		{{600,  950, 2650},  {4,  7, 31}},
		{{420, 1400, 2500},  {1, 21, 24}},
		{{300, 2500, 2850},  {0, 26, 27}},
		{{250, 1450, 2200},  {0, 29, 36}},
		{{420, 1950, 2700},  {0, 20, 20}},
		{{410, 1550, 2400},  {0, 20, 28}},
		{{600, 1650, 2600},  {3,  9, 21}},
		{{430,  880, 2950},  {2,  6, 32}},
		{{410, 1950, 2250},  {0, 31, 30}},
		/* female 15 */
		{{350, 1000, 2350},  {0, 29, 41}},
		{{850, 1100, 2800},  {2,  5, 25}},
		{{600, 1000, 2800},  {2,  6, 30}},
		{{850, 1400, 2600},  {1, 20, 31}},
		{{470, 2000, 2550},  {2, 16, 18}},
		{{250, 2650, 3250},  {0, 31, 40}},
		{{300, 2000, 2400},  {0, 23, 24}},
		{{430, 2400, 2900},  {1, 22, 27}},
		{{460, 1900, 2500},  {2, 25, 28}},
		{{650, 1900, 2900},  {2, 19, 23}},
		{{600,  900, 2400},  {3,  7, 35}},
		{{460, 2400, 2900},  {1, 26, 27}},
		/* female 16 */
		{{300,  850, 3300},  {0, 20, 52}},
		{{1050, 1400, 2450},  {6,  8, 31}},
		{{500,  950, 2400},  {4, 10, 44}},
		{{850, 1050, 2850},  {6,  7, 37}},
		{{480, 1650, 2300},  {2, 14, 32}},
		{{300, 3000, 4200},  {0, 32, 36}},
		{{300, 1900, 2450},  {0, 27, 37}},
		{{460, 2450, 3000},  {1, 32, 30}},
		{{500, 1550, 3100},  {2, 18, 40}},
		{{650, 1800, 2700},  {2, 22, 32}},
		{{450,  750, 2850},  {4,  2, 38}},
		{{440, 2200, 2900},  {0, 30, 29}},
		/* female 17 */
		{{320,  850, 2500},  {0, 18, 50}}, // djmw 20021212 L2 (was 10)
		{{1100, 1350, 2900},  {7,  9, 25}},
		{{450,  900, 2600},  {2,  8, 39}}, // djmw 20021212 L3 (was 30)
		{{800, 1200, 2800},  {2, 14, 32}},
		{{480, 1800, 2500},  {2, 24, 29}},
		{{300, 2600, 3000},  {0, 32, 32}},
		{{300, 1900, 2400},  {0, 25, 28}},
		{{460, 2300, 2900},  {2, 26, 29}},
		{{480, 1800, 2650},  {2, 27, 28}}, // djmw 20021212 L3 (was 20)
		{{650, 1850, 3000},  {4, 22, 34}},
		{{550,  800, 2850},  {4,  4, 30}},
		{{470, 2200, 3000},  {2, 28, 33}}, // djmw 20021212 L2 (was 20)
		/* female 18 */
		{{350,  750, 2550},  {0, 14, 48}}, // djmw 20021212 L3 (was 40)
		{{1050, 1700, 2850},  {0, 14, 22}},
		{{550, 1000, 2600},  {3,  4, 32}},
		{{750, 1150, 2950},  {5,  3, 30}},
		{{550, 1750, 2600},  {2, 12, 19}},
		{{300, 2400, 3000},  {0, 21, 22}},
		{{300, 1950, 2500},  {0, 18, 22}},
		{{510, 2200, 3000},  {1, 24, 27}},
		{{490, 1750, 2450},  {2, 23, 29}},
		{{600, 2200, 3000},  {3, 14, 23}},
		{{700,  850, 2900},  {6,  6, 38}},
		{{540, 2200, 3000},  {9, 22, 22}},
		/* female 19 */
		{{350,  950, 2800},  {0, 24, 51}},
		{{1000, 1500, 2900},  {6, 10, 26}},
		{{450,  950, 2800},  {2, 16, 48}},
		{{700, 1050, 3000},  {5,  5, 35}},
		{{450, 1700, 2450},  {4, 23, 28}},
		{{250, 2600, 3000},  {0, 28, 34}},
		{{330, 1900, 2700},  {0, 30, 42}},
		{{520, 2500, 2900},  {2, 28, 30}},
		{{450, 1700, 2800},  {1, 24, 30}},
		{{700, 1850, 3000},  {2, 23, 24}},
		{{600,  900, 3150},  {2, 12, 38}},
		{{490, 2400, 3100},  {2, 35, 31}},
		/* female 20 */
		{{300,  950, 2650},  {1, 21, 29}},
		{{680, 1250, 2600},  {3, 23, 50}},
		{{400,  850, 2700},  {3, 16, 44}},
		{{500, 1050, 3000},  {4, 13, 42}},
		{{390, 1550, 2500},  {2, 28, 35}},
		{{250, 2300, 2900},  {0, 36, 38}},
		{{250, 1950, 3600},  {0, 34, 45}},
		{{420, 2100, 2600},  {2, 25, 34}},
		{{420, 2100, 2600},  {2, 26, 34}},
		{{600, 1500, 2800},  {5, 25, 36}},
		{{450,  750, 2800},  {2, 11, 48}},
		{{440, 2050, 2700},  {1, 27, 38}},
		/* female 21 */
		{{330,  750, 2900},  {1,  8, 44}},
		{{1150, 1500, 2750},  {3,  8, 35}},
		{{530,  950, 2850},  {0,  8, 40}},
		{{850, 1050, 3000},  {5,  5, 36}},
		{{500, 1700, 2700},  {0, 20, 31}},
		{{300, 2600, 3100},  { -1, 23, 28}},
		{{300, 1800, 2400},  {0, 22, 27}},
		{{520, 2550, 3050},  {0, 22, 26}},
		{{520, 1650, 2550},  {0, 20, 28}},
		{{700, 1850, 2900},  {0, 16, 24}},
		{{650, 1100, 2900},  { -1, 11, 36}},
		{{450, 2300, 2800},  {0, 16, 17}},
		/* female 22 */
		{{300,  850, 2800},  {0, 15, 42}},
		{{1080, 1400, 2750},  {2,  3, 21}},
		{{480,  950, 2300},  {0,  6, 30}},
		{{800, 1100, 2800},  {3,  3, 28}},
		{{470, 1800, 2500},  {0, 12, 23}},
		{{300, 2350, 2850},  { -1, 31, 31}},
		{{300, 2100, 2600},  { -1, 25, 31}},
		{{460, 2600, 2900},  {0, 20, 23}},
		{{470, 1800, 2500},  {0, 16, 29}},
		{{600, 2000, 2700},  {2, 13, 11}},
		{{500,  750, 2650},  {4,  2, 24}},
		{{500, 2300, 2750},  {0, 20, 22}},
		/* female 23 */
		{{350,  700, 2800},  {0, 10, 46}},
		{{950, 1500, 2900},  {0, 19, 23}},
		{{530,  980, 2850},  {0, 14, 34}},
		{{750, 1100, 3000},  {4,  1, 27}},
		{{520, 1800, 2450},  {2, 20, 25}},
		{{300, 2550, 2950},  {0, 22, 24}},
		{{300, 1950, 2450},  {0, 30, 31}},
		{{500, 2300, 2800},  {0, 21, 29}},
		{{550, 1700, 2750},  {0, 21, 24}},
		{{650, 1850, 2700},  { -1, 16, 20}},
		{{650,  980, 2650},  {1,  4, 34}},
		{{500, 2400, 2800},  {4, 16, 18}},
		/* female 24 */
		{{300,  900, 2900},  {0, 12, 45}},
		{{900, 1650, 2850},  {1,  6, 23}},
		{{500,  900, 2300},  {0,  4, 39}},
		{{750, 1150, 2750},  {2,  6, 27}},
		{{550, 1650, 2500},  {0, 16, 24}},
		{{300, 2500, 2850},  {0, 27, 34}},
		{{250, 1850, 2300},  {0, 30, 26}},
		{{490, 2400, 2850},  {2, 16, 18}},
		{{520, 1800, 2450},  {3, 22, 23}},
		{{650, 2100, 2800},  {2, 14, 13}},
		{{550, 1100, 3900},  {2, 11, 32}},
		{{510, 2350, 2950},  {1, 24, 23}}, // djmw 20021212 L3 (was 33)
		/* female 25 */
		{{300,  800, 3700},  {0, 13, 50}}, // djmw 20021212 L2 (was 17)
		{{1000, 1450, 2650},  {4,  7, 24}},
		{{550, 1000, 2850},  {8, 10, 43}},
		{{750, 1050, 3000},  {2,  6, 33}},
		{{500, 1550, 2400},  {4, 16, 25}},
		{{250, 2400, 2950},  {0, 26, 24}},
		{{300, 1800, 2400},  {0, 26, 26}},
		{{490, 2300, 2800},  {8, 15, 20}},
		{{550, 1550, 2500},  {1, 14, 25}},
		{{700, 1950, 2750},  {0,  4, 13}},
		{{630,  980, 2850},  {5,  7, 39}},
		{{490, 2300, 2900},  {4, 17, 22}}
	};

	try {
		autoTable me = Table_create (nrows, ncols);

		for (integer i = 1; i <= nrows; i ++) {
			TableRow row = my rows.at [i];
			const integer vowel_id = ( (i - 1) % 12) + 1;	/* 1 - 12 */
			const integer speaker_id = (i - 1) / 12 + 1;  /* 1 - 75 */
			const integer speaker_sex = ( speaker_id <= 50 ? 0 : 1 );

			row -> cells [1]. string = Melder_dup (sex [speaker_sex]);
			row -> cells [2]. string = Melder_dup (Melder_integer (speaker_id));
			row -> cells [3]. string = Melder_dup (vowel [vowel_id - 1]);
			row -> cells [4]. string = Melder_dup (ipa [vowel_id - 1]);
			for (integer j = 0; j <= 2; j ++) {
				row -> cells [j + 5]. string = Melder_dup (Melder_integer (polsdata [i - 1]. f [j]));
				row -> cells [j + 8]. string = Melder_dup (Melder_integer (polsdata [i - 1]. l [j]));
			}
		}
		for (integer j = 1; j <= ncols; j ++) {
			Table_renameColumn_e (me.get(), j, columnLabels [j - 1]);
			my columnHeaders [j]. numericized = false;
		}
		return me;
	} catch (MelderError) {
		Melder_throw (U"Table not created from Pols & van Nierop data.");
	}
}

autoTable Table_create_weenink1983 () {
	const integer nrows = 360, ncols = 9;
	const conststring32 columnLabels [9] = {U"Type", U"Sex", U"Speaker", U"Vowel", U"IPA", U"F0", U"F1", U"F2", U"F3"};
	const conststring32 type [3] = {U"m", U"w", U"c"};
	/*
		Our vowel order: "oe", "o", "oo", "a", "aa", "u", "eu", "uu", "ie", "i", "ee", "e".
		To Pols & van Nierop order
	*/
	const integer order [13] = { 0, 1, 5, 3, 4, 7, 9, 8, 11, 6, 12, 2, 10};
	const conststring32 vowel [13] = {U"", U"oe", U"aa", U"oo", U"a", U"eu", U"ie", U"uu", U"ee", U"u", U"e", U"o", U"i"};
	const conststring32 ipa [13] = {U"", U"u", U"a", U"o", U"\\as", U"\\o/", U"i", U"y", U"e", U"\\yc", U"\\ep", U"\\ct", U"\\ic"};
	const conststring32 sex [2] = {U"m", U"f"};
	const struct weeninkdatum {
		short f [4];	  // f0, f1, f2, f3
	} weeninkdata [] = {
		{{182, 335,  748, 2332}},
		{{170, 435,  669, 2886}},
		{{164, 488,  911, 2362}},
		{{170, 671, 1005, 2822}},
		{{160, 787, 1372, 3313}},
		{{165, 396, 1624, 2374}},
		{{165, 472, 1585, 2344}},
		{{180, 293, 1791, 2177}},
		{{176, 314, 2149, 3215}},
		{{164, 393, 2120, 2694}},
		{{161, 454, 1982, 2673}},
		{{165, 502, 1902, 2632}},
		/* Speaker 2 */
		{{161, 308,  709, 2222}},
		{{155, 461,  760, 2632}},
		{{148, 510,  907, 2532}},
		{{153, 645, 1004, 2604}},
		{{145, 758, 1244, 2658}},
		{{155, 449, 1489, 2279}},
		{{145, 454, 1388, 2308}},
		{{162, 314, 1620, 2111}},
		{{158, 303, 1984, 2835}},
		{{150, 429, 1888, 2502}},
		{{156, 460, 1748, 2442}},
		{{150, 580, 1679, 2383}},
		/* Speaker 3 */
		{{124, 343,  719, 2107}},
		{{124, 519,  818, 1975}},
		{{125, 532,  937, 2044}},
		{{126, 721, 1135, 2077}},
		{{125, 850, 1328, 2299}},
		{{131, 390, 1307, 2177}},
		{{129, 481, 1438, 2203}},
		{{127, 270, 1705, 2068}},
		{{121, 275, 2081, 2882}},
		{{129, 377, 2044, 2638}},
		{{131, 464, 1949, 2536}},
		{{124, 645, 1854, 2488}},
		/* Speaker 4 */
		{{119, 322,  590, 2146}},
		{{110, 438,  703, 2392}},
		{{106, 489,  845, 2157}},
		{{120, 708, 1126, 2302}},
		{{107, 818, 1333, 2312}},
		{{109, 406, 1505, 2133}},
		{{109, 422, 1518, 2116}},
		{{115, 298, 1581, 2162}},
		{{119, 268, 2189, 2950}},
		{{109, 400, 2151, 2870}},
		{{108, 420, 2075, 2495}},
		{{108, 550, 1905, 2383}},
		/* Speaker 5 */
		{{132, 282,  656, 2355}},
		{{118, 462,  662, 2852}},
		{{117, 482,  797, 2531}},
		{{115, 723, 1036, 2456}},
		{{117, 732, 1356, 2562}},
		{{118, 454, 1516, 2178}},
		{{119, 466, 1484, 2139}},
		{{129, 271, 1642, 2154}},
		{{122, 255, 2390, 2787}},
		{{120, 353, 2114, 2507}},
		{{114, 441, 2020, 2392}},
		{{122, 601, 1858, 2315}},
		/* Speaker 6 */
		{{155, 275,  560, 2397}},
		{{141, 395,  705, 2802}},
		{{149, 419,  831, 1824}},
		{{142, 674, 1113, 2455}},
		{{137, 696, 1262, 2778}},
		{{148, 419, 1493, 2242}},
		{{148, 422, 1599, 2319}},
		{{157, 291, 1585, 2160}},
		{{152, 281, 2324, 3124}},
		{{144, 358, 2269, 2936}},
		{{146, 426, 2174, 2786}},
		{{140, 560, 1941, 2722}},
		/* Speaker 7 */
		{{165, 315,  703, 2304}},
		{{138, 531,  837, 2338}},
		{{130, 471,  838, 2391}},
		{{135, 713, 1218, 2466}},
		{{140, 807, 1478, 2390}},
		{{126, 467, 1639, 2227}},
		{{135, 504, 1520, 2268}},
		{{151, 341, 1738, 2246}},
		{{145, 302, 2255, 2887}},
		{{140, 379, 2183, 2614}},
		{{133, 400, 2198, 2551}},
		{{137, 587, 2024, 2516}},
		/* Speaker 8 */
		{{179, 361,  730, 2256}},
		{{168, 500,  714, 2782}},
		{{165, 495,  961, 2405}},
		{{162, 642,  988, 2308}},
		{{156, 921, 1409, 2909}},
		{{166, 451, 1482, 2657}},
		{{175, 505, 1546, 2443}},
		{{177, 341, 1705, 2291}},
		{{176, 285, 2288, 2817}},
		{{173, 364, 2126, 2907}},
		{{153, 492, 2102, 2778}},
		{{158, 628, 1777, 2903}},
		/* Speaker 9 */
		{{180, 358,  700, 2672}},
		{{177, 524,  759, 2784}},
		{{169, 557,  856, 2727}},
		{{178, 685, 1075, 2677}},
		{{168, 855, 1311, 2701}},
		{{170, 499, 1496, 2725}},
		{{170, 502, 1385, 2112}},
		{{178, 348, 1504, 2129}},
		{{176, 318, 2252, 2846}},
		{{169, 422, 2186, 2816}},
		{{159, 509, 1941, 2732}},
		{{165, 582, 1920, 2773}},
		/* Speaker 10 */
		{{161, 277,  592, 2493}},
		{{141, 490,  718, 2757}},
		{{137, 392,  692, 2512}},
		{{142, 642,  991, 2707}},
		{{144, 753, 1306, 2617}},
		{{140, 388, 1570, 2175}},
		{{139, 357, 1675, 2105}},
		{{156, 248, 1846, 2117}},
		{{148, 258, 2280, 3350}},
		{{149, 367, 2129, 2731}},
		{{142, 357, 2163, 2624}},
		{{143, 581, 1865, 2572}},
		/* Speaker 11 */
		{{282, 305,  842, 2366}},
		{{273, 557,  986, 2677}},
		{{277, 606, 1090, 2422}},
		{{274, 819, 1327, 2819}},
		{{277, 938, 1580, 2953}},
		{{278, 503, 1826, 2518}},
		{{274, 579, 1630, 2501}},
		{{286, 312, 2065, 3737}},
		{{282, 291, 2540, 3225}},
		{{278, 506, 2270, 3040}},
		{{277, 559, 2245, 2846}},
		{{277, 770, 2137, 2919}},
		/* Speaker 12 */
		{{255, 307,  738, 2529}},
		{{265, 505,  938, 2641}},
		{{228, 524, 1067, 2510}},
		{{233, 902, 1183, 2603}},
		{{215, 841, 1504, 2712}},
		{{239, 464, 1655, 2556}},
		{{234, 443, 1754, 2607}},
		{{245, 303, 1762, 2460}},
		{{241, 277, 2439, 3407}},
		{{262, 490, 2371, 3198}},
		{{228, 484, 2044, 2593}},
		{{234, 495, 2296, 3204}},
		/* Speaker 13 */
		{{298, 309,  627, 3029}},
		{{267, 785,  918, 2956}},
		{{259, 600,  948, 2684}},
		{{257, 893, 1136, 2844}},
		{{245, 1028, 1498, 2715}},
		{{258, 501, 1746, 2529}},
		{{265, 536, 1799, 2601}},
		{{280, 282, 1391, 2520}},
		{{270, 275, 2664, 3174}},
		{{260, 511, 2320, 2882}},
		{{255, 506, 2482, 2846}},
		{{258, 783, 2027, 2606}},
		/* Speaker 14 */
		{{187, 327,  586, 2463}},
		{{177, 520,  690, 2828}},
		{{175, 540,  956, 2599}},
		{{178, 596, 1023, 2750}},
		{{184, 692, 1382, 2432}},
		{{186, 397, 1658, 2422}},
		{{175, 512, 1674, 2458}},
		{{186, 291, 1737, 2236}},
		{{203, 271, 2299, 3260}},
		{{187, 369, 2263, 2903}},
		{{182, 471, 2057, 2729}},
		{{196, 593, 2122, 2779}},
		/* Speaker 15 */
		{{263, 448,  830, 2687}},
		{{230, 570, 1010, 2614}},
		{{244, 595, 1109, 2686}},
		{{240, 806, 1386, 2591}},
		{{239, 890, 1574, 2965}},
		{{244, 482, 1900, 2710}},
		{{251, 502, 1529, 2716}},
		{{238, 436, 2134, 2626}},
		{{254, 374, 2580, 3181}},
		{{252, 494, 2323, 2989}},
		{{242, 484, 2323, 3025}},
		{{246, 595, 1897, 2865}},
		/* Speaker 16 */
		{{238, 325,  737, 2290}},
		{{224, 462,  764, 2600}},
		{{202, 577,  944, 2651}},
		{{216, 674, 1084, 2851}},
		{{210, 833, 1333, 2256}},
		{{214, 427, 1708, 2346}},
		{{202, 502, 1546, 2326}},
		{{242, 289, 1669, 2169}},
		{{232, 337, 2060, 2841}},
		{{224, 416, 2217, 2769}},
		{{214, 491, 1950, 2674}},
		{{188, 548, 2071, 2727}},
		/* Speaker 17 */
		{{277, 313,  578, 2457}},
		{{254, 549,  855, 2793}},
		{{250, 608,  904, 2724}},
		{{267, 890, 1294, 2732}},
		{{263, 1047, 1578, 2848}},
		{{289, 549, 1728, 2844}},
		{{258, 508, 1747, 2799}},
		{{284, 287, 1982, 2793}},
		{{268, 268, 2873, 4075}},
		{{258, 440, 2590, 3090}},
		{{248, 489, 2529, 3111}},
		{{261, 606, 2078, 2980}},
		/* Speaker 18 */
		{{330, 334,  729, 3332}},
		{{247, 629,  864, 3024}},
		{{234, 521,  935, 2760}},
		{{264, 964, 1246, 3112}},
		{{261, 1008, 1573, 2932}},
		{{278, 509, 1887, 2596}},
		{{243, 487, 1799, 2393}},
		{{303, 306, 1781, 2415}},
		{{282, 296, 2597, 3571}},
		{{217, 425, 2636, 3314}},
		{{250, 490, 2642, 3494}},
		{{275, 787, 2326, 2992}},
		/* Speaker 19 */
		{{228, 322,  661, 2721}},
		{{214, 589,  821, 2729}},
		{{219, 457,  857, 2582}},
		{{229, 801, 1260, 2911}},
		{{220, 898, 1628, 2770}},
		{{233, 463, 1630, 2306}},
		{{225, 456, 1671, 2381}},
		{{225, 283, 1779, 2211}},
		{{224, 276, 2495, 3372}},
		{{224, 440, 2385, 3007}},
		{{216, 435, 2363, 2866}},
		{{212, 637, 2130, 3029}},
		/* Speaker 20 */
		{{246, 324,  730, 2644}},
		{{230, 636,  923, 2757}},
		{{233, 534,  954, 2542}},
		{{234, 820, 1256, 3064}},
		{{248, 966, 1695, 2890}},
		{{237, 484, 1753, 2718}},
		{{240, 480, 1917, 2670}},
		{{247, 291, 1981, 2701}},
		{{243, 290, 2648, 3176}},
		{{256, 501, 2358, 3058}},
		{{238, 469, 2406, 3008}},
		{{236, 693, 2128, 2991}},
		/* Speaker 21 */
		{{342, 353,  758, 3269}},
		{{309, 627,  976, 3256}},
		{{334, 663, 1017, 3075}},
		{{321, 964, 1576, 3186}},
		{{316, 1129, 2249, 3404}},
		{{330, 650, 2336, 3284}},
		{{311, 636, 2083, 3328}},
		{{331, 357, 2389, 3185}},
		{{332, 337, 2960, 3595}},
		{{351, 586, 2833, 3595}},
		{{312, 621, 2839, 3614}},
		{{309, 765, 2755, 3636}},
		/* Speaker 22 */
		{{326, 359,  682, 2948}},
		{{318, 631,  795, 3537}},
		{{296, 688, 1135, 2930}},
		{{285, 746, 1135, 1926}},
		{{290, 1006, 2166, 3263}},
		{{326, 632, 1990, 3042}},
		{{285, 570, 2015, 3155}},
		{{321, 324, 2025, 3104}},
		{{338, 346, 3069, 3573}},
		{{320, 636, 3025, 3709}},
		{{306, 611, 2770, 3648}},
		{{331, 893, 2639, 3566}},
		/* Speaker 23 */
		{{312, 414,  828, 2908}},
		{{314, 609,  936, 3139}},
		{{305, 659, 1201, 3015}},
		{{298, 890, 1303, 3003}},
		{{300, 1150, 2116, 3205}},
		{{275, 537, 1968, 3032}},
		{{277, 534, 1997, 3009}},
		{{304, 398, 2298, 2940}},
		{{302, 330, 3010, 3746}},
		{{312, 547, 2965, 3673}},
		{{275, 551, 2786, 3442}},
		{{299, 641, 2662, 3410}},
		/* Speaker 24 */
		{{301, 315,  812, 3357}},
		{{316, 621,  960, 3701}},
		{{305, 604, 1179, 3659}},
		{{331, 981, 1701, 3374}},
		{{322, 954, 2010, 3291}},
		{{297, 581, 2085, 3288}},
		{{307, 613, 1991, 3392}},
		{{322, 331, 2099, 3096}},
		{{326, 326, 3235, 3935}},
		{{307, 547, 2747, 3618}},
		{{308, 613, 2763, 3710}},
		{{315, 622, 2459, 3616}},
		/* Speaker 25 */
		{{360, 417,  981, 3610}},
		{{354, 697, 1017, 3077}},
		{{363, 741, 1302, 3381}},
		{{340, 1028, 1611, 2973}},
		{{337, 1122, 1930, 3068}},
		{{310, 608, 2105, 3177}},
		{{323, 628, 1929, 3197}},
		{{378, 384, 2255, 2992}},
		{{363, 363, 3235, 4069}},
		{{338, 527, 2698, 3458}},
		{{308, 643, 2603, 3382}},
		{{326, 712, 2351, 3498}},
		/* Speaker 26 */
		{{375, 380,  798, 3316}},
		{{371, 456,  759, 3321}},
		{{324, 656, 1118, 3296}},
		{{340, 1031, 1690, 3163}},
		{{376, 1154, 1920, 3519}},
		{{362, 604, 2162, 3603}},
		{{342, 679, 2035, 3683}},
		{{371, 374, 2042, 3206}},
		{{349, 361, 3082, 3782}},
		{{357, 499, 2899, 3934}},
		{{324, 646, 2820, 3970}},
		{{353, 706, 2785, 3942}},
		/* Speaker 27 */
		{{312, 339,  850, 4356}},
		{{319, 619,  919, 3570}},
		{{288, 576, 1156, 3681}},
		{{301, 865, 1567, 3326}},
		{{296, 914, 2090, 3113}},
		{{297, 589, 2150, 3208}},
		{{287, 580, 2146, 3340}},
		{{303, 330, 2431, 3032}},
		{{291, 313, 3090, 4039}},
		{{305, 539, 3015, 3905}},
		{{291, 577, 2589, 2778}},
		{{307, 616, 2448, 3856}},
		/* Speaker 28 */
		{{353, 354, 1085, 3082}},
		{{343, 617,  990, 3027}},
		{{281, 822, 1180, 2931}},
		{{324, 993, 1424, 2941}},
		{{297, 1192, 1792, 2922}},
		{{314, 444, 2170, 3044}},
		{{275, 608, 1817, 2909}},
		{{355, 358, 2138, 3160}},
		{{321, 323, 3169, 3625}},
		{{315, 499, 2831, 3465}},
		{{303, 608, 2707, 3369}},
		{{313, 718, 2479, 3627}},
		/* Speaker 29 */
		{{361, 363, 1050, 3239}},
		{{365, 666, 1094, 3269}},
		{{327, 830, 1289, 3145}},
		{{325, 963, 1293, 3046}},
		{{356, 1163, 1707, 3188}},
		{{377, 580, 2191, 3234}},
		{{336, 673, 2339, 3167}},
		{{342, 345, 2344, 3103}},
		{{341, 346, 2729, 3370}},
		{{326, 541, 2568, 3297}},
		{{331, 677, 2561, 3320}},
		{{354, 897, 2463, 3338}},
		/* Speaker 30 */
		{{332, 352,  979, 2988}},
		{{286, 596,  973, 2690}},
		{{304, 753, 1174, 3005}},
		{{290, 884, 1432, 2806}},
		{{347, 1060, 2079, 2706}},
		{{303, 592, 2069, 2889}},
		{{299, 644, 1775, 3005}},
		{{296, 329, 2082, 2791}},
		{{315, 326, 2807, 3612}},
		{{298, 561, 2612, 3330}},
		{{281, 569, 2415, 3239}},
		{{280, 709, 2460, 3314}}
	};

	try {
		autoTable me = Table_create (nrows, ncols);

		for (integer i = 1; i <= nrows; i ++) {
			const TableRow row = my rows.at [i];
			const integer speaker_id = (i - 1) / 12 + 1;	// 1 - 30
			const integer vowel_id = (i - 1) % 12 + 1;	// 1 - 12
			const integer index_in_data = (speaker_id - 1) * 12 + order [vowel_id] - 1;
			integer speaker_type, speaker_sex;

			if (speaker_id <= 10) {   // 10 men
				speaker_type = 0;
				speaker_sex = 0;
			} else if (speaker_id <= 20) {   // 10 women
				speaker_type = 1;
				speaker_sex = 1;
			} else {   // 10 children
				speaker_type = 2;
				speaker_sex = 0;   // which children were m/f
			}

			row -> cells [1]. string = Melder_dup (type [speaker_type]);
			row -> cells [2]. string = Melder_dup (sex [speaker_sex]);
			row -> cells [3]. string = Melder_dup (Melder_integer (speaker_id));
			row -> cells [4]. string = Melder_dup (vowel [vowel_id]);
			row -> cells [5]. string = Melder_dup (ipa [vowel_id]);

			for (integer j = 0; j <= 3; j ++)
				row -> cells [j + 6]. string = Melder_dup (Melder_integer (weeninkdata [index_in_data]. f [j]));
		}
		for (integer j = 1; j <= ncols; j ++) {
			Table_renameColumn_e (me.get(), j, columnLabels [j - 1]);
			my columnHeaders [j]. numericized = false;
		}
		return me;
	} catch (MelderError) {
		Melder_throw (U"Table not created from Weenink data.");
	}
}

/*
	P.A. Keating & C. Esposito (2006): "Linguistic voice quality." UCLA Working Papers in Phonetics 105: 85-91.
*/
autoTable Table_create_keatingEsposito2006 () {
	try {
		const conststring32 columnNames [] = { U"Language", U"Modal", U"Breathy" };
		autoTable me = Table_createWithColumnNames (10, ARRAY_TO_STRVEC (columnNames));
		Table_setStringValue (me.get(), 1, 1, U"Chong");
		Table_setNumericValue (me.get(), 1, 2, -1.5);
		Table_setNumericValue (me.get(), 1, 3, 5);
		Table_setStringValue (me.get(), 2, 1, U"Fuzhou");
		Table_setNumericValue (me.get(), 2, 2, 2);
		Table_setNumericValue (me.get(), 2, 3, 10);
		Table_setStringValue (me.get(), 3, 1, U"Green Hmong");
		Table_setNumericValue (me.get(), 3, 2, 3);
		Table_setNumericValue (me.get(), 3, 3, 12);
		Table_setStringValue (me.get(), 4, 1, U"White Hmong");
		Table_setNumericValue (me.get(), 4, 2, 2);
		Table_setNumericValue (me.get(), 4, 3, 11);
		Table_setStringValue (me.get(), 5, 1, U"Mon");
		Table_setNumericValue (me.get(), 5, 2, -1.5);
		Table_setNumericValue (me.get(), 5, 3, 0);
		Table_setStringValue (me.get(), 6, 1, U"SADV Zapotec");
		Table_setNumericValue (me.get(), 6, 2, -6);
		Table_setNumericValue (me.get(), 6, 3, -4);
		Table_setStringValue (me.get(), 7, 1, U"SLQ Zapotec");
		Table_setNumericValue (me.get(), 7, 2, 3.5);
		Table_setNumericValue (me.get(), 7, 3, 14);
		Table_setStringValue (me.get(), 8, 1, U"Tlacolula Zapotec");
		Table_setNumericValue (me.get(), 8, 2, 3);
		Table_setNumericValue (me.get(), 8, 3, 13);
		Table_setStringValue (me.get(), 9, 1, U"Tamang");
		Table_setNumericValue (me.get(), 9, 2, 1);
		Table_setNumericValue (me.get(), 9, 3, 1);
		Table_setStringValue (me.get(), 10, 1, U"!Xoo");
		Table_setNumericValue (me.get(), 10, 2, 1);
		Table_setNumericValue (me.get(), 10, 3, 14);
		return me;
	} catch (MelderError) {
		Melder_throw (U"Keating-Esposito table not created.");
	}
}

/*
	W.F. Ganong III (1980): "Phonetic categorization in auditory word perception." Journal of Experimental Psychology: 
		Human Perception and Performance 6: 110-125.
*/
autoTable Table_create_ganong1980 () {
	try {
		const conststring32 columnNames [] = { U"VOT", U"dash-tash", U"dask-task" };
		autoTable me = Table_createWithColumnNames (6, ARRAY_TO_STRVEC (columnNames));
		Table_setNumericValue (me.get(), 1, 1, -17.5);
		Table_setNumericValue (me.get(), 1, 2, 0.98);
		Table_setNumericValue (me.get(), 1, 3, 0.92);
		Table_setNumericValue (me.get(), 2, 1, -7.5);
		Table_setNumericValue (me.get(), 2, 2, 0.95);
		Table_setNumericValue (me.get(), 2, 3, 0.83);
		Table_setNumericValue (me.get(), 3, 1, -2.5);
		Table_setNumericValue (me.get(), 3, 2, 0.71);
		Table_setNumericValue (me.get(), 3, 3, 0.33);
		Table_setNumericValue (me.get(), 4, 1, 2.5);
		Table_setNumericValue (me.get(), 4, 2, 0.29);
		Table_setNumericValue (me.get(), 4, 3, 0.10);
		Table_setNumericValue (me.get(), 5, 1, 7.5);
		Table_setNumericValue (me.get(), 5, 2, 0.12);
		Table_setNumericValue (me.get(), 5, 3, 0.02);
		Table_setNumericValue (me.get(), 6, 1, 17.5);
		Table_setNumericValue (me.get(), 6, 2, 0.10);
		Table_setNumericValue (me.get(), 6, 3, 0.02);
		return me;
	} catch (MelderError) {
		Melder_throw (U"Ganong table not created.");
	}
}

autoTable Table_create_hillenbrandEtAl1995 () {
	const conststring32 columnLabels [] = {
		U"Type", U"Sex", U"Speaker", U"Vowel", U"IPA", U"f0", U"dur", U"F1", U"F2", U"F3", 
		U"F1_10", U"F2_10", U"F3_10", U"F1_20", U"F2_20", U"F3_20", U"F1_30", U"F2_30", U"F3_30", U"F1_40",
		U"F2_40", U"F3_40",	U"F1_50", U"F2_50", U"F3_50", U"F1_60", U"F2_60", U"F3_60", U"F1_70", U"F2_70",
		U"F3_70", U"F1_80", U"F2_80", U"F3_80", U"perc", U"iy", U"ih", U"ei", U"eh", U"ae", 
		U"ah", U"aw", U"oa", U"oo", U"uw", U"uh", U"er"};

	struct structHillenbrandDatum {
		char name [6];
		short data [42];	
	} hillenbranddata [] = {
		{{"b01ae"}, {257, 238, 630, 2423, 3166, 625, 2388, 3174, 651, 2413, 3115, 675, 2463, 3011, 687, 2391, 2926, 
			683, 2295, 2888, 696, 2187, 2888, 793, 2074, 2866, 806, 2049, 2961, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"b01ah"}, {212, 241, 831, 1676, 2602, 820, 1688, 2511, 845, 1684, 2583, 845, 1681, 2606, 858, 1690, 2600, 
			863, 1696, 2576, 875, 1745, 2569, 845, 1858, 2657, 807, 1980, 2893, 95, 0, 0, 0, 0, 0, 19, 0, 0, 1, 0, 0, 0}},
		{{"b01aw"}, {242, 247, 725, 1384, 2642, 722, 1411, 2669, 725, 1384, 2642, 733, 1384, 2662, 731, 1385, 2681, 
			748, 1388, 2682, 758, 1458, 2690, 772, 1579, 2739, 805, 1699, 2777, 10, 0, 0, 0, 0, 0, 9, 2, 0, 0, 0, 9, 0}},
		{{"b01eh"}, {184, 214, 713, 2095, 3129, 611, 2121, 3242, 643, 2071, 3078, 672, 2079, 3120, 700, 2109, 3162, 
			703, 2079, 3044, 682, 2133, 3092, 687, 2124, 3119, 672, 2140, 3128, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b01ei"}, {222, 230, 534, 2690, 3335, 582, 2454, 3223, 621, 2573, 3260, 579, 2632, 3287, 534, 2690, 3335, 
			498, 2797, 3299, 480, 2883, 3294, 455, 2956, 0, 440, 3012, 0, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b01er"}, {227, 240, 608, 1733, 2159, 567, 1599, 2255, 566, 1664, 2180, 579, 1709, 2168, 635, 1746, 2103, 
			644, 1678, 2046, 634, 1672, 2054, 633, 1778, 2078, 623, 1781, 2157, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b01ih"}, {197, 263, 551, 2393, 3324, 554, 2451, 3407, 543, 2385, 3242, 532, 2388, 3252, 533, 2384, 3299, 
			535, 2416, 3359, 558, 2428, 3387, 579, 2384, 3408, 622, 2412, 3428, 95, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b01iy"}, {237, 277, 554, 3022, 3541, 519, 3036, 3451, 548, 2981, 3459, 555, 3033, 3578, 556, 3038, 3470, 
			549, 2992, 3362, 518, 3039, 3330, 500, 3113, 3398, 468, 3024, 3444, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b01oa"}, {267, 250, 693, 1235, 2850, 684, 1233, 2850, 695, 1227, 2935, 692, 1179, 2962, 679, 1086, 2982, 
			633, 1066, 3018, 566, 1051, 3045, 548, 1013, 3052, 549, 993, 3083, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b01oo"}, {184, 247, 553, 1495, 2868, 537, 1435, 2926, 552, 1489, 2910, 552, 1500, 2928, 560, 1489, 2947, 
			627, 1484, 2924, 684, 1698, 2940, 682, 1691, 2961, 674, 1873, 3005, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"b01uh"}, {205, 237, 632, 1573, 2913, 634, 1538, 3039, 633, 1590, 2956, 639, 1629, 2747, 646, 1675, 2828, 
			657, 1636, 2691, 623, 1692, 2618, 634, 1797, 2838, 627, 1925, 2812, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"b01uw"}, {253, 246, 502, 1540, 3176, 506, 1607, 3191, 505, 1574, 3192, 504, 1554, 3183, 502, 1528, 3172, 
			501, 1495, 3177, 494, 1487, 3186, 489, 1473, 3126, 487, 1446, 3138, 90, 0, 0, 0, 0, 0, 0, 0, 0, 2, 18, 0, 0}},
		{{"b02ae"}, {359, 286, 829, 2495, 3218, 802, 2392, 3625, 778, 2461, 3424, 793, 2498, 3344, 825, 2503, 3227, 
			835, 2491, 3212, 865, 2339, 3250, 965, 2244, 3277, 897, 2133, 3384, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"b02ah"}, {328, 276, 1020, 1555, 2742, 990, 1611, 2888, 1023, 1555, 2809, 1016, 1547, 2724, 1024, 1504, 2668, 
			1003, 1501, 2665, 1005, 1549, 2676, 1003, 1649, 2686, 980, 1744, 2818, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"b02aw"}, {327, 263, 802, 1287, 2729, 754, 1142, 2586, 763, 1214, 2704, 780, 1243, 2691, 803, 1295, 2749, 
			797, 1315, 2779, 803, 1499, 2877, 805, 1607, 2974, 808, 1644, 3038, 85, 0, 0, 0, 0, 0, 3, 17, 0, 0, 0, 0, 0}},
		{{"b02eh"}, {195, 284, 674, 2196, 3047, 632, 2340, 3164, 657, 2178, 3082, 680, 2232, 3089, 675, 2254, 3046, 
			698, 2246, 3065, 675, 2257, 3118, 635, 2258, 3181, 621, 2250, 3214, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b02ei"}, {324, 277, 556, 2482, 3428, 558, 2482, 3459, 552, 2513, 3378, 549, 2550, 3276, 543, 2684, 3400, 
			553, 2704, 3518, 549, 2731, 3329, 540, 2772, 3283, 506, 2740, 3367, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b02er"}, {395, 283, 560, 1687, 1990, 557, 1692, 1985, 560, 1727, 2002, 564, 1708, 2000, 563, 1694, 1973, 
			560, 1690, 1990, 560, 1688, 1990, 558, 1687, 1957, 559, 1700, 1966, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b02ih"}, {239, 289, 563, 2608, 3476, 565, 2658, 3413, 563, 2608, 3476, 564, 2587, 3317, 566, 2569, 3193, 
			565, 2534, 3139, 569, 2287, 3117, 578, 2318, 3317, 583, 2318, 3312, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b02iy"}, {317, 287, 419, 3025, 3673, 501, 2996, 3672, 484, 2977, 3640, 419, 3025, 3673, 452, 3159, 3729, 
			482, 3155, 3751, 474, 3200, 3743, 477, 3262, 3757, 436, 3271, 3833, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b02oa"}, {336, 258, 557, 1106, 2808, 635, 1140, 2789, 606, 1167, 2786, 557, 1106, 2808, 516, 1085, 2781, 
			522, 1043, 2886, 552, 1005, 2968, 558, 914, 3046, 560, 934, 3115, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b02oo"}, {253, 292, 570, 1148, 2947, 582, 1161, 2953, 572, 1121, 2935, 567, 1270, 2813, 568, 1321, 2853, 
			567, 1380, 2843, 566, 1421, 2809, 564, 1674, 2773, 563, 1685, 2818, 95, 1, 0, 0, 0, 0, 0, 0, 0, 19, 0, 0, 0}},
		{{"b02uh"}, {161, 269, 731, 1550, 3363, 731, 1550, 3363, 762, 1612, 3319, 748, 1621, 3275, 742, 1666, 3259, 
			721, 1745, 3247, 682, 1849, 3288, 644, 1890, 3444, 596, 1936, 3484, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"b02uw"}, {327, 287, 559, 1312, 2870, 568, 1452, 2890, 562, 1354, 2875, 555, 1370, 2829, 556, 1372, 2817, 
			550, 1318, 2796, 555, 1242, 2818, 549, 1132, 2777, 546, 1057, 2766, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"b03ae"}, {335, 214, 631, 2801, 3508, 631, 2801, 3508, 602, 2760, 3453, 573, 2862, 3566, 580, 2774, 3558, 
			589, 2686, 3389, 775, 2472, 3240, 873, 2297, 3263, 946, 2272, 3284, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"b03ah"}, {298, 214, 982, 1823, 2865, 957, 1798, 2752, 997, 1834, 2813, 1021, 1849, 2865, 998, 1861, 2803, 
			984, 1850, 2775, 1006, 1843, 2705, 998, 1806, 2726, 982, 1848, 2773, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"b03aw"}, {334, 207, 805, 1299, 2582, 811, 1323, 2503, 808, 1335, 2501, 809, 1349, 2558, 800, 1286, 2607, 
			818, 1317, 2600, 816, 1306, 2608, 803, 1348, 2618, 806, 1387, 2613, 20, 0, 0, 0, 0, 0, 16, 4, 0, 0, 0, 0, 0}},
		{{"b03eh"}, {260, 191, 735, 2315, 3286, 641, 2369, 3472, 721, 2404, 3435, 753, 2348, 3363, 747, 2329, 3306, 
			743, 2310, 3366, 741, 2304, 3353, 727, 2287, 3339, 727, 2304, 3354, 90, 0, 0, 0, 18, 2, 0, 0, 0, 0, 0, 0, 0}},
		{{"b03ei"}, {286, 201, 608, 2836, 3599, 619, 2806, 3457, 573, 2816, 3563, 564, 2921, 3682, 555, 2942, 3561, 
			502, 2979, 3535, 475, 3004, 3421, 451, 3066, 3486, 436, 3238, 3534, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b03er"}, {280, 199, 622, 1837, 2114, 679, 1895, 2172, 630, 1834, 2099, 622, 1841, 2059, 622, 1833, 2118, 
			612, 1838, 2102, 614, 1839, 2112, 611, 1871, 2110, 609, 1899, 2124, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b03ih"}, {284, 228, 571, 2689, 3572, 499, 2798, 3597, 535, 2723, 3585, 563, 2749, 3529, 566, 2679, 3540, 
			576, 2684, 3592, 566, 2695, 3566, 587, 2695, 3585, 622, 2646, 3594, 90, 0, 18, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b03iy"}, {277, 219, 441, 3081, 3975, 441, 3081, 3975, 439, 3057, 4071, 437, 3061, 4187, 437, 3066, 4122, 
			441, 3065, 4061, 438, 3047, 3950, 441, 3077, 3894, 446, 3103, 3789, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b03oa"}, {310, 211, 624, 1456, 2933, 615, 1618, 2843, 630, 1573, 2850, 616, 1498, 2887, 626, 1455, 2932, 
			625, 1424, 2949, 624, 1293, 2912, 621, 1270, 2877, 593, 1290, 2811, 90, 0, 0, 0, 0, 0, 0, 0, 18, 1, 0, 0, 1}},
		{{"b03oo"}, {247, 200, 574, 1808, 2985, 636, 1944, 2964, 622, 1931, 3019, 614, 1837, 2998, 574, 1808, 2985, 
			574, 1807, 2955, 575, 1868, 2947, 577, 2013, 2963, 577, 2203, 3198, 85, 0, 0, 0, 1, 0, 0, 0, 0, 17, 0, 0, 2}},
		{{"b03uh"}, {226, 210, 823, 1655, 2926, 846, 1664, 2861, 819, 1661, 2915, 825, 1653, 2921, 813, 1666, 2933, 
			812, 1688, 2952, 817, 1794, 2999, 815, 2040, 3073, 674, 2197, 3213, 55, 0, 0, 0, 0, 0, 9, 0, 0, 0, 0, 11, 0}},
		{{"b03uw"}, {304, 194, 506, 2378, 2991, 551, 2397, 3114, 546, 2372, 3039, 506, 2378, 2991, 502, 2347, 2984, 
			490, 2314, 2991, 460, 2237, 2959, 445, 2120, 2933, 441, 2032, 2923, 40, 0, 9, 0, 1, 0, 0, 0, 0, 0, 8, 0, 2}},
		{{"b04ae"}, {398, 239, 712, 2608, 3247, 729, 2604, 3239, 712, 2608, 3247, 695, 2577, 3234, 690, 2516, 3277, 
			690, 2416, 3270, 802, 2294, 3203, 867, 2139, 2999, 809, 2149, 2907, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"b04ah"}, {357, 250, 932, 1874, 2994, 923, 1870, 2935, 936, 1889, 3054, 912, 1895, 2940, 945, 1905, 2990, 
			935, 1897, 3027, 902, 1887, 2966, 904, 1908, 2950, 871, 1989, 2945, 95, 0, 0, 0, 0, 0, 19, 0, 0, 0, 0, 1, 0}},
		{{"b04aw"}, {304, 244, 817, 1329, 2965, 815, 1284, 2985, 825, 1314, 3001, 811, 1337, 2945, 814, 1364, 2909, 
			856, 1409, 2978, 872, 1565, 3005, 859, 1651, 3011, 852, 1759, 3013, 85, 0, 0, 0, 0, 0, 2, 17, 0, 0, 0, 1, 0}},
		{{"b04eh"}, {364, 245, 735, 2448, 3359, 717, 2448, 3349, 733, 2457, 3353, 738, 2437, 3371, 704, 2384, 3348, 
			692, 2377, 3348, 689, 2261, 3258, 696, 2200, 3256, 731, 2183, 3340, 55, 0, 0, 0, 11, 9, 0, 0, 0, 0, 0, 0, 0}},
		{{"b04ei"}, {323, 252, 574, 2490, 3189, 606, 2437, 3176, 631, 2455, 3193, 558, 2498, 3185, 537, 2500, 3151, 
			517, 2505, 2961, 510, 2573, 3069, 548, 2558, 3111, 507, 2568, 3057, 80, 0, 0, 16, 3, 1, 0, 0, 0, 0, 0, 0, 0}},
		{{"b04er"}, {386, 230, 580, 1990, 2245, 624, 2042, 2297, 582, 1980, 2391, 603, 1954, 2345, 586, 1964, 2266, 
			580, 1990, 2245, 503, 2002, 2236, 503, 2014, 2197, 501, 2026, 2217, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b04ih"}, {268, 249, 513, 2447, 3454, 512, 2405, 3408, 520, 2447, 3450, 519, 2451, 3525, 542, 2443, 3432, 
			569, 2394, 3439, 644, 2277, 3405, 679, 2235, 3377, 686, 2188, 3375, 95, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b04iy"}, {318, 260, 509, 2873, 3475, 507, 2879, 3482, 506, 2860, 3440, 503, 2843, 3474, 501, 2818, 3315, 
			499, 2972, 3362, 499, 2978, 3316, 498, 2976, 3270, 497, 2934, 3224, 95, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b04oa"}, {322, 250, 685, 1128, 3181, 713, 1283, 3181, 706, 1180, 3181, 651, 1077, 3181, 632, 973, 3181, 
			559, 915, 3175, 513, 990, 3114, 521, 1002, 3127, 526, 1020, 3134, 95, 0, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0}},
		{{"b04oo"}, {343, 251, 558, 1742, 3271, 563, 1777, 3312, 559, 1748, 3259, 560, 1740, 3224, 580, 1705, 3220, 
			606, 1702, 3159, 648, 1785, 3144, 684, 1936, 3098, 696, 2113, 3146, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"b04uh"}, {293, 241, 756, 1672, 3183, 754, 1684, 3213, 751, 1666, 3155, 748, 1678, 3153, 743, 1632, 3189, 
			763, 1721, 3145, 756, 1738, 3145, 758, 1816, 3121, 735, 1916, 3115, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"b04uw"}, {292, 258, 503, 1104, 3346, 553, 1193, 3312, 513, 1166, 3392, 509, 1116, 3380, 505, 1102, 3370, 
			504, 1155, 3336, 506, 1148, 3356, 507, 1161, 3364, 508, 1239, 3368, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"b05ae"}, {267, 200, 748, 2589, 3042, 728, 2601, 3047, 752, 2562, 3033, 767, 2594, 3053, 766, 2577, 3068, 
			815, 2498, 3123, 872, 2320, 3141, 880, 2193, 3230, 900, 2130, 3337, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"b05ah"}, {270, 227, 1052, 1659, 3433, 1052, 1659, 3433, 1084, 1677, 3347, 1068, 1616, 3648, 1060, 1601, 3638, 
			1044, 1679, 3327, 1028, 1737, 3497, 1042, 1795, 3582, 885, 2006, 3507, 95, 0, 0, 0, 0, 0, 19, 0, 0, 0, 0, 1, 0}},
		{{"b05aw"}, {236, 199, 780, 1236, 3360, 780, 1236, 3360, 777, 1265, 3328, 788, 1240, 3356, 803, 1319, 3358, 
			810, 1355, 3377, 817, 1423, 3290, 825, 1487, 3271, 858, 1670, 3192, 85, 0, 0, 0, 0, 0, 0, 17, 0, 0, 0, 3, 0}},
		{{"b05eh"}, {207, 219, 787, 2176, 3307, 661, 2224, 3396, 765, 2182, 3298, 787, 2176, 3307, 808, 2179, 3306, 
			803, 2180, 3287, 797, 2173, 3301, 742, 2197, 3325, 705, 2222, 3350, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b05ei"}, {247, 215, 611, 2761, 3269, 517, 2621, 3168, 622, 2693, 3248, 611, 2761, 3269, 507, 2820, 3282, 
			471, 2879, 3242, 448, 2954, 3242, 442, 2988, 3157, 445, 3016, 5272, 95, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b05er"}, {228, 224, 573, 1562, 2227, 568, 1558, 2231, 574, 1565, 2220, 572, 1566, 2217, 561, 1575, 2217, 
			564, 1587, 2193, 561, 1639, 2210, 536, 1733, 2231, 482, 1796, 2199, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b05ih"}, {209, 226, 479, 2449, 3200, 480, 2511, 3208, 477, 2496, 3183, 493, 2438, 3217, 497, 2397, 3184, 
			502, 2299, 3196, 506, 2272, 3193, 507, 2268, 3203, 496, 2306, 3276, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b05iy"}, {242, 244, 475, 3203, 0, 465, 3228, 0, 481, 3166, 0, 470, 3192, 0, 473, 3185, 0, 
			483, 3145, 0, 464, 3179, 0, 458, 3173, 0, 455, 3129, 3374, 95, 19, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b05oa"}, {247, 197, 589, 1050, 3083, 628, 1045, 3132, 622, 1042, 3077, 579, 1008, 3103, 579, 972, 3109, 
			592, 951, 3057, 565, 796, 3103, 483, 930, 3062, 460, 1062, 3101, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b05oo"}, {197, 218, 543, 1473, 3166, 459, 1319, 3162, 493, 1440, 3027, 543, 1473, 3166, 553, 1536, 3133, 
			573, 1583, 3117, 605, 1612, 3138, 611, 1816, 3168, 598, 1912, 3103, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"b05uh"}, {150, 217, 637, 1424, 3328, 669, 1419, 3495, 629, 1490, 3343, 635, 1532, 3413, 640, 1521, 3387, 
			622, 1511, 3368, 635, 1541, 3231, 632, 1569, 3190, 621, 1714, 3182, 80, 0, 0, 0, 0, 0, 0, 0, 0, 4, 0, 16, 0}},
		{{"b05uw"}, {260, 240, 481, 1226, 3131, 489, 1292, 3134, 487, 1284, 3182, 488, 1240, 3183, 478, 1219, 3132, 
			479, 1197, 3113, 469, 1163, 3102, 469, 1126, 3110, 474, 1178, 3115, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"b07ae"}, {323, 262, 769, 2203, 3126, 769, 2203, 3126, 760, 2169, 3144, 813, 2225, 3161, 849, 2263, 3184, 
			862, 2154, 3193, 876, 2089, 3164, 879, 2065, 3141, 873, 2052, 3149, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"b07ah"}, {294, 308, 881, 1543, 3076, 774, 1570, 3054, 810, 1605, 3061, 845, 1522, 3134, 859, 1493, 3061, 
			882, 1542, 3072, 880, 1543, 3069, 881, 1590, 3059, 877, 1837, 3031, 80, 0, 0, 0, 0, 0, 16, 1, 0, 0, 0, 3, 0}},
		{{"b07aw"}, {378, 301, 891, 1383, 3039, 876, 1394, 3058, 896, 1383, 3039, 941, 1347, 3116, 953, 1364, 3142, 
			952, 1450, 3150, 943, 1548, 3116, 941, 1581, 3064, 933, 1675, 3061, 65, 0, 0, 0, 0, 0, 7, 13, 0, 0, 0, 0, 0}},
		{{"b07eh"}, {272, 297, 867, 2054, 3227, 820, 2256, 3284, 817, 2180, 3263, 864, 2038, 3229, 885, 2104, 3266, 
			905, 2138, 3184, 930, 2114, 3228, 923, 2042, 3242, 883, 2031, 3253, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b07ei"}, {361, 320, 629, 2663, 3132, 573, 2637, 3241, 568, 2634, 3138, 583, 2644, 3144, 621, 2734, 3170, 
			629, 2663, 3124, 627, 2676, 3137, 627, 2743, 3154, 623, 2863, 3436, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b07er"}, {375, 294, 612, 1645, 2306, 573, 1576, 2274, 592, 1551, 2260, 613, 1587, 2304, 613, 1623, 2309, 
			612, 1646, 2304, 611, 1666, 2298, 602, 1686, 2271, 596, 1711, 2240, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b07ih"}, {268, 296, 590, 2354, 3416, 553, 2412, 3475, 542, 2379, 3372, 563, 2314, 3375, 601, 2347, 3391, 
			613, 2351, 3348, 618, 2324, 3396, 616, 2328, 3410, 583, 2357, 3392, 95, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b07iy"}, {310, 307, 488, 3063, 4430, 522, 2967, 4320, 445, 3033, 4366, 401, 2974, 4451, 429, 3101, 4488, 
			489, 3074, 4447, 485, 3051, 4361, 488, 3019, 4275, 501, 2984, 4296, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b07oa"}, {311, 315, 641, 1243, 3443, 633, 1234, 3261, 640, 1225, 3326, 642, 1237, 3390, 643, 1243, 3454, 
			658, 1249, 3430, 651, 1259, 3373, 649, 1305, 3263, 655, 1412, 3207, 55, 0, 0, 0, 0, 0, 1, 1, 11, 4, 0, 0, 3}},
		{{"b07oo"}, {281, 305, 624, 1513, 3061, 571, 1424, 3063, 587, 1435, 3089, 605, 1448, 2993, 619, 1494, 3018, 
			622, 1503, 3054, 624, 1513, 3056, 624, 1510, 3047, 624, 1631, 3039, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 0, 1, 0}},
		{{"b07uh"}, {310, 289, 822, 1489, 3281, 815, 1508, 3296, 843, 1481, 3252, 843, 1489, 3263, 870, 1512, 3280, 
			870, 1547, 3301, 873, 1630, 3310, 862, 1796, 3269, 820, 1851, 3190, 90, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 18, 1}},
		{{"b07uw"}, {303, 330, 609, 1658, 2644, 563, 1576, 2762, 596, 1617, 2721, 609, 1658, 2644, 613, 1646, 2627, 
			621, 1626, 2622, 620, 1589, 2753, 610, 1535, 2696, 612, 1550, 2688, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"b08ae"}, {316, 216, 870, 2281, 3077, 765, 2252, 3214, 820, 2239, 3181, 864, 2356, 3055, 869, 2281, 3076, 
			869, 2267, 3103, 871, 2335, 3105, 869, 2329, 3078, 862, 2331, 3144, 70, 0, 0, 0, 6, 14, 0, 0, 0, 0, 0, 0, 0}},
		{{"b08ah"}, {340, 202, 945, 1604, 2991, 871, 1556, 3030, 969, 1609, 2978, 909, 1635, 3000, 972, 1679, 3029, 
			981, 1674, 3049, 972, 1821, 3044, 959, 1839, 3087, 810, 1954, 3218, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"b08aw"}, {353, 207, 610, 993, 2946, 831, 0, 2948, 590, 1016, 2826, 704, 0, 2933, 782, 0, 2934, 
			765, 0, 2890, 771, 0, 2935, 733, 1396, 2873, 733, 1650, 3064, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"b08eh"}, {135, 205, 605, 2445, 3265, 609, 2500, 3360, 609, 2476, 3270, 615, 2471, 3339, 621, 2500, 3364, 
			616, 2500, 3294, 606, 2451, 3306, 604, 2462, 3300, 593, 2458, 3279, 90, 0, 2, 0, 18, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b08ei"}, {285, 223, 606, 2396, 3015, 600, 2422, 3007, 571, 2475, 2993, 555, 2526, 2983, 546, 2565, 3016, 
			509, 2606, 3043, 466, 2641, 3046, 436, 2684, 3042, 429, 2587, 2935, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b08er"}, {261, 208, 477, 1809, 0, 575, 1928, 0, 565, 1870, 0, 553, 1821, 0, 504, 1741, 0, 
			493, 1776, 0, 443, 1755, 0, 444, 1804, 0, 436, 1856, 0, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b08ih"}, {205, 215, 447, 2784, 3287, 447, 2784, 3287, 433, 2808, 3319, 434, 2731, 3295, 436, 2683, 3354, 
			440, 2627, 3303, 479, 2562, 3290, 500, 2537, 3317, 523, 2505, 3334, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b08iy"}, {241, 220, 434, 3127, 3566, 435, 3145, 3577, 434, 3130, 3542, 434, 3118, 3498, 434, 3127, 3566, 
			433, 3121, 3596, 432, 3204, 3599, 431, 3203, 3604, 432, 3184, 3600, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b08oa"}, {341, 248, 554, 896, 2870, 635, 964, 2934, 670, 842, 2868, 559, 798, 2775, 554, 896, 2870, 
			524, 882, 2833, 499, 899, 2841, 474, 840, 2914, 436, 836, 2979, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b08oo"}, {200, 218, 469, 1097, 2801, 515, 1113, 2905, 490, 1167, 2937, 501, 1260, 2796, 517, 1264, 2901, 
			567, 1320, 2997, 616, 1523, 2961, 619, 1795, 2914, 588, 1916, 3057, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"b08uh"}, {242, 201, 667, 1442, 2882, 721, 1418, 2963, 667, 1442, 2882, 753, 1519, 2874, 762, 1545, 2938, 
			645, 1608, 2960, 580, 1649, 3105, 524, 1796, 3091, 483, 1892, 3176, 85, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 17, 0}},
		{{"b08uw"}, {262, 228, 440, 1124, 3117, 441, 1066, 3024, 441, 1111, 3068, 440, 1128, 3117, 439, 1129, 3106, 
			439, 1130, 3141, 439, 1119, 3159, 440, 1124, 3145, 440, 1291, 3109, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"b09ae"}, {245, 220, 709, 2565, 3526, 626, 2545, 3504, 709, 2565, 3526, 663, 2612, 3517, 676, 2601, 3567, 
			683, 2476, 3326, 724, 2362, 3061, 808, 2296, 3129, 826, 2225, 3022, 95, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0}},
		{{"b09ah"}, {285, 230, 987, 1610, 3010, 954, 1607, 3069, 1011, 1589, 3055, 1005, 1586, 3061, 1007, 1617, 2963, 
			1012, 1646, 2900, 986, 1670, 2902, 960, 1713, 2870, 931, 1816, 2950, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"b09aw"}, {313, 238, 845, 1111, 3049, 880, 1185, 3103, 846, 1132, 3026, 827, 1114, 3053, 823, 1131, 3037, 
			834, 1121, 3012, 840, 1164, 2966, 844, 1213, 2926, 826, 1412, 2935, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"b09eh"}, {193, 238, 699, 2202, 3393, 726, 2302, 3311, 706, 2217, 3325, 698, 2208, 3375, 700, 2193, 3411, 
			700, 2188, 3372, 695, 2186, 3385, 699, 2188, 3428, 701, 2221, 3513, 95, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0}},
		{{"b09ei"}, {258, 235, 555, 2377, 3123, 555, 2377, 3123, 565, 2403, 3134, 535, 2475, 3114, 498, 2517, 3185, 
			492, 2611, 3193, 486, 2649, 3232, 479, 2627, 3275, 484, 2645, 3319, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b09er"}, {262, 241, 505, 1691, 2074, 584, 1716, 2029, 562, 1709, 2026, 514, 1688, 2070, 503, 1701, 2106, 
			508, 1725, 2088, 517, 1774, 2069, 520, 1830, 2065, 507, 1873, 2108, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b09ih"}, {232, 239, 536, 2319, 3351, 501, 2535, 3401, 507, 2334, 3405, 511, 2334, 3404, 510, 2322, 3369, 
			530, 2322, 3356, 539, 2310, 3343, 546, 2308, 3340, 523, 2294, 3367, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b09iy"}, {296, 251, 501, 2760, 3262, 501, 2735, 3283, 498, 2767, 3201, 498, 2821, 3228, 500, 2873, 3255, 
			503, 2963, 3270, 497, 3014, 3285, 512, 3053, 3297, 512, 3026, 3312, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b09oa"}, {210, 237, 516, 1181, 3079, 522, 1352, 3150, 526, 1176, 3083, 516, 1181, 3079, 510, 1173, 3102, 
			509, 1172, 3111, 502, 1172, 3130, 497, 1215, 3144, 496, 1239, 3126, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b09oo"}, {222, 237, 615, 1345, 3346, 533, 1239, 3352, 568, 1234, 3368, 612, 1325, 3359, 631, 1378, 3323, 
			647, 1428, 3265, 678, 1505, 3229, 683, 1642, 3217, 677, 1816, 3282, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"b09uh"}, {228, 233, 695, 1423, 3210, 704, 1413, 3250, 695, 1423, 3240, 694, 1409, 3213, 694, 1395, 3221, 
			711, 1499, 3194, 725, 1556, 3143, 687, 1617, 3134, 683, 1826, 3178, 90, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 18, 0}},
		{{"b09uw"}, {211, 255, 503, 1119, 3100, 501, 1296, 3212, 502, 1264, 3142, 502, 1151, 3152, 503, 1120, 3127, 
			503, 1119, 3100, 502, 1118, 3109, 499, 1242, 3092, 498, 1277, 3045, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"b10ae"}, {396, 205, 634, 2555, 3121, 635, 2560, 3230, 642, 2559, 3126, 633, 2514, 3018, 680, 2508, 3050, 
			710, 2498, 3106, 778, 2431, 2988, 816, 2322, 2983, 832, 2305, 2951, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"b10ah"}, {413, 209, 1123, 1640, 2684, 1179, 1650, 2763, 1147, 1662, 2728, 1128, 1635, 2673, 1152, 1649, 2706, 
			1131, 1672, 2637, 1076, 1649, 2622, 1063, 1672, 2550, 1016, 1765, 2466, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"b10aw"}, {316, 208, 695, 1288, 3014, 653, 1181, 2843, 671, 1247, 2889, 691, 1288, 3021, 704, 1281, 2970, 
			707, 1286, 2873, 738, 1340, 2834, 768, 1456, 2723, 751, 1615, 2663, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"b10eh"}, {234, 218, 646, 2435, 3262, 626, 2450, 3275, 628, 2488, 3252, 631, 2484, 3343, 639, 2435, 3297, 
			656, 2431, 3230, 665, 2426, 3179, 671, 2431, 3181, 676, 2455, 3190, 90, 0, 0, 0, 18, 2, 0, 0, 0, 0, 0, 0, 0}},
		{{"b10ei"}, {314, 197, 580, 2754, 3405, 591, 2727, 3440, 579, 2763, 3405, 580, 2763, 3436, 579, 2783, 3449, 
			574, 2783, 3401, 560, 2814, 3337, 524, 2821, 3265, 497, 2808, 3256, 95, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b10er"}, {326, 220, 620, 1823, 2188, 620, 1835, 2166, 620, 1823, 2188, 613, 1887, 2168, 579, 1920, 2169, 
			568, 1946, 2171, 577, 1929, 2173, 584, 1915, 2175, 589, 1865, 2177, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b10ih"}, {263, 209, 529, 2649, 3280, 513, 2684, 3305, 517, 2669, 3310, 519, 2666, 3295, 525, 2654, 3279, 
			535, 2642, 3258, 547, 2660, 3289, 553, 2651, 3302, 557, 2583, 3275, 80, 0, 16, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b10iy"}, {266, 222, 424, 3198, 3818, 399, 3168, 3780, 408, 3228, 3726, 427, 3243, 3827, 432, 3230, 3890, 
			429, 3215, 3867, 424, 3198, 3818, 434, 3224, 3770, 443, 3154, 3764, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b10oa"}, {316, 209, 623, 1241, 2770, 622, 1250, 2793, 623, 1241, 2770, 625, 1244, 2764, 625, 1251, 2801, 
			626, 1229, 2838, 626, 1197, 2866, 625, 1050, 2821, 609, 1069, 2822, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b10oo"}, {259, 214, 613, 1561, 2856, 610, 1557, 2938, 608, 1601, 2854, 602, 1556, 2893, 613, 1561, 2856, 
			617, 1579, 2843, 619, 1581, 2859, 621, 1664, 2887, 615, 1811, 2912, 80, 0, 0, 0, 0, 0, 0, 0, 0, 16, 1, 1, 2}},
		{{"b10uh"}, {248, 203, 612, 1432, 3046, 612, 1432, 3096, 622, 1430, 3021, 626, 1453, 3063, 630, 1489, 3035, 
			637, 1548, 2949, 643, 1629, 2916, 633, 1691, 2852, 628, 1929, 2880, 45, 0, 0, 0, 0, 0, 0, 1, 0, 10, 0, 9, 0}},
		{{"b10uw"}, {275, 215, 463, 1676, 2976, 458, 1668, 2983, 461, 1688, 2976, 460, 1632, 2992, 455, 1608, 3004, 
			447, 1520, 2992, 443, 1489, 2995, 440, 1461, 3009, 440, 1456, 3003, 90, 0, 0, 0, 0, 0, 0, 0, 0, 2, 18, 0, 0}},
		{{"b11ae"}, {298, 209, 630, 2509, 3112, 630, 2509, 3112, 627, 2513, 3098, 616, 2531, 3135, 632, 2507, 3127, 
			693, 2411, 3058, 750, 2282, 3000, 755, 2123, 2973, 743, 2066, 2988, 95, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0}},
		{{"b11ah"}, {282, 203, 832, 1602, 2551, 819, 1588, 2551, 849, 1604, 2585, 862, 1582, 2615, 866, 1562, 2629, 
			822, 1564, 2644, 792, 1627, 2638, 784, 1670, 2670, 761, 1734, 2723, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"b11aw"}, {331, 194, 761, 1310, 2801, 796, 1362, 2800, 765, 1322, 2825, 758, 1310, 2800, 759, 1331, 2785, 
			764, 1364, 2781, 775, 1466, 2764, 775, 1553, 2770, 761, 1638, 2825, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"b11eh"}, {282, 214, 794, 2066, 3160, 742, 2076, 3148, 751, 2054, 3130, 802, 2080, 3167, 809, 2099, 3143, 
			785, 2072, 3131, 764, 2077, 3138, 715, 2083, 3135, 685, 2095, 3114, 90, 0, 0, 0, 18, 1, 0, 0, 0, 0, 0, 1, 0}},
		{{"b11ei"}, {309, 210, 571, 2586, 3293, 598, 2610, 3282, 578, 2708, 3284, 560, 2729, 3264, 560, 2742, 3258, 
			557, 2754, 3273, 538, 2743, 3245, 514, 2732, 3234, 497, 2763, 3208, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b11er"}, {322, 211, 586, 1681, 2069, 603, 1684, 2015, 576, 1686, 2063, 564, 1645, 2009, 558, 1646, 1997, 
			563, 1634, 2000, 560, 1675, 2004, 556, 1672, 1991, 521, 1673, 2014, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b11ih"}, {227, 197, 522, 2387, 3297, 479, 2458, 3355, 495, 2469, 3228, 496, 2449, 3296, 500, 2408, 3303, 
			533, 2381, 3287, 558, 2363, 3256, 559, 2343, 3259, 558, 2335, 3279, 95, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b11iy"}, {254, 207, 429, 2984, 3348, 438, 2933, 3304, 436, 2985, 3383, 432, 2970, 3341, 429, 2984, 3348, 
			421, 3013, 3345, 393, 3002, 3341, 381, 3009, 3335, 380, 2901, 3329, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b11oa"}, {270, 198, 591, 1168, 2769, 637, 1256, 2749, 632, 1207, 2745, 625, 1175, 2744, 611, 1161, 2792, 
			582, 1168, 2769, 569, 1174, 2749, 564, 1173, 2699, 549, 1169, 2665, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b11oo"}, {271, 213, 593, 1507, 2997, 556, 1459, 2956, 567, 1495, 2993, 593, 1507, 2997, 619, 1509, 2993, 
			607, 1631, 3004, 578, 1721, 3001, 577, 1796, 2976, 580, 1931, 3053, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"b11uh"}, {266, 213, 699, 1454, 2765, 699, 1454, 2765, 699, 1478, 2743, 721, 1491, 2750, 693, 1607, 2806, 
			662, 1641, 2815, 622, 1751, 2849, 603, 1824, 2858, 567, 1927, 2982, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"b11uw"}, {246, 197, 430, 1448, 2896, 446, 1520, 2848, 441, 1511, 2847, 441, 1483, 2881, 434, 1451, 2882, 
			432, 1446, 2897, 431, 1445, 2893, 434, 1438, 2926, 434, 1527, 2930, 65, 0, 0, 0, 0, 0, 0, 0, 0, 7, 13, 0, 0}},
		{{"b12ae"}, {415, 252, 736, 2505, 3332, 729, 2544, 3261, 736, 2504, 3307, 739, 2502, 3312, 747, 2467, 3222, 
			771, 2326, 3178, 814, 2223, 3163, 818, 2227, 3183, 797, 2242, 3112, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"b12ah"}, {425, 258, 968, 1808, 3600, 976, 1863, 3611, 972, 1850, 3626, 975, 1839, 3641, 963, 1814, 3600, 
			976, 1858, 3660, 974, 1878, 3626, 983, 2049, 3713, 908, 2123, 3684, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"b12aw"}, {369, 251, 985, 1630, 2989, 993, 1626, 2948, 984, 1630, 2988, 967, 1615, 3053, 889, 1621, 3103, 
			817, 1649, 3128, 774, 1666, 3143, 727, 1776, 3027, 694, 1826, 3057, 25, 0, 0, 0, 0, 0, 7, 5, 0, 0, 0, 8, 0}},
		{{"b12eh"}, {363, 249, 742, 2355, 3514, 740, 2398, 3483, 742, 2394, 3487, 742, 2366, 3494, 743, 2351, 3573, 
			745, 2350, 3558, 745, 2355, 3590, 744, 2344, 3550, 737, 2313, 3541, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b12ei"}, {357, 260, 554, 2620, 3604, 554, 2605, 3628, 548, 2715, 3738, 540, 2841, 3645, 537, 2824, 3619, 
			519, 2812, 3607, 513, 2864, 3595, 513, 2912, 3648, 512, 2888, 3550, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b12er"}, {366, 263, 523, 1874, 2284, 541, 1677, 2348, 531, 1703, 2341, 527, 1773, 2314, 526, 1790, 2208, 
			520, 1848, 2297, 519, 1866, 2257, 523, 1877, 2181, 536, 1883, 2147, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b12ih"}, {305, 262, 548, 2592, 3485, 550, 2698, 3637, 528, 2711, 3641, 524, 2624, 3610, 531, 2597, 3615, 
			541, 2589, 3516, 548, 2598, 3479, 552, 2594, 3483, 552, 2583, 3501, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b12iy"}, {349, 262, 516, 3126, 3621, 540, 3199, 3785, 538, 3196, 3746, 532, 3172, 3755, 519, 3132, 3686, 
			516, 3126, 3645, 516, 3130, 3636, 517, 3130, 3645, 515, 3129, 3627, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b12oa"}, {403, 261, 716, 0, 2864, 631, 0, 2925, 673, 0, 2888, 717, 0, 2877, 690, 0, 2888, 
			649, 0, 2925, 607, 977, 2933, 568, 1040, 2942, 551, 1129, 2964, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b12oo"}, {325, 268, 588, 1578, 3113, 559, 1649, 3175, 572, 1599, 3106, 567, 1600, 3107, 569, 1646, 3052, 
			582, 1840, 3052, 608, 1931, 3047, 587, 2029, 3027, 561, 2131, 2894, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 0, 1, 0}},
		{{"b12uh"}, {298, 254, 762, 1721, 3234, 760, 1671, 3227, 764, 1737, 3234, 770, 1786, 3242, 766, 1852, 3231, 
			761, 1873, 3206, 754, 1860, 3277, 754, 2034, 3347, 743, 2155, 3381, 85, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 17, 0}},
		{{"b12uw"}, {313, 258, 503, 1255, 3105, 505, 1250, 3106, 505, 1285, 3104, 506, 1281, 3093, 504, 1244, 3087, 
			502, 1260, 3137, 503, 1261, 3124, 503, 1379, 3095, 503, 1521, 3109, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19, 0, 0}},
		{{"b13ae"}, {281, 216, 634, 2535, 3260, 634, 2535, 3260, 630, 2532, 3248, 623, 2533, 3282, 609, 2556, 3300, 
			673, 2481, 3187, 744, 2382, 3102, 803, 2204, 3015, 830, 2136, 2943, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"b13ah"}, {273, 211, 933, 1694, 2860, 919, 1662, 2817, 914, 1686, 2822, 945, 1700, 2855, 939, 1679, 2850, 
			945, 1673, 2862, 941, 1786, 2888, 885, 1903, 2878, 841, 2004, 2846, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"b13aw"}, {322, 210, 875, 1402, 2830, 883, 1401, 2837, 929, 1402, 2826, 844, 1402, 2831, 830, 1403, 2854, 
			817, 1397, 2829, 811, 1417, 2819, 855, 1572, 2836, 842, 1710, 2822, 90, 0, 0, 0, 0, 0, 2, 18, 0, 0, 0, 0, 0}},
		{{"b13eh"}, {258, 227, 640, 2244, 3086, 641, 2262, 3091, 641, 2239, 3086, 644, 2222, 3097, 717, 2224, 3097, 
			790, 2178, 3085, 801, 2233, 3117, 781, 2233, 3114, 723, 2254, 3207, 95, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0}},
		{{"b13ei"}, {280, 242, 532, 2673, 3363, 523, 2674, 3353, 522, 2677, 3346, 507, 2665, 3307, 508, 2615, 3353, 
			512, 2662, 3339, 547, 2752, 3236, 524, 2797, 3168, 484, 2791, 3262, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b13er"}, {255, 193, 574, 1751, 2078, 489, 1676, 2122, 489, 1687, 2094, 482, 1675, 2102, 515, 1665, 2107, 
			554, 1700, 2078, 574, 1751, 2078, 560, 1740, 2094, 553, 1778, 2077, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b13ih"}, {253, 223, 491, 2409, 3166, 489, 2408, 3123, 501, 2387, 3114, 545, 2389, 3131, 592, 2385, 3148, 
			619, 2336, 3155, 590, 2334, 3178, 562, 2317, 3157, 558, 2282, 3181, 95, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b13iy"}, {234, 239, 476, 2772, 3371, 460, 2756, 3376, 478, 2815, 3394, 478, 2830, 3361, 469, 2813, 3392, 
			443, 2808, 3370, 439, 2857, 3359, 434, 2806, 3337, 431, 2772, 3233, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b13oa"}, {259, 218, 586, 1217, 3041, 613, 1189, 2978, 603, 1102, 2993, 590, 1153, 3036, 580, 1072, 2988, 
			624, 1060, 2976, 608, 1023, 2965, 564, 1062, 2955, 523, 1127, 2951, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b13oo"}, {244, 227, 626, 1578, 2786, 626, 1579, 2808, 624, 1578, 2784, 623, 1615, 2856, 627, 1608, 2874, 
			633, 1736, 2871, 624, 1845, 2927, 578, 2016, 2985, 568, 2110, 3016, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"b13uh"}, {250, 211, 639, 1454, 3098, 637, 1445, 3092, 639, 1454, 3098, 635, 1474, 3087, 629, 1536, 3068, 
			643, 1666, 3023, 631, 1793, 2984, 613, 1845, 3037, 567, 1956, 3041, 85, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 17, 0}},
		{{"b13uw"}, {233, 209, 435, 1312, 2715, 438, 1366, 2697, 437, 1337, 2746, 436, 1320, 2775, 435, 1309, 2715, 
			431, 1302, 2718, 432, 1291, 2761, 451, 1288, 2780, 462, 1348, 2803, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"b14ae"}, {314, 198, 697, 2418, 3371, 681, 2444, 3430, 657, 2471, 3376, 697, 2418, 3371, 719, 2377, 3393, 
			760, 2320, 3276, 803, 2203, 2949, 925, 2105, 2797, 908, 2040, 2841, 95, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0}},
		{{"b14ah"}, {263, 201, 998, 1560, 2745, 958, 1469, 2874, 970, 1529, 2800, 998, 1560, 2745, 983, 1555, 2799, 
			992, 1559, 2783, 986, 1590, 2747, 967, 1632, 2797, 934, 1796, 2898, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"b14aw"}, {243, 206, 950, 1246, 2853, 942, 1253, 2885, 953, 1234, 2836, 953, 1254, 2848, 944, 1263, 2864, 
			939, 1325, 2900, 923, 1420, 2902, 884, 1555, 2883, 813, 1746, 2907, 70, 0, 0, 0, 0, 0, 4, 14, 0, 0, 0, 1, 1}},
		{{"b14eh"}, {168, 200, 806, 2172, 3430, 804, 2169, 3452, 806, 2227, 3430, 803, 2166, 3416, 796, 2156, 3389, 
			766, 2256, 3430, 754, 2292, 3485, 749, 2257, 3459, 667, 2252, 3436, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b14ei"}, {260, 203, 436, 2658, 3519, 436, 2658, 3519, 447, 2768, 3525, 452, 2888, 3491, 438, 2917, 3457, 
			432, 2943, 3422, 418, 2943, 3400, 394, 2950, 3388, 388, 2979, 3358, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b14er"}, {283, 213, 619, 1498, 1696, 527, 1476, 1889, 618, 1420, 1818, 584, 1461, 1797, 617, 1495, 1751, 
			622, 1525, 1710, 619, 1505, 1695, 617, 1503, 1723, 602, 1505, 1829, 95, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19}},
		{{"b14ih"}, {177, 221, 470, 2505, 3377, 470, 2505, 3377, 450, 2483, 3170, 473, 2451, 3179, 497, 2413, 3198, 
			554, 2385, 3208, 582, 2324, 3238, 593, 2315, 3247, 580, 2307, 3273, 95, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b14iy"}, {327, 203, 421, 3337, 0, 427, 3279, 0, 416, 3344, 0, 418, 3341, 0, 423, 3312, 0, 
			420, 3350, 0, 418, 3308, 0, 416, 3308, 0, 422, 3318, 0, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b14oa"}, {263, 203, 623, 1044, 3142, 641, 1071, 3159, 626, 1044, 3145, 590, 1004, 3149, 575, 981, 3178, 
			571, 940, 3186, 559, 792, 3191, 474, 754, 3208, 446, 858, 3102, 95, 0, 0, 1, 0, 0, 0, 0, 19, 0, 0, 0, 0}},
		{{"b14oo"}, {179, 212, 494, 1245, 3037, 494, 1274, 3156, 534, 1243, 3085, 561, 1189, 3103, 564, 1260, 3043, 
			567, 1362, 3047, 576, 1428, 3009, 573, 1563, 2986, 569, 1749, 2928, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"b14uh"}, {215, 210, 697, 1305, 3167, 697, 1305, 3167, 715, 1350, 3108, 732, 1399, 3074, 735, 1429, 3034, 
			741, 1450, 3002, 750, 1635, 2967, 743, 1671, 2984, 688, 1855, 2990, 90, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 18, 0}},
		{{"b14uw"}, {263, 206, 418, 940, 2567, 438, 958, 2595, 436, 965, 2618, 432, 923, 2546, 427, 893, 2536, 
			419, 920, 2551, 418, 940, 2567, 413, 960, 2565, 412, 987, 2538, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"b15ae"}, {382, 272, 607, 2620, 3350, 607, 2620, 3350, 617, 2599, 3369, 628, 2563, 3378, 693, 2543, 3401, 
			752, 2382, 3399, 806, 2321, 3459, 807, 2251, 3397, 809, 2185, 3361, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"b15ah"}, {340, 287, 871, 2042, 3410, 852, 1986, 3189, 838, 1968, 3271, 861, 1961, 3302, 862, 1988, 3326, 
			869, 2045, 3354, 873, 2060, 3371, 876, 2043, 3415, 869, 2038, 3407, 85, 0, 0, 0, 1, 0, 17, 0, 0, 1, 0, 1, 0}},
		{{"b15aw"}, {312, 258, 750, 1251, 3003, 749, 1254, 2984, 750, 1252, 3004, 758, 1257, 2952, 753, 1278, 3029, 
			767, 1324, 3016, 783, 1426, 3047, 773, 1624, 2979, 807, 1836, 3011, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"b15eh"}, {312, 279, 734, 2242, 3601, 757, 2189, 3506, 766, 2185, 3504, 763, 2197, 3555, 749, 2232, 3587, 
			734, 2242, 3601, 694, 2236, 3575, 663, 2221, 3571, 607, 2229, 3667, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b15ei"}, {382, 262, 558, 2643, 3216, 558, 2635, 3214, 555, 2685, 3219, 550, 2708, 3222, 545, 2711, 3216, 
			546, 2710, 3248, 550, 2743, 3277, 518, 2785, 3233, 557, 2735, 3246, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b15er"}, {371, 281, 562, 1682, 1981, 544, 1621, 2165, 559, 1662, 2144, 562, 1686, 1983, 562, 1679, 1974, 
			563, 1694, 1990, 563, 1705, 1994, 562, 1728, 2039, 565, 1761, 2085, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b15ih"}, {255, 287, 564, 2515, 3537, 562, 2513, 3438, 563, 2520, 3527, 565, 2507, 3552, 568, 2499, 3595, 
			581, 2412, 3663, 593, 2383, 3777, 602, 2377, 3823, 600, 2376, 3811, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b15iy"}, {329, 259, 446, 3011, 3587, 446, 3018, 3573, 448, 3044, 3546, 452, 3043, 3548, 468, 3058, 3557, 
			461, 3071, 3539, 447, 3115, 3497, 439, 3133, 3474, 444, 3013, 3360, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b15oa"}, {394, 282, 562, 1051, 3094, 654, 1185, 3051, 612, 1074, 3093, 560, 988, 3143, 562, 1050, 3062, 
			563, 1099, 3088, 562, 1083, 3088, 562, 1051, 3094, 562, 1000, 3069, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b15oo"}, {329, 291, 570, 1423, 3457, 563, 1368, 3360, 565, 1394, 3435, 570, 1424, 3463, 570, 1412, 3451, 
			572, 1418, 3421, 572, 1562, 3258, 572, 1754, 3184, 567, 2057, 3154, 90, 0, 0, 0, 0, 0, 0, 0, 0, 18, 2, 0, 0}},
		{{"b15uh"}, {307, 280, 654, 0, 3466, 628, 0, 3360, 631, 0, 3451, 669, 0, 3458, 656, 0, 3436, 
			654, 0, 3466, 680, 0, 3373, 678, 1692, 3500, 635, 1751, 3405, 25, 0, 0, 0, 0, 0, 0, 0, 4, 11, 0, 5, 0}},
		{{"b15uw"}, {324, 282, 548, 1123, 3152, 555, 1355, 3037, 551, 1242, 3026, 555, 1280, 3060, 550, 1209, 3060, 
			546, 1106, 3064, 546, 1112, 3101, 548, 1124, 3162, 551, 1133, 3196, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"b16ae"}, {367, 187, 753, 2227, 3064, 788, 2244, 3150, 750, 2233, 3042, 749, 2241, 2974, 749, 2242, 2978, 
			746, 2235, 2931, 759, 2179, 2927, 815, 2117, 2901, 825, 2080, 2912, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"b16ah"}, {364, 187, 859, 0, 2789, 864, 0, 2833, 836, 0, 2774, 839, 0, 2786, 850, 0, 2814, 
			854, 1401, 2791, 874, 1373, 2813, 882, 1515, 2820, 874, 1671, 2850, 55, 0, 0, 0, 0, 0, 11, 9, 0, 0, 0, 0, 0}},
		{{"b16aw"}, {367, 201, 819, 1074, 2918, 817, 1054, 2895, 830, 1084, 2894, 809, 1053, 2929, 822, 1024, 2925, 
			796, 1001, 2938, 925, 1138, 2869, 898, 1302, 2793, 901, 1574, 2708, 90, 0, 0, 0, 0, 0, 1, 18, 0, 0, 0, 1, 0}},
		{{"b16eh"}, {291, 202, 616, 2356, 3200, 629, 2337, 3242, 619, 2368, 3223, 625, 2368, 3218, 615, 2339, 3184, 
			613, 2326, 3164, 609, 2310, 3189, 608, 2328, 3253, 582, 2325, 3329, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b16ei"}, {346, 193, 564, 2523, 3286, 599, 2545, 3292, 564, 2529, 3298, 564, 2525, 3286, 562, 2623, 3304, 
			559, 2670, 3332, 555, 2626, 3365, 512, 2736, 3417, 493, 2750, 3426, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b16er"}, {334, 200, 591, 1626, 2051, 625, 1607, 2164, 610, 1730, 2135, 599, 1668, 2058, 603, 1666, 2084, 
			596, 1631, 2059, 582, 1615, 2044, 572, 1621, 2039, 564, 1684, 2032, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b16ih"}, {290, 198, 562, 2549, 3101, 476, 2564, 3174, 547, 2536, 3138, 561, 2558, 3114, 562, 2549, 3101, 
			563, 2530, 3102, 564, 2517, 3045, 562, 2502, 3142, 559, 2492, 3212, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b16iy"}, {324, 207, 428, 3070, 0, 436, 2974, 0, 433, 2945, 0, 434, 2963, 0, 433, 2958, 0, 
			432, 3051, 0, 429, 3071, 0, 428, 3069, 0, 428, 3063, 0, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b16oa"}, {315, 217, 588, 1063, 3228, 596, 1126, 3270, 574, 1125, 3216, 576, 1103, 3203, 586, 995, 3173, 
			590, 1026, 3236, 597, 1059, 3245, 559, 1064, 3227, 532, 1057, 3228, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b16oo"}, {261, 202, 615, 1397, 3109, 628, 1415, 3104, 615, 1397, 3109, 601, 1381, 3126, 594, 1394, 3119, 
			579, 1475, 3124, 574, 1546, 3103, 573, 1565, 3074, 570, 1774, 3072, 90, 0, 0, 0, 0, 0, 0, 0, 0, 18, 0, 1, 1}},
		{{"b16uh"}, {300, 183, 739, 1411, 3122, 707, 1262, 3150, 739, 1411, 3122, 739, 1456, 3158, 736, 1564, 3138, 
			741, 1656, 3117, 735, 1856, 3053, 703, 2058, 3086, 616, 2156, 3141, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"b16uw"}, {274, 241, 491, 1467, 2936, 511, 1523, 2904, 500, 1547, 2940, 498, 1532, 2994, 497, 1508, 2979, 
			496, 1497, 2964, 490, 1466, 2934, 482, 1432, 2915, 472, 1423, 2877, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"b17ae"}, {352, 246, 726, 2231, 2932, 726, 2231, 2932, 742, 2246, 2902, 745, 2252, 2898, 743, 2184, 2892, 
			767, 2003, 2873, 886, 1948, 2871, 930, 1861, 2865, 921, 1870, 2958, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"b17ah"}, {327, 245, 898, 1544, 2862, 892, 1559, 2862, 920, 1491, 2814, 935, 1493, 2830, 950, 1495, 2778, 
			945, 1498, 2783, 946, 1576, 2749, 940, 1614, 2722, 855, 1674, 2779, 95, 1, 0, 0, 0, 0, 19, 0, 0, 0, 0, 0, 0}},
		{{"b17aw"}, {315, 264, 744, 1093, 2898, 693, 1004, 3000, 759, 1063, 2930, 762, 1098, 2922, 740, 1083, 2895, 
			752, 1092, 2895, 808, 1319, 2876, 814, 1617, 2886, 763, 1757, 2931, 80, 0, 0, 0, 0, 0, 4, 16, 0, 0, 0, 0, 0}},
		{{"b17eh"}, {189, 241, 749, 1869, 3107, 739, 1879, 3054, 757, 1864, 3074, 764, 1834, 2897, 762, 1817, 2896, 
			761, 1801, 2888, 755, 1824, 2912, 742, 1974, 2941, 711, 2005, 2993, 90, 0, 0, 0, 18, 0, 1, 0, 0, 0, 0, 1, 0}},
		{{"b17ei"}, {245, 250, 499, 2356, 2983, 501, 2269, 3036, 499, 2303, 2952, 499, 2356, 2983, 500, 2402, 2977, 
			501, 2408, 2971, 504, 2453, 2990, 515, 2434, 3115, 526, 2415, 3201, 90, 0, 2, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b17er"}, {304, 247, 556, 1741, 2245, 557, 1741, 2242, 540, 1751, 2263, 531, 1772, 2257, 549, 1804, 2290, 
			553, 1819, 2261, 556, 1832, 2268, 528, 1816, 2315, 513, 1814, 2281, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b17ih"}, {276, 252, 499, 2230, 3249, 498, 2267, 3270, 499, 2230, 3249, 502, 2258, 3177, 511, 2256, 3123, 
			519, 2133, 3136, 535, 2124, 3122, 544, 2134, 3104, 552, 2181, 3089, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b17iy"}, {331, 259, 493, 2811, 0, 500, 2815, 0, 493, 2802, 0, 493, 2811, 0, 494, 2824, 0, 
			497, 2831, 0, 497, 2831, 0, 499, 2788, 0, 500, 2754, 3177, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b17oa"}, {290, 269, 581, 1115, 2924, 681, 1215, 2914, 626, 1163, 2895, 598, 1133, 2920, 580, 1115, 2923, 
			572, 1113, 2943, 562, 1110, 2888, 557, 1241, 2857, 545, 1345, 2802, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b17oo"}, {247, 263, 508, 1245, 2885, 508, 1246, 2802, 520, 1298, 2824, 529, 1358, 2893, 531, 1446, 2895, 
			541, 1565, 2892, 544, 1640, 2884, 540, 1832, 2913, 530, 1871, 2934, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 0, 1, 0}},
		{{"b17uh"}, {239, 257, 696, 1297, 2881, 707, 1317, 2890, 730, 1355, 2912, 737, 1393, 2911, 747, 1432, 2897, 
			748, 1519, 2919, 728, 1603, 2920, 631, 1756, 2947, 563, 1891, 2996, 65, 0, 0, 0, 0, 0, 0, 0, 0, 7, 0, 13, 0}},
		{{"b17uw"}, {228, 262, 505, 1296, 2619, 503, 1521, 2571, 504, 1462, 2576, 504, 1313, 2602, 504, 1290, 2609, 
			505, 1286, 2621, 505, 1297, 2620, 508, 1308, 2637, 511, 1377, 2670, 90, 0, 0, 0, 0, 0, 0, 0, 0, 2, 18, 0, 0}},
		{{"b18ae"}, {307, 249, 741, 2444, 3043, 735, 2446, 3008, 746, 2455, 3021, 748, 2433, 2984, 769, 2340, 2980, 
			819, 2119, 2942, 893, 2116, 2861, 913, 2108, 2843, 888, 2109, 2858, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"b18ah"}, {281, 254, 1010, 1705, 2753, 1013, 1711, 2888, 1007, 1719, 2828, 1010, 1705, 2753, 1029, 1693, 2749, 
			1031, 1668, 2748, 1015, 1663, 2697, 939, 1710, 2687, 920, 1896, 2731, 95, 0, 0, 0, 0, 0, 19, 0, 1, 0, 0, 0, 0}},
		{{"b18aw"}, {269, 249, 808, 1467, 2997, 927, 1353, 2864, 818, 1449, 2780, 814, 1446, 2823, 809, 1462, 2981, 
			807, 1478, 2990, 814, 1473, 3007, 830, 1540, 2988, 821, 1562, 2909, 80, 0, 0, 0, 0, 0, 4, 16, 0, 0, 0, 0, 0}},
		{{"b18eh"}, {195, 241, 710, 1993, 3268, 722, 2036, 3243, 708, 1993, 3258, 710, 1993, 3268, 724, 1991, 3270, 
			719, 2023, 3270, 714, 2061, 3242, 715, 2144, 3225, 718, 2183, 3230, 95, 0, 0, 0, 19, 0, 0, 0, 0, 0, 0, 1, 0}},
		{{"b18ei"}, {265, 257, 621, 2476, 3195, 610, 2466, 3179, 621, 2476, 3195, 589, 2551, 3278, 558, 2571, 3298, 
			539, 2582, 3278, 510, 2614, 3217, 507, 2635, 3166, 512, 2656, 3187, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b18er"}, {304, 272, 559, 1828, 2163, 543, 1819, 2341, 546, 1856, 2282, 553, 1859, 2211, 558, 1829, 2156, 
			560, 1844, 2168, 561, 1835, 2181, 560, 1882, 2186, 557, 1903, 2286, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b18ih"}, {196, 264, 535, 2190, 2998, 560, 2308, 3085, 546, 2198, 2901, 543, 2181, 2956, 530, 2206, 2987, 
			526, 2241, 2992, 529, 2252, 3012, 534, 2245, 3022, 538, 2295, 3063, 90, 0, 18, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b18iy"}, {269, 285, 504, 2803, 3130, 518, 2619, 2930, 508, 2598, 2938, 501, 2598, 3019, 503, 2640, 2979, 
			504, 2698, 3015, 501, 2707, 3072, 504, 2803, 3130, 490, 2809, 3358, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b18oa"}, {260, 257, 613, 1236, 2766, 651, 1300, 2704, 685, 1248, 2703, 671, 1214, 2728, 637, 1222, 2740, 
			617, 1236, 2755, 598, 1234, 2779, 562, 1217, 2769, 530, 1273, 2800, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b18oo"}, {189, 236, 563, 1585, 2841, 527, 1509, 2832, 561, 1552, 2834, 563, 1602, 2832, 553, 1619, 2818, 
			548, 1650, 2848, 526, 1689, 2847, 509, 1758, 3000, 507, 1807, 3037, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"b18uh"}, {233, 266, 780, 1428, 3070, 788, 1423, 3054, 780, 1441, 3035, 789, 1463, 2995, 787, 1479, 3001, 
			776, 1562, 2942, 771, 1601, 2920, 771, 1782, 2935, 751, 1925, 2926, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"b18uw"}, {269, 278, 548, 1377, 2702, 527, 1482, 2730, 538, 1474, 2812, 538, 1432, 2808, 540, 1385, 2716, 
			545, 1376, 2689, 538, 1361, 2707, 536, 1269, 2725, 547, 1345, 2976, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"b19ae"}, {312, 209, 674, 2663, 3243, 684, 2665, 3268, 693, 2672, 3256, 733, 2670, 3271, 731, 2626, 3273, 
			713, 2559, 3233, 752, 2466, 3182, 765, 2348, 3112, 764, 2313, 3107, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"b19ah"}, {292, 218, 951, 1982, 2807, 1072, 2093, 3024, 937, 2048, 2859, 951, 1982, 2807, 877, 1966, 2832, 
			854, 1979, 2875, 856, 2016, 2920, 865, 2021, 2993, 856, 2078, 2988, 55, 0, 0, 0, 0, 9, 11, 0, 0, 0, 0, 0, 0}},
		{{"b19aw"}, {294, 222, 820, 1356, 2700, 812, 1335, 2684, 818, 1357, 2716, 815, 1315, 2697, 818, 1332, 2712, 
			830, 1399, 2720, 829, 1527, 2714, 826, 1678, 2688, 816, 1848, 2743, 55, 0, 0, 0, 0, 0, 9, 11, 0, 0, 0, 0, 0}},
		{{"b19eh"}, {227, 203, 632, 2431, 3091, 659, 2455, 3091, 622, 2396, 3083, 626, 2378, 3070, 682, 2321, 3101, 
			694, 2310, 3138, 693, 2306, 3158, 691, 2301, 3175, 689, 2281, 3183, 95, 0, 1, 0, 19, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b19ei"}, {285, 222, 652, 2661, 3448, 652, 2707, 3457, 610, 2774, 3448, 605, 2836, 3415, 614, 2865, 3384, 
			616, 2891, 3372, 608, 2924, 3421, 604, 2938, 3422, 543, 2942, 3428, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b19er"}, {278, 217, 571, 1984, 2206, 667, 1895, 2309, 625, 1923, 2223, 622, 1927, 2196, 597, 1952, 2222, 
			571, 1984, 2206, 556, 1996, 2231, 525, 2009, 2257, 452, 2040, 2292, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b19ih"}, {263, 212, 520, 2695, 3373, 468, 2716, 3436, 546, 2695, 3329, 571, 2635, 3313, 586, 2567, 3284, 
			611, 2508, 3316, 620, 2502, 3294, 624, 2467, 3337, 627, 2377, 3312, 90, 0, 18, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b19iy"}, {288, 228, 442, 3135, 0, 445, 3143, 0, 488, 3060, 0, 486, 3117, 0, 464, 3081, 0, 
			463, 3154, 0, 441, 3140, 0, 460, 3135, 0, 477, 3035, 3338, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b19oa"}, {335, 185, 548, 1062, 2926, 694, 1189, 2710, 685, 1119, 2777, 637, 1103, 2844, 570, 1156, 2885, 
			558, 1104, 2881, 555, 1093, 2894, 552, 1089, 2919, 553, 1105, 2922, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b19oo"}, {294, 252, 578, 1574, 3026, 573, 1573, 3014, 594, 1569, 3023, 615, 1619, 3037, 724, 1793, 3049, 
			724, 1844, 3073, 724, 1891, 3085, 720, 1942, 3097, 731, 2000, 3138, 90, 0, 0, 0, 0, 0, 0, 0, 0, 18, 1, 1, 0}},
		{{"b19uh"}, {241, 235, 754, 1742, 3002, 846, 1739, 3025, 761, 1750, 3006, 740, 1725, 3010, 707, 1925, 3079, 
			709, 1952, 3075, 748, 2023, 3049, 752, 2062, 2990, 751, 2066, 3078, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"b19uw"}, {270, 281, 550, 1114, 3012, 544, 1111, 2981, 545, 1105, 2996, 546, 1115, 2985, 542, 1117, 2932, 
			549, 1114, 2958, 549, 1112, 3024, 553, 1120, 3003, 557, 1212, 2946, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"b21ae"}, {352, 205, 769, 2234, 2910, 766, 2245, 2917, 771, 2215, 2889, 771, 2171, 2866, 761, 2113, 2808, 
			771, 2047, 2753, 770, 1961, 2733, 768, 1923, 2708, 760, 1911, 2718, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"b21ah"}, {363, 226, 1010, 1604, 2588, 970, 1628, 2577, 979, 1590, 2574, 1021, 1632, 2608, 1017, 1657, 2587, 
			978, 1686, 2602, 907, 1683, 2589, 879, 1694, 2565, 834, 1827, 2635, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"b21aw"}, {376, 240, 831, 0, 2566, 759, 0, 2552, 813, 0, 2573, 818, 0, 2562, 831, 0, 2566, 
			839, 0, 2553, 866, 0, 2516, 914, 0, 2509, 917, 0, 2513, 80, 0, 0, 0, 0, 0, 4, 16, 0, 0, 0, 0, 0}},
		{{"b21eh"}, {274, 204, 642, 2015, 2916, 592, 2053, 2965, 623, 2025, 2957, 642, 2018, 2940, 624, 2027, 2943, 
			624, 2034, 2937, 630, 2025, 2965, 639, 2027, 2983, 630, 2030, 3021, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b21ei"}, {465, 232, 463, 2745, 3155, 465, 2572, 3051, 472, 2657, 3040, 466, 2731, 3104, 456, 2746, 3157, 
			447, 2753, 3171, 445, 2785, 3173, 462, 2835, 3089, 480, 2839, 3122, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b21er"}, {378, 247, 495, 1555, 1948, 589, 1557, 2119, 611, 1550, 2122, 545, 1568, 2096, 499, 1545, 1967, 
			495, 1543, 1944, 496, 1512, 1966, 498, 1501, 1991, 500, 1701, 1969, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b21ih"}, {300, 257, 506, 2432, 3008, 506, 2432, 3008, 505, 2424, 3031, 511, 2357, 3050, 531, 2313, 3052, 
			552, 2312, 3029, 571, 2313, 3051, 604, 2279, 3006, 558, 2257, 3006, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b21iy"}, {398, 247, 427, 2952, 3556, 432, 2943, 3428, 435, 3021, 3543, 442, 2956, 3513, 439, 2937, 3480, 
			434, 2951, 3523, 413, 2950, 3547, 391, 2978, 3481, 409, 2987, 3454, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b21oa"}, {397, 243, 584, 936, 2679, 565, 913, 2698, 579, 935, 2672, 575, 933, 2695, 566, 917, 2767, 
			541, 933, 2715, 500, 972, 2704, 498, 984, 2768, 494, 953, 2821, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b21oo"}, {270, 221, 531, 1540, 2931, 475, 1525, 2899, 497, 1531, 2953, 505, 1532, 2936, 521, 1538, 2926, 
			552, 1550, 2927, 566, 1562, 2876, 571, 1589, 2855, 574, 1734, 2788, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 0, 0, 1}},
		{{"b21uh"}, {254, 212, 632, 1352, 2740, 484, 1332, 2748, 619, 1341, 2748, 631, 1352, 2743, 632, 1348, 2718, 
			630, 1361, 2713, 639, 1445, 2745, 650, 1489, 2748, 631, 1544, 2744, 85, 0, 0, 0, 0, 0, 1, 0, 0, 2, 0, 17, 0}},
		{{"b21uw"}, {320, 260, 504, 997, 2826, 499, 1020, 2816, 501, 956, 2823, 503, 945, 2828, 504, 987, 2825, 
			504, 998, 2832, 504, 1005, 2849, 505, 1020, 2850, 506, 1023, 2860, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"b22ae"}, {256, 229, 678, 2524, 3418, 687, 2580, 3288, 678, 2501, 3424, 677, 2509, 3370, 659, 2465, 3353, 
			595, 2370, 3381, 581, 2222, 3321, 586, 2146, 3293, 614, 2051, 3333, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"b22ah"}, {350, 263, 854, 1518, 2988, 854, 1518, 2988, 903, 1499, 2991, 943, 1536, 2986, 912, 1595, 2992, 
			871, 1655, 2968, 835, 1725, 2949, 820, 1728, 2872, 831, 1764, 2837, 85, 0, 0, 0, 0, 0, 17, 2, 0, 0, 0, 1, 0}},
		{{"b22aw"}, {384, 249, 746, 1309, 2957, 752, 1304, 2851, 746, 1309, 2957, 737, 1368, 2928, 736, 1422, 2929, 
			735, 1439, 2929, 736, 1450, 2932, 726, 1456, 2930, 723, 1635, 2896, 30, 0, 0, 0, 0, 0, 0, 6, 0, 0, 0, 14, 0}},
		{{"b22eh"}, {212, 242, 699, 2169, 3243, 699, 2169, 3243, 702, 2177, 3181, 692, 2162, 3186, 691, 2151, 3211, 
			690, 2138, 3236, 687, 2131, 3270, 686, 2134, 3311, 690, 2159, 3420, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b22ei"}, {322, 250, 558, 2559, 3311, 549, 2519, 3427, 558, 2559, 3311, 545, 2625, 3323, 531, 2658, 3361, 
			513, 2673, 3342, 500, 2684, 3351, 499, 2713, 3260, 506, 2734, 3247, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b22er"}, {336, 259, 678, 1594, 2048, 663, 1591, 2053, 696, 1591, 2041, 685, 1663, 1985, 673, 1723, 1995, 
			637, 1682, 2054, 610, 1752, 2056, 592, 1821, 2068, 510, 1876, 2225, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b22ih"}, {187, 231, 489, 2544, 3720, 489, 2544, 3720, 489, 2494, 3716, 494, 2482, 3769, 509, 2452, 3824, 
			530, 2434, 3794, 541, 2350, 3682, 526, 2321, 3623, 501, 2266, 3664, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b22iy"}, {325, 260, 475, 2916, 3399, 475, 2916, 3399, 471, 2966, 3391, 465, 2999, 3360, 462, 3003, 3359, 
			476, 2994, 3326, 484, 2975, 3336, 482, 3016, 3350, 478, 2983, 3365, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b22oa"}, {370, 257, 588, 1117, 2848, 649, 1139, 2827, 634, 1172, 2889, 606, 1160, 2860, 588, 1117, 2848, 
			613, 1043, 2852, 603, 1025, 2877, 569, 1035, 2963, 550, 1067, 3014, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b22oo"}, {232, 237, 627, 1703, 2883, 530, 1628, 2978, 632, 1709, 2817, 641, 1816, 2948, 638, 1893, 3055, 
			646, 1916, 3101, 660, 1956, 3103, 646, 2012, 3206, 623, 2175, 3283, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 0, 1, 0}},
		{{"b22uh"}, {296, 255, 746, 1552, 3044, 745, 1570, 3161, 746, 1552, 3044, 725, 1587, 3057, 709, 1661, 3049, 
			703, 1692, 3061, 695, 1686, 3063, 694, 1714, 3061, 692, 1871, 3048, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"b22uw"}, {243, 256, 504, 1515, 2952, 543, 1573, 2909, 504, 1515, 2937, 502, 1508, 2965, 499, 1508, 2939, 
			498, 1495, 2942, 498, 1440, 2937, 499, 1432, 2925, 502, 1482, 2817, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19, 0, 0}},
		{{"b23ae"}, {346, 267, 809, 2592, 3331, 805, 2604, 3340, 796, 2595, 3344, 797, 2558, 3308, 915, 2464, 3304, 
			978, 2309, 3264, 992, 2170, 3200, 1039, 1910, 3153, 1010, 1896, 3137, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"b23ah"}, {292, 277, 1119, 1588, 3010, 1127, 1584, 3003, 1115, 1577, 3013, 1097, 1553, 2966, 1108, 1552, 2914, 
			1188, 1462, 2934, 1225, 1552, 2951, 1195, 1587, 2861, 1070, 1955, 2885, 90, 0, 0, 0, 0, 0, 18, 2, 0, 0, 0, 0, 0}},
		{{"b23aw"}, {313, 268, 765, 1067, 3266, 795, 1111, 3310, 765, 1067, 3266, 749, 1062, 3231, 664, 1045, 3262, 
			1006, 1322, 3192, 1006, 1367, 3142, 1045, 1393, 3122, 996, 1678, 3121, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"b23eh"}, {238, 270, 1057, 2426, 3441, 1052, 2421, 3441, 1055, 2415, 3444, 1031, 2398, 3463, 997, 2387, 3430, 
			951, 2400, 3464, 834, 2494, 3521, 767, 2487, 3548, 729, 2545, 3618, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b23ei"}, {336, 271, 586, 2720, 3446, 602, 2752, 3440, 566, 2727, 3468, 554, 2766, 3367, 522, 2821, 3439, 
			511, 2803, 3409, 497, 2894, 3522, 493, 2919, 3488, 499, 2981, 3570, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b23er"}, {332, 246, 614, 1867, 2435, 575, 1963, 2476, 608, 1840, 2453, 620, 1912, 2501, 630, 1886, 2445, 
			611, 1879, 2431, 591, 1857, 2403, 556, 1943, 2383, 527, 1958, 2368, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b23ih"}, {205, 276, 559, 2753, 3502, 562, 2760, 3505, 559, 2762, 3464, 555, 2698, 3482, 545, 2689, 3466, 
			522, 2684, 3515, 503, 2652, 3500, 501, 2647, 3550, 499, 2605, 3571, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b23iy"}, {277, 293, 366, 3207, 3722, 361, 3075, 3602, 364, 3218, 3722, 362, 3184, 3830, 388, 3168, 3811, 
			426, 3176, 3785, 466, 3138, 3767, 487, 3115, 3774, 498, 3091, 3718, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b23oa"}, {314, 274, 557, 1065, 3040, 611, 1304, 3029, 573, 1250, 3021, 562, 1136, 3066, 560, 1105, 3043, 
			557, 1065, 3040, 555, 1063, 3101, 555, 1044, 3171, 552, 1046, 3240, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b23oo"}, {206, 277, 561, 1627, 3037, 562, 1628, 3049, 561, 1627, 3023, 561, 1667, 3024, 562, 1728, 3058, 
			563, 1884, 3100, 563, 1957, 3121, 562, 1991, 3128, 562, 2023, 3130, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"b23uh"}, {219, 282, 863, 1566, 3134, 860, 1567, 3102, 863, 1565, 3117, 848, 1564, 3101, 818, 1629, 3094, 
			809, 1798, 3142, 796, 1902, 3146, 778, 2127, 3202, 745, 2262, 3353, 95, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 19, 0}},
		{{"b23uw"}, {251, 283, 557, 0, 3111, 560, 0, 3108, 555, 0, 3112, 557, 0, 3064, 553, 0, 3050, 
			545, 0, 3101, 522, 0, 3050, 516, 0, 3129, 511, 0, 3116, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"b24ae"}, {216, 206, 545, 2690, 0, 502, 2701, 0, 536, 2698, 0, 553, 2695, 0, 584, 2698, 0, 
			674, 2319, 2733, 754, 2153, 2741, 785, 2101, 2739, 795, 2060, 2739, 90, 0, 0, 0, 2, 18, 0, 0, 0, 0, 0, 0, 0}},
		{{"b24ah"}, {238, 228, 1205, 0, 2708, 1196, 0, 2730, 1219, 0, 2732, 1253, 0, 2735, 1240, 0, 2688, 
			1196, 0, 2664, 1169, 1470, 2582, 1056, 1513, 2508, 930, 1643, 2547, 85, 0, 0, 0, 0, 0, 17, 3, 0, 0, 0, 0, 0}},
		{{"b24aw"}, {196, 203, 770, 1010, 2411, 746, 1090, 2464, 752, 1077, 2464, 770, 1010, 2411, 775, 985, 2393, 
			793, 1003, 2395, 815, 1119, 2396, 818, 1335, 2392, 804, 1491, 2359, 90, 0, 0, 0, 0, 0, 2, 18, 0, 0, 0, 0, 0}},
		{{"b24eh"}, {165, 212, 744, 2227, 2966, 705, 2239, 2995, 740, 2227, 2975, 771, 2207, 2969, 790, 2196, 2978, 
			791, 2183, 3024, 775, 2185, 3083, 748, 2192, 3182, 680, 2247, 3248, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b24ei"}, {199, 212, 613, 2593, 3103, 623, 2562, 3099, 620, 2596, 3037, 600, 2608, 3099, 512, 2644, 3058, 
			465, 2741, 3067, 449, 2741, 3024, 440, 2764, 3047, 438, 2802, 3085, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b24er"}, {270, 248, 568, 1512, 1783, 564, 1495, 1783, 562, 1519, 1746, 554, 1549, 1728, 583, 1473, 1746, 
			570, 1389, 1784, 623, 1470, 1745, 606, 1454, 1808, 560, 1550, 1880, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b24ih"}, {137, 229, 447, 2645, 3302, 447, 2645, 3302, 445, 2632, 3277, 441, 2618, 3200, 440, 2568, 3159, 
			441, 2481, 3051, 443, 2448, 3015, 447, 2424, 3022, 446, 2409, 3043, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b24iy"}, {205, 213, 424, 3199, 3815, 429, 3178, 3823, 432, 3166, 3772, 434, 3172, 3668, 435, 3175, 3586, 
			437, 3181, 3579, 439, 3198, 3657, 436, 3168, 3690, 437, 3076, 3695, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b24oa"}, {216, 212, 615, 0, 2691, 626, 0, 2614, 617, 0, 2652, 618, 0, 2693, 612, 0, 2702, 
			569, 0, 2737, 556, 794, 2716, 512, 814, 2761, 489, 0, 2944, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b24oo"}, {169, 237, 493, 1248, 2751, 493, 1248, 2751, 494, 1293, 2761, 494, 1351, 2761, 494, 1394, 2714, 
			497, 1410, 2686, 535, 1563, 2667, 558, 1731, 2695, 557, 1896, 2752, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"b24uh"}, {158, 213, 685, 1485, 2679, 685, 1485, 2679, 710, 1458, 2675, 725, 1445, 2667, 721, 1447, 2644, 
			726, 1451, 2641, 727, 1511, 2664, 708, 1630, 2688, 626, 1713, 2880, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"b24uw"}, {214, 221, 438, 1336, 3117, 439, 1341, 3119, 437, 1335, 3115, 437, 1255, 3107, 437, 1213, 3114, 
			438, 1245, 3117, 439, 1266, 3119, 440, 1297, 3108, 441, 1319, 3086, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"b25ae"}, {284, 223, 669, 2440, 0, 645, 2467, 0, 675, 2441, 0, 661, 2394, 0, 675, 2332, 0, 
			765, 2275, 0, 853, 2190, 0, 856, 2072, 0, 838, 2023, 0, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"b25ah"}, {295, 220, 872, 1565, 2531, 875, 1696, 2529, 868, 1587, 2531, 876, 1599, 2562, 864, 1700, 2600, 
			885, 1746, 2621, 870, 1776, 2617, 874, 1803, 2639, 861, 1966, 2721, 95, 0, 0, 0, 0, 0, 19, 0, 0, 0, 0, 1, 0}},
		{{"b25aw"}, {340, 213, 839, 1240, 2636, 818, 1245, 2684, 843, 1246, 2636, 855, 1264, 2762, 857, 1282, 2762, 
			856, 1289, 2695, 861, 1401, 2747, 863, 1476, 2749, 857, 1741, 2753, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"b25eh"}, {215, 232, 692, 2354, 3179, 699, 2354, 3175, 692, 2316, 3178, 770, 2268, 3171, 854, 2180, 3160, 
			880, 2107, 3188, 877, 2099, 3188, 866, 2126, 3197, 669, 2214, 3175, 90, 0, 0, 0, 18, 2, 0, 0, 0, 0, 0, 0, 0}},
		{{"b25ei"}, {294, 219, 441, 2585, 3014, 531, 2555, 2981, 502, 2608, 2975, 480, 2582, 3008, 441, 2585, 3014, 
			438, 2616, 3019, 436, 2672, 3025, 435, 2613, 3065, 436, 2634, 3064, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b25er"}, {277, 227, 488, 1584, 1810, 480, 1596, 1863, 489, 1577, 1833, 485, 1588, 1808, 465, 1593, 1824, 
			460, 1588, 1820, 462, 1605, 1814, 462, 1617, 1837, 465, 1635, 1859, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b25ih"}, {249, 212, 479, 2492, 3187, 462, 2537, 3233, 479, 2492, 3187, 495, 2409, 3194, 498, 2357, 3152, 
			504, 2293, 3163, 524, 2200, 3123, 556, 2205, 3131, 568, 2216, 3136, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b25iy"}, {309, 239, 382, 2987, 0, 393, 2965, 0, 421, 3011, 3598, 444, 2992, 3445, 431, 3009, 0, 
			438, 3069, 0, 445, 3032, 0, 438, 2918, 3357, 387, 2877, 3356, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b25oa"}, {292, 223, 625, 1100, 2747, 628, 1101, 2795, 593, 1080, 2822, 549, 1102, 2763, 556, 1120, 2690, 
			549, 1124, 2741, 533, 1127, 2942, 504, 1132, 2972, 487, 1126, 2952, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b25oo"}, {254, 238, 512, 1260, 2826, 511, 1232, 2818, 512, 1274, 2826, 547, 1306, 2796, 553, 1375, 2840, 
			557, 1395, 2854, 565, 1434, 2843, 563, 1466, 2860, 548, 1703, 2851, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"b25uh"}, {223, 220, 688, 1251, 2795, 689, 1265, 2931, 687, 1246, 2779, 685, 1243, 2753, 686, 1244, 2761, 
			693, 1245, 2749, 735, 1336, 2754, 751, 1443, 2749, 740, 1582, 2877, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"b25uw"}, {286, 219, 436, 1170, 0, 444, 1324, 3150, 438, 1237, 3110, 436, 1203, 3186, 437, 1146, 0, 
			435, 1094, 0, 435, 1033, 0, 436, 1037, 0, 437, 1026, 2987, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"b26ae"}, {451, 220, 643, 2434, 3326, 646, 2445, 3306, 643, 2422, 3335, 660, 2391, 3318, 677, 2262, 3291, 
			791, 2094, 2958, 874, 1986, 3056, 918, 1898, 2833, 917, 1897, 3062, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"b26ah"}, {314, 210, 776, 1490, 2730, 756, 1467, 3001, 765, 1481, 2726, 781, 1476, 2721, 808, 1478, 2725, 
			785, 1488, 2662, 811, 1503, 0, 879, 1573, 0, 814, 1693, 0, 55, 0, 0, 0, 0, 0, 11, 9, 0, 0, 0, 0, 0}},
		{{"b26aw"}, {362, 220, 635, 1163, 0, 702, 1259, 0, 635, 1163, 0, 635, 1076, 0, 639, 1075, 0, 
			633, 1206, 0, 652, 1237, 0, 736, 1348, 0, 797, 1483, 0, 70, 0, 0, 0, 0, 0, 3, 14, 0, 0, 0, 3, 0}},
		{{"b26eh"}, {331, 217, 668, 2106, 3353, 676, 2136, 3418, 673, 2127, 3353, 661, 2074, 3358, 684, 2065, 3329, 
			682, 2038, 3275, 679, 2026, 3252, 671, 2065, 3257, 664, 2074, 3303, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b26ei"}, {426, 233, 654, 2549, 3427, 654, 2549, 3427, 614, 2540, 3491, 614, 2572, 3489, 570, 2648, 3503, 
			493, 2703, 3558, 456, 2805, 3444, 447, 2885, 3431, 442, 2942, 3339, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b26er"}, {347, 224, 479, 1579, 1963, 469, 1579, 1963, 489, 1569, 1858, 487, 1561, 1854, 477, 1551, 1802, 
			468, 1554, 1780, 465, 1572, 1813, 454, 1612, 1884, 460, 1665, 2061, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b26ih"}, {316, 218, 481, 2567, 3550, 481, 2580, 3507, 495, 2296, 3298, 517, 2346, 3337, 565, 2269, 3442, 
			567, 2204, 3428, 559, 2210, 3471, 575, 2187, 3503, 573, 2147, 3468, 95, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b26iy"}, {342, 218, 441, 3079, 3474, 484, 3085, 3846, 477, 3069, 3585, 449, 3119, 3412, 441, 3111, 3500, 
			438, 3062, 3456, 440, 3070, 3352, 446, 2934, 3311, 454, 2951, 0, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b26oa"}, {338, 215, 646, 1244, 3023, 652, 1258, 3037, 630, 1239, 2990, 629, 1135, 2932, 625, 990, 2968, 
			620, 971, 2952, 586, 899, 2917, 488, 883, 3032, 448, 918, 3101, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b26oo"}, {303, 227, 546, 1254, 3333, 546, 1254, 3333, 571, 1263, 3301, 586, 1364, 3273, 621, 1507, 3231, 
			635, 1630, 3213, 638, 1747, 3228, 648, 1888, 3121, 694, 2000, 3139, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"b26uh"}, {292, 216, 670, 1538, 3264, 694, 1597, 3284, 677, 1559, 3241, 655, 1531, 3261, 661, 1532, 3261, 
			664, 1544, 3232, 661, 1583, 3289, 681, 1628, 3293, 679, 1720, 3243, 95, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 19, 0}},
		{{"b26uw"}, {320, 235, 449, 916, 3002, 442, 977, 3100, 440, 946, 3049, 440, 888, 3005, 440, 830, 3000, 
			441, 833, 2977, 451, 918, 2994, 477, 937, 3024, 492, 958, 2994, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"b27ae"}, {243, 211, 634, 2410, 3303, 635, 2401, 3307, 675, 2370, 3261, 713, 2258, 3236, 759, 2214, 3196, 
			787, 2152, 3175, 785, 2082, 3154, 776, 2069, 3195, 745, 2128, 3236, 80, 0, 0, 0, 4, 16, 0, 0, 0, 0, 0, 0, 0}},
		{{"b27ah"}, {270, 223, 1147, 1553, 2877, 1113, 1562, 2905, 1143, 1553, 2873, 1112, 1541, 2885, 1119, 1546, 2886, 
			1077, 1562, 2870, 1055, 1585, 2888, 1041, 1605, 2904, 838, 1806, 2911, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"b27aw"}, {250, 215, 792, 1268, 2988, 792, 1268, 2988, 784, 1268, 2940, 797, 1277, 2886, 791, 1307, 2882, 
			801, 1347, 2879, 791, 1520, 2913, 777, 1695, 2923, 706, 1936, 3037, 80, 0, 0, 1, 0, 0, 3, 16, 0, 0, 0, 0, 0}},
		{{"b27eh"}, {192, 231, 699, 2194, 3317, 695, 2196, 3314, 696, 2187, 3317, 703, 2190, 3289, 701, 2197, 3276, 
			698, 2199, 3303, 695, 2197, 3305, 692, 2259, 3320, 688, 2304, 3383, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b27ei"}, {293, 239, 494, 2823, 3444, 511, 2651, 3430, 523, 2731, 3451, 501, 2719, 3447, 494, 2823, 3444, 
			493, 2870, 3465, 494, 2879, 3453, 493, 2877, 3388, 492, 2877, 3421, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b27er"}, {251, 211, 615, 1742, 2009, 622, 1703, 2081, 616, 1699, 2037, 615, 1742, 2009, 616, 1790, 1981, 
			618, 1794, 2041, 614, 1813, 2128, 601, 1835, 2216, 570, 1876, 2272, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b27ih"}, {174, 230, 551, 2362, 3354, 532, 2369, 3330, 551, 2362, 3354, 550, 2346, 3387, 550, 2345, 3396, 
			557, 2360, 3384, 558, 2364, 3387, 515, 2376, 3423, 507, 2377, 3447, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b27iy"}, {241, 239, 410, 2874, 3505, 368, 2862, 3534, 397, 2866, 3506, 421, 2881, 3493, 424, 2895, 3478, 
			415, 2906, 3472, 424, 2886, 3486, 431, 2885, 3479, 433, 2890, 3449, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b27oa"}, {264, 225, 658, 1272, 2944, 650, 1272, 2937, 644, 1227, 2929, 626, 1109, 2915, 567, 1094, 2932, 
			557, 1072, 2936, 503, 1043, 2929, 488, 1021, 2967, 443, 1048, 3006, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b27oo"}, {174, 211, 622, 1417, 2993, 620, 1362, 3002, 622, 1417, 2993, 621, 1451, 2956, 623, 1520, 2959, 
			623, 1660, 2981, 624, 1784, 3013, 621, 1901, 3088, 607, 1951, 3133, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"b27uh"}, {172, 214, 763, 1522, 2931, 763, 1529, 2960, 750, 1514, 2931, 746, 1496, 2876, 730, 1549, 2858, 
			710, 1712, 2864, 692, 1768, 2867, 651, 1870, 2943, 643, 1955, 3013, 95, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 19, 0}},
		{{"b27uw"}, {233, 254, 495, 1283, 3033, 493, 1447, 2988, 494, 1479, 2988, 494, 1348, 3006, 495, 1301, 3022, 
			494, 1271, 3033, 489, 1253, 3015, 486, 1256, 2991, 487, 1285, 2986, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"b28ae"}, {284, 227, 676, 2253, 3121, 675, 2252, 3121, 681, 2257, 3146, 690, 2267, 3104, 714, 2211, 3054, 
			744, 2053, 3010, 826, 1894, 2950, 861, 1817, 2931, 789, 1783, 2876, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"b28ah"}, {271, 221, 1065, 1483, 2723, 1080, 1547, 2859, 1068, 1516, 2780, 1065, 1483, 2723, 1067, 1490, 2665, 
			1048, 1455, 2652, 1040, 1481, 2641, 966, 1541, 2716, 885, 1590, 2957, 75, 0, 0, 0, 0, 0, 15, 5, 0, 0, 0, 0, 0}},
		{{"b28aw"}, {267, 216, 640, 929, 2654, 638, 895, 2701, 640, 929, 2654, 635, 932, 2624, 671, 954, 2627, 
			678, 981, 2639, 690, 996, 2704, 690, 1154, 2746, 687, 1255, 2770, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"b28eh"}, {210, 221, 680, 2187, 3099, 675, 2186, 3134, 680, 2187, 3073, 682, 2186, 3056, 682, 2190, 3098, 
			682, 2183, 3099, 683, 2158, 3098, 690, 2124, 3119, 688, 2124, 3169, 95, 0, 1, 0, 19, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b28ei"}, {309, 239, 510, 2438, 3126, 521, 2382, 3207, 520, 2401, 3154, 510, 2438, 3126, 500, 2438, 3121, 
			496, 2449, 3092, 494, 2568, 3084, 497, 2582, 2996, 493, 2673, 3070, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b28er"}, {263, 220, 457, 1570, 1926, 501, 1559, 2051, 473, 1560, 1992, 459, 1578, 1963, 454, 1555, 1925, 
			450, 1542, 1892, 447, 1530, 1864, 443, 1514, 1885, 440, 1542, 1929, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b28ih"}, {194, 240, 490, 2449, 3330, 493, 2464, 3347, 490, 2452, 3306, 492, 2430, 3285, 493, 2393, 3274, 
			495, 2376, 3215, 496, 2363, 3162, 497, 2310, 3131, 497, 2292, 3109, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b28iy"}, {268, 240, 361, 2938, 3810, 380, 2945, 3794, 369, 2984, 3851, 342, 2934, 3730, 359, 2934, 3822, 
			391, 2967, 3768, 417, 2981, 3752, 411, 3002, 3731, 415, 2945, 3629, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b28oa"}, {293, 224, 616, 1047, 2669, 622, 1010, 2644, 616, 1047, 2669, 595, 1059, 2686, 555, 1045, 2643, 
			528, 1006, 2598, 499, 979, 2598, 491, 960, 2623, 495, 985, 2546, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b28oo"}, {213, 239, 492, 1200, 2622, 491, 1188, 2653, 492, 1200, 2622, 494, 1201, 2620, 497, 1224, 2631, 
			498, 1273, 2624, 503, 1347, 2617, 512, 1408, 2690, 513, 1436, 2716, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"b28uh"}, {212, 206, 667, 1400, 3017, 629, 1402, 3027, 667, 1400, 3017, 668, 1407, 3019, 675, 1414, 3017, 
			677, 1423, 3048, 673, 1438, 3012, 650, 1475, 2956, 632, 1706, 2982, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"b28uw"}, {255, 216, 434, 1297, 2694, 434, 1428, 2697, 434, 1297, 2694, 433, 1259, 2690, 433, 1268, 2677, 
			433, 1258, 2638, 433, 1262, 2632, 433, 1284, 2614, 432, 1278, 2579, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"b29ae"}, {291, 212, 860, 2503, 3002, 899, 2460, 3048, 867, 2468, 3022, 856, 2501, 2996, 861, 2488, 2975, 
			907, 2232, 2886, 994, 2080, 2867, 1024, 1976, 2818, 1010, 1973, 2896, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"b29ah"}, {285, 227, 944, 1540, 2896, 954, 1540, 2921, 942, 1548, 2892, 932, 1559, 2892, 953, 1583, 2921, 
			931, 1587, 2862, 902, 1646, 2863, 905, 1750, 2847, 881, 1856, 2833, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"b29aw"}, {305, 214, 941, 1409, 2729, 967, 1419, 2723, 934, 1410, 2733, 934, 1378, 2713, 955, 1461, 2740, 
			924, 1505, 2749, 899, 1557, 2799, 881, 1692, 2872, 902, 1868, 2990, 45, 0, 0, 0, 0, 0, 11, 9, 0, 0, 0, 0, 0}},
		{{"b29eh"}, {212, 231, 856, 2304, 2916, 745, 2249, 2940, 840, 2279, 2893, 856, 2304, 2916, 856, 2298, 2936, 
			823, 2286, 2975, 804, 2280, 2998, 758, 2288, 3027, 701, 2266, 3037, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b29ei"}, {289, 236, 655, 2772, 0, 679, 2779, 0, 645, 2788, 0, 516, 2873, 0, 500, 2949, 0, 
			496, 2929, 0, 496, 2936, 0, 488, 2887, 0, 493, 2900, 0, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b29er"}, {390, 250, 614, 1733, 2034, 659, 1693, 2161, 638, 1686, 2120, 616, 1716, 2061, 615, 1745, 2062, 
			604, 1758, 2048, 585, 1777, 2103, 554, 1770, 2154, 505, 1840, 2132, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"b29ih"}, {294, 243, 503, 2433, 3150, 506, 2451, 3149, 504, 2457, 3155, 515, 2438, 3161, 527, 2441, 3188, 
			557, 2407, 3269, 579, 2353, 3332, 574, 2341, 3366, 570, 2319, 3373, 90, 0, 18, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b29iy"}, {269, 255, 366, 3024, 3867, 394, 2930, 3893, 381, 3007, 3857, 381, 3031, 3862, 366, 3024, 3853, 
			375, 3046, 3864, 378, 3047, 3861, 386, 3036, 3827, 414, 2989, 3717, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"b29oa"}, {267, 241, 578, 1122, 2833, 678, 1245, 2888, 685, 1197, 2842, 669, 1162, 2838, 620, 1126, 2832, 
			586, 1114, 2833, 574, 1151, 2830, 559, 1164, 2826, 506, 1140, 2844, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"b29oo"}, {221, 233, 684, 1623, 3019, 509, 1603, 2864, 679, 1626, 3000, 691, 1625, 3016, 690, 1624, 3019, 
			691, 1628, 2989, 688, 1630, 3017, 686, 1710, 3023, 686, 1825, 2930, 90, 0, 0, 0, 0, 0, 0, 0, 0, 18, 0, 2, 0}},
		{{"b29uh"}, {243, 234, 778, 1422, 3266, 780, 1435, 3245, 769, 1407, 3255, 778, 1385, 3262, 776, 1419, 3266, 
			769, 1474, 3241, 754, 1597, 3189, 736, 1725, 3108, 694, 1882, 3161, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 19, 0}},
		{{"b29uw"}, {302, 249, 498, 1509, 2866, 491, 1537, 2906, 476, 1508, 2849, 483, 1458, 2844, 498, 1509, 2866, 
			498, 1500, 2888, 498, 1509, 2910, 498, 1512, 2926, 498, 1515, 2926, 90, 0, 0, 0, 0, 0, 0, 0, 0, 2, 18, 0, 0}},
		{{"g01ae"}, {295, 242, 741, 2433, 3341, 746, 2390, 3355, 712, 2329, 3328, 722, 2375, 3307, 769, 2305, 3220, 
			875, 2176, 2888, 861, 2061, 2945, 954, 2028, 2853, 889, 2008, 2939, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"g01ah"}, {276, 238, 894, 1420, 2828, 788, 1326, 2751, 873, 1416, 2866, 895, 1447, 2895, 920, 1467, 2893, 
			906, 1453, 2912, 954, 1528, 2907, 929, 1560, 2970, 833, 1631, 2920, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"g01aw"}, {349, 240, 867, 1292, 3296, 850, 1244, 3232, 891, 1293, 3326, 894, 1352, 3296, 885, 1398, 3295, 
			888, 1431, 3302, 885, 1433, 3301, 843, 1625, 3181, 738, 1826, 3133, 80, 0, 0, 0, 0, 0, 4, 16, 0, 0, 0, 0, 0}},
		{{"g01eh"}, {202, 227, 653, 2241, 3167, 705, 2205, 3304, 707, 2195, 3229, 702, 2250, 3179, 641, 2207, 3159, 
			656, 2247, 3117, 673, 2236, 3163, 679, 2267, 3205, 680, 2293, 3254, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g01ei"}, {320, 243, 496, 2670, 3256, 497, 2497, 3253, 496, 2657, 3222, 492, 2674, 3188, 487, 2808, 3267, 
			477, 2847, 3331, 464, 2836, 3333, 456, 2855, 3190, 443, 2903, 3151, 90, 2, 0, 18, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g01er"}, {297, 228, 556, 1687, 2002, 572, 1679, 2137, 566, 1645, 2051, 564, 1689, 2007, 554, 1681, 1999, 
			542, 1709, 2017, 518, 1718, 2019, 503, 1751, 2062, 509, 1882, 2156, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"g01ih"}, {233, 240, 495, 2574, 3362, 501, 2497, 3304, 495, 2550, 3387, 492, 2571, 3313, 496, 2571, 3297, 
			501, 2500, 3304, 535, 2462, 3257, 534, 2400, 3308, 526, 2435, 3371, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g01iy"}, {314, 222, 389, 3068, 3544, 382, 3006, 3507, 382, 3034, 3419, 372, 3022, 3496, 394, 3069, 3483, 
			392, 3059, 3495, 407, 3070, 3481, 403, 3033, 3491, 382, 3078, 3441, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g01oa"}, {263, 251, 571, 1243, 2854, 554, 1330, 2823, 584, 1255, 2896, 571, 1243, 2854, 542, 1246, 2868, 
			519, 1241, 2854, 504, 1164, 2893, 499, 1104, 2874, 499, 1179, 2824, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"g01oo"}, {224, 243, 499, 1440, 2980, 503, 1326, 3012, 499, 1442, 2981, 502, 1468, 3024, 543, 1528, 3038, 
			581, 1758, 3014, 615, 1825, 3071, 624, 1860, 3058, 616, 2073, 3143, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"g01uh"}, {200, 247, 740, 1502, 3001, 740, 1502, 3001, 734, 1496, 2996, 726, 1638, 3049, 714, 1717, 3050, 
			693, 1873, 3124, 690, 1926, 3127, 690, 2145, 3123, 683, 2192, 3143, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"g01uw"}, {327, 251, 500, 1724, 2988, 502, 1747, 2919, 499, 1753, 2985, 497, 1688, 2952, 492, 1638, 3051, 
			488, 1461, 3061, 481, 1411, 3073, 482, 1404, 3103, 491, 1364, 3095, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"g02ae"}, {283, 196, 729, 2878, 3792, 739, 2939, 3719, 729, 2878, 3792, 703, 2880, 3574, 689, 2912, 3597, 
			654, 2804, 3636, 863, 2538, 3684, 981, 2346, 3533, 1000, 2236, 3285, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"g02ah"}, {298, 208, 1312, 1820, 3308, 1291, 1787, 3337, 1333, 1853, 3231, 1340, 1852, 3247, 1366, 1850, 3243, 
			1364, 1915, 3319, 1290, 1935, 3327, 1179, 2004, 3386, 995, 2112, 3449, 95, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0}},
		{{"g02aw"}, {249, 217, 875, 1501, 3294, 872, 1444, 3263, 867, 1504, 3267, 841, 1497, 3266, 903, 1535, 3235, 
			955, 1604, 3183, 921, 1641, 3244, 937, 1804, 3198, 802, 1941, 3341, 55, 0, 0, 0, 0, 0, 7, 11, 0, 1, 0, 1, 0}},
		{{"g02eh"}, {218, 188, 928, 2254, 3612, 821, 2305, 3650, 899, 2259, 3638, 928, 2254, 3612, 894, 2250, 3458, 
			861, 2262, 3615, 915, 2281, 3646, 875, 2335, 3652, 858, 2364, 3659, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g02ei"}, {286, 211, 648, 2788, 3723, 673, 2731, 3692, 655, 2812, 3677, 642, 2815, 3731, 628, 2909, 3678, 
			623, 2991, 3755, 551, 3005, 3728, 512, 3150, 3720, 477, 3130, 3650, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g02er"}, {258, 223, 620, 1750, 2165, 504, 1831, 2420, 541, 1797, 2228, 561, 1751, 2197, 628, 1740, 2128, 
			634, 1778, 2105, 622, 1766, 2099, 567, 1747, 2116, 592, 1720, 2094, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"g02ih"}, {197, 258, 557, 2634, 3691, 557, 2634, 3691, 547, 2557, 3623, 564, 2579, 3611, 623, 2595, 3578, 
			684, 2547, 3612, 657, 2489, 3587, 627, 2504, 3635, 611, 2463, 3590, 95, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g02iy"}, {248, 246, 498, 3451, 4100, 497, 3346, 4065, 498, 3451, 4100, 493, 3402, 4113, 485, 3397, 3997, 
			444, 3388, 4040, 434, 3406, 4033, 417, 3423, 3997, 388, 3386, 4011, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g02oa"}, {295, 230, 683, 1038, 3200, 687, 1110, 3126, 683, 1038, 3200, 666, 1051, 3181, 639, 1061, 3145, 
			624, 1004, 3158, 590, 970, 3188, 566, 930, 3171, 562, 940, 3164, 95, 0, 0, 0, 1, 0, 0, 0, 19, 0, 0, 0, 0}},
		{{"g02oo"}, {206, 228, 639, 1518, 3253, 543, 1445, 3258, 624, 1497, 3248, 642, 1509, 3209, 641, 1504, 3253, 
			621, 1508, 3179, 601, 1520, 3047, 594, 1621, 3209, 588, 1796, 3246, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 0, 1, 0}},
		{{"g02uh"}, {215, 220, 882, 1623, 3434, 766, 1580, 3487, 888, 1632, 3390, 893, 1638, 3345, 846, 1688, 3411, 
			805, 1690, 3307, 770, 1820, 3304, 777, 1941, 3328, 712, 2064, 3571, 90, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 18, 0}},
		{{"g02uw"}, {236, 188, 435, 1059, 3389, 434, 1040, 3418, 435, 1059, 3389, 433, 1034, 3384, 428, 1035, 3362, 
			402, 919, 3395, 386, 892, 3438, 388, 1029, 3463, 413, 1179, 3340, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"g04ae"}, {385, 255, 932, 2523, 3644, 907, 2521, 3653, 905, 2512, 3704, 927, 2515, 3661, 976, 2496, 3703, 
			977, 2325, 3434, 1059, 2137, 3219, 1050, 2057, 3081, 949, 2100, 3110, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"g04ah"}, {324, 246, 1007, 1742, 2982, 830, 1690, 3098, 933, 1709, 3102, 960, 1747, 3023, 1047, 1744, 2964, 
			1036, 1742, 2965, 1002, 1773, 2978, 1073, 1811, 2931, 983, 1883, 2927, 95, 0, 0, 0, 0, 0, 19, 0, 1, 0, 0, 0, 0}},
		{{"g04aw"}, {430, 244, 885, 1415, 3196, 904, 1421, 3245, 930, 1306, 3246, 921, 1378, 3242, 885, 1415, 3196, 
			918, 1438, 3138, 931, 1558, 3130, 932, 1725, 3081, 888, 1953, 3311, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"g04eh"}, {277, 245, 802, 2427, 3488, 764, 2363, 3531, 774, 2419, 3400, 801, 2416, 3433, 801, 2394, 3432, 
			858, 2379, 3400, 853, 2390, 3462, 821, 2397, 3454, 763, 2437, 3425, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g04ei"}, {350, 266, 561, 2663, 3550, 563, 2615, 3555, 561, 2663, 3550, 557, 2640, 3530, 546, 2645, 3516, 
			517, 2747, 3488, 506, 2792, 3553, 501, 2770, 3510, 501, 2817, 3480, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g04er"}, {359, 285, 564, 2003, 2349, 575, 2019, 2376, 570, 2003, 2421, 565, 2001, 2353, 565, 2005, 2347, 
			564, 2015, 2408, 564, 2027, 2470, 568, 2024, 2529, 571, 2065, 2479, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"g04ih"}, {313, 271, 558, 2427, 3502, 583, 2595, 3520, 588, 2548, 3520, 560, 2495, 3517, 555, 2448, 3508, 
			556, 2432, 3488, 559, 2435, 3506, 556, 2414, 3529, 550, 2417, 3600, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g04iy"}, {324, 294, 463, 3225, 3699, 423, 3187, 3872, 429, 3194, 3794, 436, 3198, 3777, 443, 3214, 3733, 
			462, 3203, 3667, 464, 3202, 3745, 451, 3193, 3800, 449, 3191, 3796, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g04oa"}, {380, 282, 583, 1352, 3140, 601, 1362, 3082, 621, 1301, 3176, 614, 1253, 3185, 595, 1350, 3131, 
			580, 1366, 3130, 568, 1318, 3127, 566, 1366, 3129, 567, 1431, 3156, 95, 0, 0, 0, 0, 0, 0, 0, 19, 0, 1, 0, 0}},
		{{"g04oo"}, {367, 271, 564, 1613, 3065, 581, 1550, 3177, 575, 1532, 3128, 564, 1613, 3065, 573, 1630, 3093, 
			615, 1826, 3160, 598, 1941, 3150, 570, 2090, 3211, 557, 2254, 3309, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"g04uh"}, {275, 268, 784, 1614, 3392, 746, 1517, 3388, 768, 1616, 3392, 765, 1586, 3389, 763, 1557, 3388, 
			758, 1556, 3384, 776, 1748, 3385, 756, 1943, 3350, 742, 2057, 3355, 90, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 18, 0}},
		{{"g04uw"}, {300, 288, 556, 1736, 3005, 555, 1749, 2874, 556, 1736, 3005, 556, 1785, 3050, 555, 1812, 3041, 
			555, 1725, 3036, 557, 1636, 2933, 558, 1608, 2842, 564, 1755, 2877, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19, 0, 0}},
		{{"g05ae"}, {456, 227, 682, 2638, 3510, 653, 2557, 3491, 678, 2616, 3481, 681, 2642, 3507, 677, 2634, 3492, 
			723, 2494, 3459, 843, 2272, 3363, 843, 2162, 3188, 868, 2112, 3159, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"g05ah"}, {375, 204, 978, 1817, 2843, 1022, 1754, 2914, 1030, 1758, 2864, 1000, 1783, 2849, 993, 1815, 2859, 
			978, 1818, 2841, 955, 1819, 2826, 956, 1852, 2850, 935, 1901, 2800, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"g05aw"}, {368, 210, 882, 1309, 2834, 945, 1347, 2777, 901, 1284, 2836, 906, 1313, 2848, 912, 1350, 2808, 
			928, 1402, 2795, 944, 1568, 2759, 919, 1709, 2767, 874, 1952, 2882, 60, 0, 0, 0, 0, 0, 8, 12, 0, 0, 0, 0, 0}},
		{{"g05eh"}, {328, 226, 809, 2265, 3309, 762, 2337, 3355, 758, 2323, 3335, 808, 2294, 3321, 809, 2251, 3286, 
			800, 2241, 3230, 802, 2212, 3161, 796, 2244, 3155, 770, 2247, 3137, 95, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0}},
		{{"g05ei"}, {321, 220, 638, 2553, 3348, 632, 2526, 3361, 623, 2570, 3370, 589, 2646, 3360, 504, 2726, 3340, 
			453, 2725, 3306, 436, 2713, 3323, 430, 2700, 3302, 423, 2637, 3407, 95, 0, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0}},
		{{"g05er"}, {348, 223, 601, 1791, 2612, 613, 1793, 2653, 601, 1791, 2612, 587, 1781, 2609, 569, 1780, 2581, 
			575, 1823, 2550, 602, 1871, 2539, 572, 1890, 2587, 564, 1990, 2634, 70, 0, 0, 1, 0, 0, 0, 0, 0, 3, 0, 2, 14}},
		{{"g05ih"}, {323, 219, 441, 2728, 3285, 441, 2745, 3304, 441, 2706, 3285, 452, 2673, 3242, 550, 2569, 3249, 
			602, 2515, 3267, 615, 2477, 3287, 616, 2449, 3323, 583, 2472, 3353, 90, 0, 18, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g05iy"}, {305, 239, 475, 3320, 0, 455, 3314, 0, 475, 3320, 0, 471, 3310, 0, 449, 3317, 0, 
			447, 3322, 0, 440, 3337, 0, 438, 3292, 0, 436, 3141, 3341, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g05oa"}, {380, 239, 678, 1306, 3202, 688, 1367, 3176, 670, 1321, 3234, 649, 1299, 3181, 619, 1203, 3177, 
			573, 1165, 3223, 519, 1128, 3260, 513, 1099, 3285, 476, 1192, 3324, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"g05oo"}, {375, 225, 504, 1440, 3000, 517, 1476, 3284, 506, 1499, 3023, 559, 1521, 2976, 614, 1563, 2933, 
			633, 1642, 2913, 628, 1816, 2926, 628, 1949, 2942, 614, 2164, 3098, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"g05uh"}, {290, 225, 897, 1494, 2960, 904, 1497, 3031, 893, 1492, 2960, 880, 1537, 3002, 865, 1555, 2993, 
			844, 1688, 3057, 823, 1824, 3071, 804, 1986, 3230, 756, 2131, 3330, 55, 0, 0, 0, 0, 0, 7, 2, 0, 0, 0, 11, 0}},
		{{"g05uw"}, {378, 219, 437, 1733, 3208, 450, 1600, 3248, 438, 1621, 3177, 437, 1705, 3242, 437, 1706, 3226, 
			437, 1671, 3175, 435, 1586, 3214, 434, 1609, 3158, 429, 1614, 3188, 75, 0, 0, 0, 0, 0, 0, 0, 0, 5, 15, 0, 0}},
		{{"g06ae"}, {292, 222, 799, 2397, 3125, 716, 2382, 3115, 804, 2410, 3126, 808, 2414, 3122, 867, 2442, 3077, 
			867, 2441, 3052, 894, 2074, 3091, 891, 2041, 3126, 886, 2023, 3152, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"g06ah"}, {275, 225, 1026, 1544, 2536, 1083, 1580, 2552, 1025, 1558, 2532, 1026, 1544, 2536, 1047, 1539, 2578, 
			1014, 1521, 2600, 1047, 1582, 2624, 1036, 1692, 2716, 914, 1736, 3063, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"g06aw"}, {300, 224, 888, 1266, 2632, 954, 1337, 2975, 887, 1252, 2712, 892, 1262, 2652, 895, 1269, 2594, 
			898, 1287, 2564, 942, 1316, 2613, 958, 1491, 2634, 880, 1517, 2609, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"g06eh"}, {174, 222, 660, 2223, 3354, 651, 2259, 3360, 651, 2247, 3386, 655, 2239, 3367, 665, 2234, 3342, 
			647, 2212, 3360, 637, 2212, 3362, 633, 2229, 3363, 633, 2234, 3367, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g06ei"}, {261, 225, 542, 2755, 3283, 554, 2643, 3310, 554, 2759, 3399, 555, 2762, 3283, 522, 2740, 3306, 
			480, 2932, 3354, 449, 2947, 3390, 442, 2943, 3426, 440, 2940, 3462, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g06er"}, {299, 224, 632, 1689, 2154, 577, 1710, 2192, 585, 1636, 2199, 623, 1667, 2169, 635, 1718, 2110, 
			631, 1705, 2094, 611, 1681, 2072, 591, 1715, 2107, 574, 1765, 2225, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"g06ih"}, {150, 236, 488, 2623, 3549, 488, 2579, 3541, 479, 2579, 3481, 496, 2613, 3475, 513, 2600, 3472, 
			556, 2563, 3450, 584, 2538, 3467, 597, 2514, 3506, 591, 2503, 3509, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g06iy"}, {230, 250, 478, 2997, 3755, 478, 2997, 3741, 481, 2999, 3749, 484, 2991, 3773, 485, 2991, 3806, 
			488, 3020, 3833, 489, 2973, 3771, 493, 3011, 3709, 496, 2973, 3647, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g06oa"}, {268, 226, 483, 903, 2954, 558, 1179, 3077, 576, 1095, 2931, 523, 951, 2954, 503, 913, 2984, 
			492, 906, 2977, 463, 871, 2951, 459, 876, 2972, 460, 1043, 2979, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"g06oo"}, {234, 254, 545, 1307, 3032, 504, 1302, 3028, 547, 1317, 3032, 571, 1387, 3008, 606, 1471, 2990, 
			658, 1537, 2941, 703, 1728, 2920, 706, 1870, 2935, 686, 2011, 3031, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"g06uh"}, {218, 238, 754, 1430, 3034, 754, 1430, 3034, 863, 1412, 2983, 870, 1413, 2977, 869, 1532, 2975, 
			820, 1561, 2965, 812, 1662, 2963, 780, 1751, 2987, 670, 1887, 3025, 90, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 18, 0}},
		{{"g06uw"}, {246, 256, 480, 1011, 2574, 499, 1265, 2914, 497, 1143, 2808, 492, 1052, 2698, 490, 1018, 2658, 
			482, 1011, 2585, 476, 1015, 2559, 469, 1034, 2565, 469, 1094, 2597, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"g07ae"}, {341, 242, 722, 2418, 3017, 732, 2414, 3010, 701, 2412, 3021, 701, 2369, 2945, 692, 2360, 2979, 
			683, 2355, 2963, 681, 2146, 2976, 733, 2173, 2936, 552, 2211, 3007, 85, 0, 0, 0, 3, 17, 0, 0, 0, 0, 0, 0, 0}},
		{{"g07ah"}, {384, 227, 843, 1803, 2562, 799, 1748, 2582, 872, 1782, 2583, 842, 1803, 2582, 850, 1800, 2634, 
			860, 1817, 2636, 824, 1834, 2626, 793, 1864, 2669, 704, 1888, 2719, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"g07aw"}, {342, 233, 697, 1358, 2263, 772, 1255, 2180, 744, 1293, 2238, 712, 1303, 2276, 697, 1348, 2321, 
			696, 1357, 2271, 706, 1366, 2289, 689, 1377, 2290, 685, 1420, 2269, 40, 0, 0, 0, 0, 0, 11, 8, 0, 0, 0, 1, 0}},
		{{"g07eh"}, {284, 230, 649, 2268, 2827, 540, 2219, 3008, 569, 2218, 2910, 613, 2248, 2845, 635, 2260, 2836, 
			649, 2269, 2861, 618, 2270, 2863, 576, 2274, 2902, 520, 2277, 2965, 80, 0, 4, 0, 16, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g07ei"}, {421, 242, 544, 2444, 2605, 596, 2278, 2833, 627, 2403, 2846, 572, 2454, 2676, 516, 2426, 2681, 
			501, 2474, 2683, 496, 2442, 2731, 495, 2486, 2737, 501, 2521, 2876, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g07er"}, {365, 237, 585, 1694, 2278, 582, 1673, 2259, 592, 1688, 2259, 553, 1707, 2274, 517, 1726, 2230, 
			511, 1789, 2206, 502, 1798, 2171, 498, 1838, 2151, 493, 1867, 2152, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"g07ih"}, {311, 252, 506, 2432, 3262, 516, 2513, 3257, 501, 2401, 3261, 500, 2446, 3227, 499, 2449, 3219, 
			498, 2437, 3215, 498, 2429, 3206, 498, 2396, 3185, 500, 2389, 3230, 95, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g07iy"}, {386, 250, 457, 2776, 4268, 460, 2683, 4267, 456, 2740, 4257, 451, 2772, 4273, 464, 2793, 4259, 
			456, 2941, 4351, 454, 2945, 4354, 460, 2959, 4321, 465, 2953, 4337, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g07oa"}, {442, 241, 492, 927, 2671, 627, 1193, 2727, 628, 1181, 2801, 521, 1171, 2750, 516, 923, 2805, 
			499, 922, 2751, 492, 927, 2671, 497, 943, 2677, 497, 936, 2601, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"g07oo"}, {186, 249, 507, 1720, 2978, 507, 1720, 2978, 502, 1720, 2950, 500, 1697, 2919, 494, 1682, 2812, 
			493, 1725, 2817, 504, 1795, 2819, 561, 1853, 2764, 565, 1895, 2778, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 0, 0, 1}},
		{{"g07uh"}, {286, 234, 620, 1579, 2861, 620, 1579, 2861, 646, 1572, 2902, 641, 1635, 2891, 642, 1661, 2873, 
			625, 1752, 2870, 632, 1785, 2882, 629, 1917, 2846, 600, 1978, 2882, 90, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 18, 0}},
		{{"g07uw"}, {367, 255, 503, 821, 2874, 519, 788, 2935, 504, 815, 2856, 504, 846, 2882, 503, 871, 2905, 
			502, 902, 2954, 500, 927, 2955, 498, 958, 2986, 500, 984, 2836, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"g08ae"}, {384, 266, 695, 2704, 3731, 698, 2814, 3662, 702, 2702, 3693, 712, 2679, 3655, 746, 2635, 3686, 
			761, 2550, 3655, 764, 2415, 3544, 767, 2254, 3441, 777, 2129, 3495, 85, 0, 0, 0, 3, 17, 0, 0, 0, 0, 0, 0, 0}},
		{{"g08ah"}, {407, 249, 993, 2000, 3174, 1033, 2054, 3129, 998, 2005, 3198, 984, 1988, 3229, 978, 1982, 3240, 
			971, 2070, 3229, 956, 2085, 3241, 937, 1980, 3210, 928, 1995, 3257, 90, 0, 0, 0, 0, 2, 18, 0, 0, 0, 0, 0, 0}},
		{{"g08aw"}, {377, 270, 787, 1234, 2924, 778, 1143, 2840, 794, 1151, 2867, 784, 1207, 2921, 793, 1263, 2930, 
			791, 1220, 2940, 804, 1233, 2936, 812, 1281, 2901, 811, 1561, 2937, 80, 0, 0, 0, 0, 0, 2, 16, 0, 0, 0, 1, 1}},
		{{"g08eh"}, {294, 244, 741, 2418, 3665, 742, 2407, 3611, 725, 2391, 3624, 711, 2384, 3536, 698, 2365, 3642, 
			692, 2298, 3477, 695, 2281, 3543, 691, 2332, 3520, 689, 2321, 3224, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g08ei"}, {438, 257, 513, 2881, 3667, 553, 2862, 3646, 545, 2703, 3660, 528, 2732, 3580, 517, 2766, 3639, 
			508, 2887, 3681, 504, 2917, 3674, 503, 2940, 3565, 502, 2968, 3557, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g08er"}, {346, 260, 540, 1571, 0, 550, 1869, 2091, 542, 1602, 0, 546, 1603, 0, 529, 1575, 0, 
			527, 1572, 0, 545, 1607, 0, 546, 1628, 1913, 546, 1644, 1973, 95, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19}},
		{{"g08ih"}, {378, 261, 517, 2619, 3499, 512, 2689, 3500, 517, 2619, 3499, 519, 2578, 3522, 526, 2557, 3608, 
			527, 2562, 3473, 533, 2522, 3381, 548, 2512, 3352, 542, 2555, 3471, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g08iy"}, {390, 263, 492, 3236, 3720, 455, 3204, 3792, 480, 3085, 3622, 486, 3274, 3665, 491, 3179, 3658, 
			489, 3092, 3682, 488, 3131, 3734, 494, 3148, 3685, 496, 3235, 3551, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g08oa"}, {309, 259, 536, 789, 2863, 589, 859, 2789, 528, 791, 2840, 522, 792, 2768, 515, 788, 2795, 
			524, 912, 2891, 516, 939, 2886, 509, 986, 2846, 507, 993, 2821, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"g08oo"}, {292, 271, 585, 1636, 2941, 570, 1766, 2988, 625, 1611, 2944, 623, 1621, 2980, 609, 1668, 3014, 
			591, 1674, 3005, 592, 1829, 2988, 684, 1919, 2968, 584, 2156, 2981, 85, 0, 0, 0, 0, 0, 1, 0, 0, 17, 0, 1, 1}},
		{{"g08uh"}, {303, 242, 695, 1832, 3000, 730, 1677, 3005, 709, 1858, 3032, 698, 1780, 3019, 678, 1779, 3170, 
			697, 1829, 3065, 702, 1830, 3050, 748, 1796, 3019, 727, 2015, 2946, 70, 0, 0, 0, 0, 0, 1, 4, 0, 1, 0, 14, 0}},
		{{"g08uw"}, {315, 271, 540, 1653, 2799, 542, 1634, 2842, 540, 1653, 2799, 540, 1661, 2756, 534, 1661, 2709, 
			534, 1701, 2701, 535, 1720, 2761, 531, 1719, 2723, 520, 1606, 2913, 95, 0, 0, 0, 0, 0, 0, 0, 1, 0, 19, 0, 0}},
		{{"g09ae"}, {382, 265, 758, 2373, 3327, 761, 2439, 3278, 757, 2406, 3363, 755, 2378, 3326, 758, 2373, 3327, 
			758, 2358, 3313, 769, 2295, 3274, 863, 2203, 3278, 850, 2122, 3223, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"g09ah"}, {321, 245, 1197, 1734, 3187, 1218, 1783, 3207, 1180, 1637, 3243, 1150, 1665, 3261, 1123, 1685, 3250, 
			1127, 1717, 3217, 1109, 1738, 3164, 1079, 1764, 3106, 1055, 1858, 3144, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"g09aw"}, {425, 301, 932, 1445, 3107, 918, 1432, 3041, 932, 1445, 3107, 943, 1434, 3131, 946, 1425, 3163, 
			945, 1422, 3208, 970, 1400, 3258, 936, 1453, 3231, 959, 1502, 3113, 80, 0, 0, 0, 0, 0, 4, 16, 0, 0, 0, 0, 0}},
		{{"g09eh"}, {292, 238, 763, 2393, 3241, 715, 2497, 3404, 750, 2443, 3226, 758, 2421, 3212, 763, 2393, 3241, 
			758, 2388, 3245, 754, 2385, 3246, 742, 2381, 3323, 718, 2376, 3350, 85, 0, 0, 0, 17, 3, 0, 0, 0, 0, 0, 0, 0}},
		{{"g09ei"}, {364, 258, 536, 2859, 3507, 561, 2764, 3478, 554, 2855, 3523, 532, 2877, 3501, 515, 2828, 3451, 
			508, 2851, 0, 511, 2879, 0, 520, 2926, 0, 534, 2935, 0, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g09er"}, {413, 286, 573, 1748, 2245, 578, 1755, 2231, 573, 1748, 2246, 585, 1760, 2219, 582, 1757, 2252, 
			584, 1785, 2320, 570, 1935, 2351, 565, 1982, 2404, 568, 2022, 2420, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"g09ih"}, {308, 264, 515, 2828, 3627, 515, 2827, 3606, 528, 2795, 3508, 529, 2625, 3510, 519, 2575, 3574, 
			548, 2562, 3559, 561, 2561, 3583, 586, 2533, 3552, 605, 2494, 3561, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g09iy"}, {399, 294, 503, 3256, 4056, 438, 3270, 4151, 489, 3265, 4076, 505, 3253, 4052, 518, 3242, 4014, 
			519, 3313, 4028, 537, 3207, 3993, 541, 3204, 3969, 553, 3136, 3939, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g09oa"}, {423, 275, 561, 1151, 2996, 565, 1214, 3064, 565, 1155, 2990, 561, 1144, 2990, 560, 1118, 2992, 
			558, 1107, 2979, 560, 1067, 2927, 563, 1114, 2940, 563, 1114, 2980, 75, 0, 0, 0, 0, 0, 0, 0, 15, 2, 3, 0, 0}},
		{{"g09oo"}, {284, 294, 595, 1737, 3235, 569, 1699, 3239, 603, 1693, 3228, 610, 1698, 3227, 620, 1752, 3251, 
			622, 1761, 3273, 627, 1794, 3316, 639, 1895, 3347, 639, 2068, 3361, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"g09uh"}, {248, 262, 756, 1579, 3388, 732, 1617, 3493, 756, 1588, 3461, 756, 1577, 3344, 761, 1647, 3364, 
			761, 1739, 3358, 777, 1802, 3370, 793, 1880, 3358, 758, 1948, 3531, 90, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 18, 0}},
		{{"g09uw"}, {307, 303, 568, 1508, 2987, 570, 1505, 3050, 571, 1490, 2979, 573, 1495, 2946, 573, 1497, 2977, 
			567, 1503, 2976, 568, 1508, 2987, 564, 1507, 3000, 573, 1536, 3003, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"g10ae"}, {311, 237, 946, 2540, 3229, 952, 2647, 3300, 938, 2561, 3205, 902, 2545, 3111, 871, 2468, 3182, 
			887, 2566, 3147, 915, 2488, 3132, 934, 2335, 3114, 878, 2279, 3172, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"g10ah"}, {282, 218, 855, 1845, 3055, 910, 1759, 3211, 855, 1845, 3055, 858, 1770, 3152, 1038, 1876, 3023, 
			1003, 1933, 3009, 1012, 2045, 3000, 970, 2064, 2981, 961, 2078, 2875, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"g10aw"}, {285, 226, 893, 1563, 3131, 911, 1570, 3052, 916, 1566, 3115, 893, 1562, 3158, 886, 1512, 3150, 
			898, 1492, 3144, 974, 1562, 3080, 902, 1674, 2862, 872, 1856, 2733, 85, 0, 0, 0, 0, 0, 3, 17, 0, 0, 0, 0, 0}},
		{{"g10eh"}, {278, 227, 816, 2404, 3448, 794, 2431, 3580, 812, 2443, 3514, 821, 2413, 3427, 828, 2434, 3445, 
			817, 2484, 3443, 805, 2520, 3514, 767, 2525, 3483, 676, 2514, 3487, 95, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0}},
		{{"g10ei"}, {310, 236, 740, 2651, 3049, 744, 2655, 2885, 732, 2700, 3080, 682, 2694, 3039, 639, 2665, 3009, 
			635, 2672, 3041, 628, 2569, 3032, 621, 2600, 3114, 623, 2520, 3053, 90, 0, 0, 18, 1, 0, 0, 1, 0, 0, 0, 0, 0}},
		{{"g10er"}, {320, 212, 631, 1722, 2140, 571, 1628, 2312, 625, 1639, 2260, 708, 1697, 2204, 681, 1660, 2189, 
			641, 1662, 2161, 630, 1721, 2128, 624, 1731, 2137, 621, 1830, 2230, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"g10ih"}, {241, 257, 518, 2870, 3681, 531, 2856, 3677, 515, 2847, 3686, 510, 2791, 3695, 527, 2811, 3705, 
			637, 2765, 3721, 630, 2765, 3668, 627, 2752, 3602, 625, 2696, 3699, 90, 0, 18, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g10iy"}, {292, 222, 456, 3389, 4027, 496, 3273, 4107, 485, 3257, 3994, 478, 3321, 4025, 467, 3283, 4013, 
			453, 3389, 4065, 440, 3316, 4059, 435, 3386, 4076, 434, 3253, 4013, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g10oa"}, {338, 238, 630, 1491, 3010, 585, 1427, 2992, 645, 1452, 3018, 676, 1367, 2977, 633, 1307, 2959, 
			630, 1240, 2985, 615, 1166, 3055, 591, 982, 3085, 571, 984, 3130, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"g10oo"}, {268, 229, 667, 1446, 3220, 513, 1435, 3235, 671, 1469, 3219, 653, 1549, 3216, 623, 1507, 3121, 
			582, 1621, 3138, 572, 1701, 3134, 573, 1815, 3163, 587, 2066, 3155, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"g10uh"}, {222, 229, 880, 1680, 3280, 873, 1680, 3399, 880, 1646, 3186, 864, 1655, 3209, 839, 1686, 3274, 
			814, 1667, 3285, 811, 1922, 3189, 805, 2068, 3161, 758, 2351, 3278, 95, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 19, 0}},
		{{"g10uw"}, {318, 216, 460, 1422, 3099, 500, 1523, 3046, 492, 1444, 3100, 451, 1427, 3116, 434, 1480, 3134, 
			435, 1456, 3041, 416, 1363, 3041, 408, 1366, 3068, 427, 1370, 3119, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"g11ae"}, {353, 230, 648, 2334, 2834, 683, 2348, 2893, 648, 2334, 2834, 681, 2363, 2848, 827, 2331, 2860, 
			864, 2213, 2842, 855, 2133, 2802, 861, 2054, 2819, 881, 2032, 2911, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"g11ah"}, {298, 240, 1047, 1488, 2879, 948, 1409, 2859, 1062, 1514, 2846, 1069, 1529, 2857, 1050, 1633, 2823, 
			1066, 1575, 2893, 1052, 1701, 2951, 986, 1802, 3033, 897, 1916, 3082, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"g11aw"}, {380, 232, 898, 1359, 2850, 880, 1368, 2841, 896, 1350, 2850, 897, 1359, 2846, 895, 1371, 2854, 
			897, 1430, 2945, 884, 1615, 3064, 881, 1842, 3155, 844, 2044, 3287, 45, 0, 0, 0, 0, 1, 10, 9, 0, 0, 0, 0, 0}},
		{{"g11eh"}, {227, 226, 793, 2228, 3560, 578, 2252, 3415, 782, 2227, 3442, 773, 2235, 3462, 760, 2249, 3657, 
			748, 2243, 3664, 732, 2258, 3778, 692, 2247, 3799, 641, 2262, 3896, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g11ei"}, {315, 240, 494, 2676, 3280, 497, 2676, 3290, 493, 2680, 3230, 492, 2686, 3197, 485, 2676, 3159, 
			478, 2664, 3065, 475, 2659, 3075, 482, 2769, 3092, 489, 2738, 3127, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g11er"}, {296, 233, 671, 1556, 2302, 670, 1486, 2365, 671, 1563, 2263, 646, 1564, 2240, 608, 1563, 2159, 
			609, 1568, 2170, 562, 1564, 2134, 548, 1589, 2118, 509, 1675, 2122, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"g11ih"}, {250, 226, 499, 2437, 3758, 499, 2440, 3626, 496, 2419, 3772, 495, 2389, 3789, 502, 2409, 3785, 
			499, 2388, 3826, 499, 2427, 3770, 496, 2419, 3755, 495, 2395, 3733, 90, 0, 18, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g11iy"}, {282, 238, 465, 2875, 3699, 476, 2812, 3724, 456, 2779, 3669, 463, 2874, 3684, 462, 2870, 3647, 
			448, 2837, 3557, 445, 2900, 3288, 445, 2882, 3381, 453, 2892, 3487, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g11oa"}, {363, 232, 455, 923, 3258, 527, 1191, 3202, 496, 1032, 3234, 473, 922, 3270, 454, 920, 3264, 
			455, 927, 3291, 468, 931, 3326, 466, 930, 3269, 475, 933, 3291, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"g11oo"}, {310, 219, 627, 1515, 3034, 565, 1429, 3125, 595, 1473, 3096, 625, 1511, 3037, 627, 1536, 3028, 
			629, 1581, 2996, 629, 1746, 2940, 630, 1901, 2912, 631, 2049, 2886, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"g11uh"}, {227, 226, 701, 1563, 3058, 672, 1490, 3167, 691, 1551, 3053, 699, 1568, 3058, 701, 1563, 3058, 
			698, 1559, 3001, 645, 1651, 2962, 621, 1802, 2927, 619, 2013, 2884, 95, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 19, 0}},
		{{"g11uw"}, {324, 238, 460, 1291, 3010, 488, 1289, 2893, 476, 1288, 2917, 473, 1295, 2936, 460, 1274, 2945, 
			460, 1295, 3014, 456, 1281, 3033, 448, 1205, 3040, 456, 1243, 3000, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19, 0, 0}},
		{{"g12ae"}, {322, 240, 730, 2668, 3564, 801, 2707, 3537, 753, 2680, 3559, 723, 2622, 3559, 807, 2534, 3459, 
			876, 2465, 3251, 873, 2437, 3076, 903, 2327, 3062, 899, 2284, 3112, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"g12ah"}, {317, 200, 1154, 1932, 3044, 1155, 1897, 3257, 1103, 1885, 3192, 1073, 1879, 3137, 1129, 1890, 3056, 
			1159, 1945, 2974, 1080, 1901, 3075, 1006, 1963, 3063, 1011, 2151, 3116, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"g12aw"}, {323, 241, 952, 1665, 3063, 872, 1464, 3098, 883, 1462, 3137, 936, 1502, 3195, 952, 1621, 3099, 
			954, 1655, 3054, 957, 1691, 3024, 957, 1814, 2949, 947, 2004, 3105, 80, 0, 0, 0, 0, 0, 4, 16, 0, 0, 0, 0, 0}},
		{{"g12eh"}, {250, 236, 762, 2349, 3484, 874, 2281, 3490, 828, 2282, 3531, 776, 2280, 3549, 760, 2256, 3518, 
			774, 2349, 3477, 731, 2368, 3487, 741, 2376, 3430, 742, 2376, 3487, 100, 0, 0, 0, 19, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g12ei"}, {293, 262, 594, 2733, 3413, 612, 2664, 3379, 584, 2763, 3487, 606, 2782, 3427, 611, 2817, 3476, 
			506, 2884, 3496, 499, 2971, 3456, 490, 3027, 3510, 491, 3070, 3524, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g12er"}, {270, 253, 627, 1780, 2415, 577, 1854, 2502, 636, 1806, 2466, 633, 1789, 2423, 607, 1785, 2333, 
			580, 1799, 2334, 558, 1818, 2325, 551, 1871, 2362, 550, 1927, 2417, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"g12ih"}, {240, 242, 533, 2762, 3466, 562, 2814, 3463, 533, 2762, 3466, 556, 2716, 3466, 618, 2746, 3474, 
			671, 2669, 3482, 675, 2589, 3461, 660, 2563, 3439, 642, 2518, 3466, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g12iy"}, {304, 212, 436, 3340, 3862, 348, 3227, 3913, 422, 3312, 3983, 535, 3340, 3954, 500, 3342, 3939, 
			447, 3351, 3881, 427, 3353, 3845, 413, 3369, 3825, 420, 3295, 3898, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g12oa"}, {237, 225, 637, 1142, 3360, 587, 1322, 3305, 706, 1287, 3333, 682, 1216, 3308, 669, 1166, 3341, 
			637, 1142, 3360, 634, 1167, 3379, 567, 1263, 3381, 510, 1440, 3342, 80, 0, 0, 0, 0, 0, 0, 0, 16, 3, 0, 0, 1}},
		{{"g12oo"}, {214, 259, 571, 1819, 3274, 560, 1869, 3225, 566, 1826, 3307, 571, 1819, 3274, 605, 1813, 3278, 
			625, 1882, 3290, 630, 1956, 3317, 621, 2058, 3303, 586, 2087, 3330, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"g12uh"}, {226, 248, 796, 1669, 3254, 812, 1664, 3270, 792, 1676, 3238, 822, 1682, 3257, 844, 1754, 3211, 
			824, 1811, 3133, 752, 1884, 3148, 706, 1967, 3194, 690, 2094, 3258, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 19, 0}},
		{{"g12uw"}, {273, 276, 540, 1998, 3104, 540, 1998, 3104, 520, 2036, 3150, 511, 2008, 3138, 503, 1998, 3094, 
			503, 1950, 3102, 505, 1924, 3099, 512, 1943, 3134, 537, 1972, 3163, 80, 0, 0, 0, 0, 0, 0, 0, 1, 3, 16, 0, 0}},
		{{"g13ae"}, {301, 251, 752, 2501, 0, 723, 2430, 0, 750, 2509, 0, 749, 2490, 0, 761, 2446, 0, 
			832, 2383, 0, 926, 2292, 0, 915, 2145, 0, 894, 2159, 0, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"g13ah"}, {334, 241, 931, 1749, 3205, 1009, 1768, 3298, 987, 1727, 3322, 960, 1799, 3299, 939, 1776, 3281, 
			931, 1749, 3205, 947, 1786, 3225, 926, 1830, 3221, 892, 1928, 3342, 90, 0, 0, 0, 0, 0, 18, 0, 0, 0, 0, 2, 0}},
		{{"g13aw"}, {339, 234, 823, 1356, 3419, 770, 1299, 3411, 796, 1278, 3460, 825, 1298, 3432, 823, 1356, 3419, 
			837, 1359, 3420, 878, 1407, 3407, 880, 1566, 3306, 871, 1852, 3271, 95, 0, 0, 0, 0, 0, 0, 19, 0, 0, 0, 1, 0}},
		{{"g13eh"}, {250, 263, 759, 2488, 3502, 792, 2460, 3496, 759, 2488, 3502, 753, 2490, 3524, 750, 2503, 3553, 
			747, 2494, 3555, 746, 2489, 3591, 743, 2476, 3615, 716, 2443, 3623, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g13ei"}, {345, 258, 550, 2780, 3468, 565, 2680, 3477, 555, 2746, 3375, 522, 2788, 3474, 511, 2828, 3380, 
			502, 2906, 3435, 496, 2999, 3479, 489, 3086, 3530, 500, 3021, 3563, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g13er"}, {288, 273, 561, 1692, 2358, 561, 1692, 2358, 555, 1618, 2361, 553, 1588, 2320, 616, 1556, 2286, 
			635, 1643, 2325, 632, 1678, 2374, 581, 1698, 2359, 559, 1730, 2370, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 19}},
		{{"g13ih"}, {273, 257, 508, 2642, 3494, 560, 2665, 3489, 505, 2642, 3533, 512, 2623, 3579, 518, 2591, 3580, 
			515, 2551, 3555, 517, 2522, 3551, 523, 2434, 3544, 519, 2405, 3552, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g13iy"}, {396, 278, 464, 3104, 3692, 337, 3314, 3952, 323, 3247, 3930, 397, 3124, 3955, 440, 3086, 3856, 
			463, 3070, 3673, 475, 3092, 3690, 479, 3118, 3636, 503, 3097, 3656, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g13oa"}, {351, 251, 502, 960, 3274, 556, 1025, 3261, 515, 937, 3321, 505, 960, 3307, 502, 972, 3282, 
			499, 947, 3328, 495, 933, 3346, 488, 938, 3336, 497, 971, 3349, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"g13oo"}, {269, 279, 563, 1577, 3159, 571, 1433, 3178, 563, 1468, 3150, 560, 1626, 3207, 559, 1574, 3224, 
			585, 1627, 3262, 640, 1777, 3238, 651, 1976, 3283, 642, 2132, 3301, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"g13uh"}, {232, 229, 738, 1626, 3315, 683, 1569, 3392, 731, 1548, 3343, 677, 1641, 3312, 800, 1651, 3316, 
			812, 1749, 3320, 806, 1832, 3324, 781, 1972, 3328, 670, 2064, 3382, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"g13uw"}, {286, 273, 527, 1407, 3253, 464, 1536, 3246, 556, 1468, 3177, 559, 1463, 3166, 551, 1438, 3269, 
			547, 1419, 3257, 520, 1394, 3242, 521, 1375, 3241, 522, 1357, 3241, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"g14ae"}, {306, 194, 717, 2313, 3150, 687, 2321, 3154, 717, 2313, 3150, 736, 2313, 3111, 747, 2310, 3073, 
			753, 2277, 3150, 755, 2242, 3158, 756, 2220, 3147, 755, 2159, 3170, 90, 0, 0, 0, 2, 18, 0, 0, 0, 0, 0, 0, 0}},
		{{"g14ah"}, {308, 189, 954, 1472, 2844, 971, 1366, 2876, 972, 1395, 2898, 963, 1456, 2875, 954, 1472, 2844, 
			971, 1490, 2890, 984, 1514, 2891, 963, 1542, 2888, 933, 1640, 2781, 50, 0, 0, 0, 0, 0, 10, 9, 0, 0, 0, 1, 0}},
		{{"g14aw"}, {332, 214, 854, 1109, 2960, 842, 1208, 2929, 861, 1164, 2942, 859, 1103, 2954, 853, 1111, 2948, 
			856, 1141, 2913, 853, 1182, 2940, 850, 1196, 2997, 828, 1295, 2974, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"g14eh"}, {297, 211, 621, 2275, 3146, 594, 2329, 3153, 622, 2258, 3165, 621, 2254, 3159, 614, 2229, 3165, 
			606, 2224, 3133, 614, 2195, 3120, 616, 2188, 3105, 616, 2179, 3127, 35, 0, 13, 0, 7, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g14ei"}, {351, 222, 564, 2455, 3100, 553, 2460, 3114, 529, 2446, 3080, 530, 2560, 0, 518, 2604, 0, 
			475, 2630, 3055, 449, 2678, 3023, 429, 2745, 3093, 422, 2725, 3105, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g14er"}, {312, 215, 603, 1663, 1983, 643, 1564, 2101, 603, 1663, 1983, 588, 1659, 1954, 542, 1641, 1904, 
			529, 1621, 1928, 498, 1603, 1856, 502, 1577, 1856, 476, 1564, 1864, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"g14ih"}, {311, 216, 435, 2623, 3084, 435, 2623, 3084, 426, 2519, 3033, 420, 2509, 2979, 433, 2450, 2925, 
			487, 2347, 2927, 537, 2293, 2875, 557, 2235, 2848, 562, 2187, 2833, 90, 0, 18, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0}},
		{{"g14iy"}, {320, 220, 432, 2688, 3161, 451, 2613, 3011, 441, 2758, 3075, 436, 2692, 3124, 431, 2678, 3148, 
			430, 2729, 3134, 427, 2769, 3138, 419, 2744, 3142, 413, 2763, 3149, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g14oa"}, {330, 240, 499, 992, 3121, 683, 1096, 2990, 671, 1083, 3010, 635, 1092, 3028, 620, 1034, 3068, 
			587, 1024, 3070, 559, 1011, 3075, 513, 1000, 3080, 497, 992, 3138, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"g14oo"}, {314, 245, 498, 1491, 2924, 498, 1491, 2924, 493, 1456, 3037, 489, 1436, 3048, 473, 1423, 3041, 
			450, 1377, 2958, 445, 1359, 2911, 450, 1599, 2881, 459, 1805, 2887, 45, 0, 0, 0, 0, 0, 0, 0, 0, 9, 11, 0, 0}},
		{{"g14uh"}, {259, 208, 633, 1454, 2986, 635, 1442, 2967, 633, 1454, 2986, 635, 1489, 3034, 642, 1521, 3020, 
			654, 1563, 2986, 666, 1603, 2981, 675, 1625, 2930, 657, 1745, 2908, 85, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 17, 0}},
		{{"g14uw"}, {304, 239, 486, 968, 2756, 487, 991, 2756, 482, 906, 2783, 456, 927, 2732, 442, 915, 2685, 
			439, 880, 2713, 437, 874, 2709, 435, 867, 2719, 433, 869, 2692, 85, 0, 0, 0, 0, 0, 0, 0, 1, 2, 17, 0, 0}},
		{{"g15ae"}, {354, 226, 888, 2605, 3651, 941, 2614, 3647, 887, 2619, 3592, 894, 2570, 3537, 922, 2558, 3449, 
			1018, 2511, 3367, 1113, 2285, 3014, 1099, 2153, 2888, 1055, 2134, 2867, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"g15ah"}, {299, 233, 1316, 1752, 3113, 1303, 1776, 3249, 1245, 1731, 3177, 1345, 1746, 3111, 1328, 1760, 3137, 
			1246, 1787, 3152, 1213, 1817, 3112, 1178, 1853, 3071, 1063, 1901, 3024, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"g15aw"}, {360, 239, 894, 1366, 3371, 887, 1383, 3460, 877, 1391, 3416, 896, 1367, 3371, 880, 1371, 3376, 
			882, 1381, 3345, 902, 1405, 3332, 930, 1539, 3326, 911, 1681, 3269, 90, 0, 0, 0, 0, 0, 2, 18, 0, 0, 0, 0, 0}},
		{{"g15eh"}, {266, 226, 880, 2638, 3567, 946, 2621, 3618, 876, 2606, 3572, 879, 2632, 3533, 880, 2638, 3567, 
			874, 2650, 3616, 870, 2669, 3627, 868, 2656, 3630, 813, 2666, 3656, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g15ei"}, {386, 237, 497, 2804, 3315, 508, 2746, 3312, 494, 2808, 3346, 494, 2795, 3346, 487, 2803, 3365, 
			476, 2822, 3452, 457, 2930, 3481, 445, 2945, 3522, 445, 2965, 3476, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g15er"}, {331, 248, 707, 1706, 2245, 680, 1701, 2303, 701, 1702, 2229, 690, 1699, 2195, 662, 1691, 2119, 
			619, 1684, 2054, 604, 1678, 1979, 592, 1670, 1977, 561, 1691, 2048, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"g15ih"}, {337, 244, 497, 2684, 3546, 517, 2757, 3558, 497, 2684, 3546, 494, 2667, 3552, 496, 2719, 3535, 
			498, 2731, 3544, 545, 2698, 3514, 555, 2683, 3532, 600, 2647, 3532, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g15iy"}, {357, 251, 477, 3266, 3889, 462, 3199, 3925, 478, 3183, 3918, 488, 3292, 3904, 477, 3250, 3875, 
			465, 3270, 3867, 463, 3265, 3833, 459, 3230, 3783, 452, 3200, 3741, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g15oa"}, {382, 243, 577, 1217, 3225, 718, 1312, 3292, 694, 1291, 3303, 681, 1249, 3312, 644, 1231, 3233, 
			589, 1221, 3236, 571, 1215, 3195, 557, 1188, 3198, 538, 1201, 3175, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"g15oo"}, {298, 233, 597, 1599, 3270, 585, 1559, 3222, 599, 1557, 3270, 624, 1616, 3306, 632, 1608, 3297, 
			639, 1618, 3267, 634, 1731, 3329, 623, 1805, 3312, 627, 1860, 3242, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"g15uh"}, {302, 247, 684, 1732, 3241, 684, 1740, 3241, 692, 1705, 3243, 700, 1711, 3319, 696, 1704, 3438, 
			694, 1704, 3505, 684, 1704, 3518, 666, 1733, 3364, 660, 1913, 3325, 90, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 18, 0}},
		{{"g15uw"}, {368, 244, 496, 1739, 2873, 499, 1767, 2833, 497, 1740, 2870, 498, 1742, 2853, 496, 1735, 2883, 
			496, 1744, 2882, 495, 1733, 2870, 494, 1726, 2868, 496, 1733, 2870, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19, 0, 0}},
		{{"g17ae"}, {308, 208, 727, 2394, 3320, 708, 2399, 3326, 749, 2408, 3317, 793, 2413, 3267, 812, 2282, 3183, 
			832, 2131, 3165, 869, 2035, 3192, 878, 2003, 3210, 812, 2063, 3271, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"g17ah"}, {304, 248, 911, 1624, 3230, 903, 1617, 3276, 923, 1617, 3222, 911, 1628, 3221, 934, 1665, 3223, 
			933, 1691, 3225, 933, 1730, 3214, 927, 1770, 3194, 865, 1860, 3165, 90, 0, 0, 0, 0, 0, 18, 1, 0, 0, 0, 1, 0}},
		{{"g17aw"}, {291, 233, 757, 1242, 3288, 750, 1235, 3295, 770, 1279, 3285, 776, 1310, 3300, 824, 1329, 3320, 
			868, 1378, 3333, 874, 1434, 3332, 867, 1507, 3339, 866, 1629, 3306, 95, 0, 0, 0, 0, 0, 0, 19, 0, 0, 0, 1, 0}},
		{{"g17eh"}, {204, 249, 755, 2137, 3384, 755, 2133, 3384, 748, 2161, 3372, 747, 2152, 3372, 754, 2139, 3346, 
			755, 2127, 3312, 755, 2123, 3317, 750, 2160, 3358, 747, 2214, 3409, 95, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0}},
		{{"g17ei"}, {280, 258, 575, 2502, 3445, 596, 2430, 3434, 575, 2502, 3445, 556, 2511, 3409, 544, 2528, 3416, 
			523, 2524, 3367, 507, 2548, 3335, 503, 2578, 3322, 502, 2596, 3306, 85, 0, 3, 17, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g17er"}, {269, 225, 667, 1688, 2246, 644, 1678, 2293, 667, 1714, 2248, 645, 1711, 2292, 635, 1708, 2294, 
			628, 1747, 2257, 624, 1819, 2239, 621, 1860, 2273, 616, 1992, 2394, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"g17ih"}, {212, 243, 504, 2408, 3422, 507, 2452, 3439, 504, 2429, 3439, 505, 2389, 3425, 510, 2398, 3435, 
			532, 2336, 3364, 542, 2308, 3362, 555, 2313, 3384, 552, 2297, 3405, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g17iy"}, {266, 273, 466, 3004, 3634, 457, 3030, 3636, 476, 3005, 3650, 463, 3010, 3623, 465, 3013, 3628, 
			469, 3004, 3615, 467, 2996, 3586, 484, 3001, 3545, 495, 2977, 3536, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g17oa"}, {262, 267, 785, 1551, 3138, 785, 1558, 3129, 749, 1520, 3106, 715, 1480, 3089, 695, 1379, 3146, 
			670, 1327, 3144, 618, 1249, 3123, 553, 1208, 3112, 526, 1359, 2974, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"g17oo"}, {188, 236, 643, 1860, 3306, 570, 1815, 3290, 633, 1842, 3304, 651, 1877, 3305, 659, 1883, 3303, 
			662, 1897, 3294, 655, 1938, 3296, 652, 2072, 3296, 627, 2143, 3364, 90, 0, 0, 0, 0, 0, 0, 0, 0, 18, 1, 1, 0}},
		{{"g17uh"}, {215, 256, 782, 1622, 3171, 782, 1622, 3203, 753, 1580, 3167, 750, 1604, 3166, 747, 1671, 3129, 
			745, 1728, 3099, 738, 1825, 3094, 710, 1892, 3115, 688, 2040, 3151, 95, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 19, 0}},
		{{"g17uw"}, {245, 236, 490, 2179, 3131, 497, 2186, 3143, 482, 2163, 3097, 491, 2129, 3113, 493, 2103, 3121, 
			491, 2044, 3097, 494, 1930, 3124, 496, 1932, 3121, 500, 1912, 3056, 75, 0, 4, 0, 0, 0, 0, 0, 0, 1, 15, 0, 0}},
		{{"g18ae"}, {317, 219, 591, 2632, 0, 564, 2608, 0, 556, 2610, 0, 514, 2586, 0, 552, 2566, 0, 
			591, 2504, 0, 804, 2387, 0, 913, 2341, 0, 944, 2297, 0, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"g18ah"}, {416, 214, 1033, 1929, 3428, 1041, 1929, 3428, 1064, 1931, 3357, 1056, 1907, 3273, 1050, 1906, 3393, 
			1066, 1922, 3286, 1054, 1959, 3150, 1025, 1977, 3211, 846, 2031, 3408, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"g18aw"}, {441, 215, 1062, 0, 3233, 1017, 0, 3284, 1063, 0, 3212, 1054, 0, 3274, 1050, 0, 3139, 
			1048, 0, 3121, 1047, 1728, 3317, 1027, 1799, 3371, 888, 1862, 3480, 75, 0, 0, 0, 0, 0, 5, 15, 0, 0, 0, 0, 0}},
		{{"g18eh"}, {258, 205, 533, 2480, 3378, 515, 2504, 3279, 538, 2458, 3418, 524, 2423, 3242, 546, 2490, 3280, 
			655, 2462, 3284, 727, 2464, 3295, 748, 2457, 3355, 733, 2418, 3335, 70, 0, 0, 0, 14, 6, 0, 0, 0, 0, 0, 0, 0}},
		{{"g18ei"}, {325, 209, 570, 3256, 0, 567, 3200, 0, 558, 3239, 0, 574, 3340, 0, 575, 3274, 0, 
			603, 3299, 0, 447, 3269, 0, 442, 3345, 0, 447, 3270, 0, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g18er"}, {287, 219, 611, 2033, 2522, 622, 2428, 2675, 616, 2142, 2554, 609, 2022, 2565, 589, 1995, 2489, 
			595, 2063, 2454, 561, 2099, 2457, 500, 2166, 2485, 528, 2237, 2522, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"g18ih"}, {267, 212, 557, 2664, 3468, 521, 2719, 3422, 543, 2711, 3448, 560, 2654, 3459, 572, 2687, 3436, 
			575, 2677, 3370, 575, 2588, 3367, 579, 2603, 3308, 565, 2605, 3348, 95, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g18iy"}, {343, 222, 436, 3443, 3843, 439, 3434, 3799, 436, 3443, 3843, 438, 3361, 3743, 439, 3363, 3964, 
			444, 3384, 3901, 458, 3384, 3877, 478, 3466, 3865, 482, 3367, 3819, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g18oa"}, {300, 221, 568, 1096, 2991, 575, 1130, 3197, 611, 1123, 2995, 572, 1102, 2935, 566, 1085, 3103, 
			569, 1063, 3055, 573, 1041, 3037, 558, 1098, 2864, 519, 1119, 2933, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"g18oo"}, {211, 217, 556, 1558, 3490, 559, 1587, 3538, 558, 1559, 3506, 552, 1560, 3474, 549, 1600, 3388, 
			549, 1628, 3343, 557, 1675, 3316, 564, 1772, 3318, 592, 1810, 3310, 85, 0, 0, 0, 0, 0, 0, 0, 0, 17, 0, 3, 0}},
		{{"g18uh"}, {226, 207, 744, 1854, 3360, 679, 1886, 3592, 716, 1905, 3613, 740, 1844, 3530, 744, 1854, 3360, 
			740, 1865, 3377, 719, 1869, 3459, 706, 1946, 3477, 674, 2089, 3420, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"g18uw"}, {315, 214, 435, 1829, 3316, 436, 1810, 3315, 432, 1836, 3378, 433, 1851, 3405, 434, 1750, 3339, 
			435, 1684, 3276, 435, 1658, 3287, 435, 1678, 3209, 444, 1716, 3229, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"g19ae"}, {238, 248, 747, 2703, 3829, 748, 2696, 3871, 744, 2708, 3830, 742, 2671, 3738, 740, 2615, 3667, 
			792, 2405, 3547, 930, 2325, 3425, 921, 2277, 3383, 916, 2177, 3339, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"g19ah"}, {220, 235, 1129, 2005, 2826, 1140, 2027, 2868, 1129, 1988, 2709, 1144, 1990, 2787, 1179, 1997, 2764, 
			980, 2030, 2741, 1131, 2056, 2746, 1100, 2077, 2786, 1085, 2096, 2848, 75, 0, 0, 0, 0, 4, 15, 0, 0, 1, 0, 0, 0}},
		{{"g19aw"}, {218, 221, 1090, 1465, 2981, 1081, 1472, 2985, 1103, 1458, 2978, 1110, 1458, 2971, 1126, 1488, 2981, 
			1107, 1557, 2766, 1111, 1602, 2943, 1112, 1740, 2835, 1103, 1827, 2752, 65, 0, 0, 0, 1, 0, 6, 13, 0, 0, 0, 0, 0}},
		{{"g19eh"}, {222, 229, 868, 2318, 3567, 812, 2329, 3645, 849, 2336, 3601, 873, 2304, 3691, 877, 2281, 3586, 
			867, 2334, 3565, 852, 2353, 3573, 833, 2363, 3592, 746, 2373, 3656, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g19ei"}, {223, 234, 618, 2752, 4040, 617, 2753, 4049, 618, 2738, 4046, 596, 2720, 4028, 578, 2740, 4046, 
			563, 2766, 4120, 550, 2780, 4145, 541, 2748, 4126, 518, 2699, 4037, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g19er"}, {197, 236, 670, 1899, 2323, 673, 1910, 2323, 663, 1932, 2344, 663, 1986, 2379, 675, 2014, 2418, 
			670, 2022, 2438, 665, 2025, 2447, 653, 2028, 2467, 643, 2017, 2454, 95, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 19}},
		{{"g19ih"}, {168, 236, 550, 2625, 3774, 536, 2633, 3668, 557, 2622, 3773, 570, 2622, 3646, 602, 2598, 3577, 
			634, 2590, 3561, 666, 2583, 3561, 669, 2576, 3666, 667, 2554, 3686, 85, 0, 17, 0, 3, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g19iy"}, {193, 237, 479, 3093, 3755, 471, 3064, 3753, 480, 3086, 3773, 478, 3087, 3735, 476, 3077, 3737, 
			471, 3078, 3750, 467, 3112, 3722, 460, 3159, 3743, 455, 3159, 3743, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g19oa"}, {222, 234, 688, 1197, 3184, 696, 1353, 3207, 692, 1292, 3141, 688, 1219, 3167, 684, 1160, 3178, 
			678, 1151, 3144, 675, 1124, 3148, 667, 1249, 3086, 640, 1374, 3025, 95, 0, 0, 0, 0, 0, 0, 0, 19, 0, 0, 0, 1}},
		{{"g19oo"}, {155, 237, 682, 1571, 3394, 673, 1504, 3445, 682, 1631, 3373, 673, 1657, 3333, 666, 1701, 3363, 
			667, 1776, 3452, 677, 1937, 3519, 675, 2050, 3567, 659, 2156, 3551, 75, 0, 0, 0, 0, 0, 0, 0, 0, 15, 0, 5, 0}},
		{{"g19uh"}, {184, 227, 885, 1629, 3430, 885, 1629, 3430, 887, 1675, 3418, 892, 1712, 3418, 883, 1745, 3395, 
			869, 1774, 3319, 855, 1862, 3319, 839, 1965, 3319, 748, 2072, 3326, 85, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 17, 0}},
		{{"g19uw"}, {192, 243, 497, 1334, 3067, 498, 1356, 3067, 491, 1335, 3067, 490, 1355, 3068, 493, 1376, 3068, 
			492, 1402, 3044, 487, 1417, 3023, 481, 1448, 2994, 478, 1490, 2961, 90, 0, 0, 0, 0, 0, 0, 0, 0, 2, 18, 0, 0}},
		{{"g20ae"}, {290, 227, 672, 2484, 3492, 672, 2484, 3492, 754, 2523, 3470, 940, 2484, 3294, 1030, 2277, 3143, 
			1070, 2100, 3162, 1090, 1937, 3045, 1067, 1935, 3020, 1048, 2009, 3316, 95, 0, 0, 0, 0, 19, 0, 1, 0, 0, 0, 0, 0}},
		{{"g20ah"}, {259, 234, 1145, 1655, 3062, 1047, 1550, 3169, 1072, 1622, 3245, 1128, 1664, 3166, 1135, 1645, 3108, 
			1150, 1726, 3067, 1132, 1721, 3137, 1096, 1871, 3139, 981, 1992, 3329, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"g20aw"}, {274, 246, 998, 1368, 3282, 987, 1395, 3318, 998, 1368, 3282, 988, 1377, 3340, 986, 1367, 3290, 
			1003, 1455, 3344, 1060, 1544, 3364, 980, 1667, 3370, 929, 1956, 3396, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"g20eh"}, {228, 228, 882, 2524, 3555, 895, 2578, 3551, 882, 2535, 3539, 884, 2524, 3549, 872, 2439, 3527, 
			864, 2351, 3514, 855, 2337, 3598, 835, 2293, 3592, 792, 2344, 3638, 95, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0}},
		{{"g20ei"}, {294, 249, 540, 2687, 3504, 543, 2659, 3488, 506, 2733, 3537, 493, 2832, 3435, 476, 2833, 3412, 
			457, 2914, 3369, 457, 2925, 3360, 471, 2933, 3353, 495, 2943, 3344, 95, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g20er"}, {286, 230, 627, 1628, 2058, 554, 1666, 2110, 635, 1624, 2055, 646, 1634, 2003, 643, 1689, 2006, 
			636, 1784, 2050, 621, 1786, 2185, 639, 1845, 2278, 553, 1925, 2346, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"g20ih"}, {220, 234, 489, 2730, 3763, 500, 2774, 3761, 489, 2719, 3783, 516, 2688, 3730, 571, 2665, 3697, 
			602, 2681, 3712, 613, 2662, 3739, 612, 2624, 3772, 610, 2589, 3764, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g20iy"}, {243, 232, 469, 3278, 4165, 493, 3284, 4114, 488, 3288, 4249, 487, 3346, 4226, 468, 3315, 4203, 
			461, 3291, 4180, 473, 3265, 4158, 472, 3265, 4135, 478, 3261, 4113, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g20oa"}, {317, 213, 622, 1223, 3380, 670, 1358, 3518, 649, 1352, 3511, 644, 1287, 3449, 626, 1256, 3469, 
			622, 1223, 3380, 598, 1207, 3470, 529, 1167, 3462, 486, 1223, 3410, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"g20oo"}, {241, 239, 512, 1644, 3483, 508, 1662, 3479, 517, 1635, 3487, 552, 1636, 3259, 608, 1672, 3313, 
			623, 1811, 3338, 631, 1861, 3343, 610, 1998, 3415, 545, 2126, 3511, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"g20uh"}, {210, 248, 1045, 1609, 3456, 1049, 1588, 3432, 1031, 1636, 3456, 991, 1677, 3448, 928, 1686, 3494, 
			898, 1858, 3480, 800, 1874, 3435, 680, 2046, 3444, 675, 2274, 3500, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"g20uw"}, {248, 248, 498, 1740, 3291, 500, 1744, 3298, 496, 1728, 3327, 494, 1700, 3362, 485, 1668, 3308, 
			467, 1665, 3312, 449, 1657, 3337, 448, 1706, 3376, 450, 1821, 3383, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19, 0, 0}},
		{{"g21ae"}, {250, 227, 555, 2569, 3424, 539, 2590, 3465, 570, 2549, 3424, 627, 2391, 3204, 873, 2197, 3241, 
			923, 2115, 3286, 992, 2062, 3329, 979, 2057, 3389, 969, 2057, 3427, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"g21ah"}, {186, 215, 1021, 1720, 3186, 1178, 1804, 3261, 1075, 1745, 3268, 1058, 1736, 3181, 1099, 1658, 3220, 
			1050, 1773, 3250, 1051, 1883, 3195, 1026, 2069, 3320, 947, 2146, 3333, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"g21aw"}, {259, 218, 1001, 1488, 3149, 1019, 1492, 3162, 1001, 1488, 3149, 1024, 1509, 3150, 1021, 1538, 3189, 
			868, 1548, 3273, 827, 1564, 3349, 813, 1714, 3379, 784, 1849, 3076, 30, 0, 0, 0, 0, 0, 8, 6, 0, 0, 0, 6, 0}},
		{{"g21eh"}, {227, 230, 733, 2421, 3250, 688, 2313, 3429, 697, 2351, 3443, 717, 2399, 3327, 733, 2421, 3212, 
			725, 2421, 3246, 700, 2443, 3299, 679, 2486, 3351, 676, 2498, 3369, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g21ei"}, {259, 235, 558, 2533, 3224, 543, 2645, 3283, 546, 2612, 3045, 526, 2762, 3037, 473, 2765, 3155, 
			445, 2668, 3273, 443, 2651, 3164, 448, 2640, 3194, 446, 2651, 3122, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g21er"}, {262, 225, 620, 1689, 2361, 626, 1654, 2326, 629, 1637, 2221, 627, 1724, 2179, 624, 1761, 2147, 
			623, 1710, 2130, 623, 1675, 2148, 625, 1690, 2269, 623, 1715, 2345, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"g21ih"}, {170, 226, 444, 2846, 3518, 446, 2830, 3591, 443, 2869, 3518, 443, 2918, 3570, 444, 2831, 3619, 
			445, 2711, 3628, 449, 2680, 3631, 458, 2660, 3675, 460, 2638, 3632, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g21iy"}, {258, 238, 472, 3120, 3574, 482, 3126, 3735, 483, 3152, 3677, 481, 3141, 3698, 480, 3108, 3568, 
			472, 3120, 3574, 473, 3136, 3565, 472, 3138, 3572, 471, 3132, 3579, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"g21oa"}, {246, 222, 643, 1262, 3194, 614, 1272, 3303, 640, 1259, 3244, 643, 1262, 3194, 617, 1265, 3257, 
			586, 1268, 3339, 557, 1272, 3368, 517, 1316, 3320, 497, 1350, 3324, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"g21oo"}, {179, 243, 575, 1580, 3509, 575, 1580, 3509, 575, 1608, 3520, 579, 1666, 3520, 585, 1682, 3472, 
			583, 1797, 3446, 601, 1874, 3443, 576, 1977, 3443, 528, 2096, 3463, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"g21uh"}, {177, 229, 646, 1578, 3366, 634, 1586, 3354, 665, 1545, 3380, 678, 1535, 3380, 681, 1612, 3386, 
			677, 1779, 3369, 659, 1960, 3486, 638, 2071, 3564, 634, 2175, 3526, 90, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 18, 0}},
		{{"g21uw"}, {227, 225, 446, 1533, 3269, 484, 1649, 3309, 465, 1622, 3261, 451, 1591, 3239, 446, 1548, 3271, 
			444, 1525, 3265, 445, 1548, 3241, 445, 1561, 3175, 442, 1619, 3099, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19, 0, 0}},
		{{"m01ae"}, {323, 174, 663, 2012, 2659, 675, 2018, 2664, 669, 2008, 2671, 658, 2012, 2665, 663, 2012, 2659, 
			671, 1992, 2659, 682, 1933, 2645, 686, 1834, 2632, 685, 1773, 2680, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"m01ah"}, {316, 159, 813, 1283, 2687, 808, 1280, 2659, 809, 1280, 2687, 811, 1279, 2690, 822, 1279, 2687, 
			839, 1259, 2629, 828, 1250, 2619, 823, 1301, 2612, 752, 1496, 2620, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"m01aw"}, {330, 177, 699, 996, 2548, 675, 973, 2503, 697, 996, 2548, 701, 1003, 2551, 710, 1004, 2552, 
			731, 1033, 2554, 752, 1154, 2554, 744, 1314, 2530, 688, 1394, 2539, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"m01eh"}, {266, 171, 609, 1980, 2678, 601, 1986, 2678, 609, 1980, 2678, 612, 1953, 2681, 615, 1926, 2677, 
			624, 1887, 2682, 624, 1868, 2681, 614, 1867, 2673, 561, 1851, 2697, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m01ei"}, {319, 166, 494, 2113, 2700, 497, 2120, 2698, 494, 2128, 2742, 490, 2174, 2762, 480, 2198, 2754, 
			442, 2248, 2755, 411, 2310, 2756, 368, 2317, 2793, 352, 2332, 2797, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m01er"}, {324, 175, 506, 1418, 1849, 513, 1385, 1905, 506, 1418, 1849, 504, 1406, 1829, 502, 1418, 1825, 
			499, 1424, 1802, 491, 1444, 1801, 474, 1485, 1787, 431, 1545, 1854, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m01ih"}, {268, 179, 370, 2161, 2757, 372, 2179, 2755, 369, 2161, 2757, 369, 2164, 2717, 373, 2112, 2695, 
			393, 2080, 2679, 429, 2004, 2665, 448, 1937, 2681, 443, 1936, 2705, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m01iy"}, {339, 172, 328, 2418, 2939, 332, 2399, 2948, 331, 2432, 2924, 332, 2438, 2877, 333, 2442, 2872, 
			332, 2438, 2859, 331, 2453, 2824, 338, 2453, 2824, 357, 2458, 2856, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m01oa"}, {369, 169, 509, 989, 2561, 525, 997, 2552, 512, 992, 2549, 507, 989, 2561, 505, 977, 2563, 
			501, 954, 2559, 497, 915, 2553, 483, 900, 2550, 428, 895, 2568, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m01oo"}, {255, 169, 450, 995, 2519, 424, 1010, 2473, 439, 1005, 2508, 450, 995, 2519, 465, 995, 2506, 
			486, 999, 2494, 499, 1119, 2489, 501, 1317, 2482, 499, 1508, 2474, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m01uh"}, {292, 171, 670, 1178, 2566, 658, 1159, 2566, 665, 1175, 2573, 671, 1196, 2560, 661, 1240, 2561, 
			661, 1257, 2561, 649, 1331, 2558, 639, 1437, 2561, 594, 1550, 2575, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 19, 0}},
		{{"m01uw"}, {281, 184, 374, 931, 2433, 374, 930, 2430, 373, 930, 2443, 372, 918, 2467, 371, 911, 2458, 
			370, 915, 2484, 372, 928, 2473, 374, 941, 2426, 375, 967, 2470, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m02ae"}, {250, 102, 628, 1871, 2477, 630, 1975, 2472, 627, 1871, 2456, 629, 1880, 2477, 630, 1893, 2509, 
			636, 1881, 2455, 634, 1841, 2440, 634, 1820, 2451, 628, 1793, 2451, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"m02ah"}, {249, 101, 749, 1060, 2842, 751, 1060, 2843, 718, 1049, 2804, 716, 1054, 2800, 726, 1058, 2784, 
			742, 1109, 2804, 746, 1138, 2804, 751, 1104, 2783, 760, 1138, 2758, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"m02aw"}, {289, 106, 644, 856, 2627, 627, 855, 2602, 593, 847, 2547, 623, 867, 2624, 570, 857, 2682, 
			615, 881, 2633, 586, 864, 2634, 631, 863, 2632, 632, 906, 2603, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"m02eh"}, {167, 103, 517, 1917, 2438, 517, 1917, 2438, 533, 1899, 2419, 533, 1898, 2414, 533, 1857, 2414, 
			549, 1823, 2428, 555, 1797, 2428, 555, 1769, 2431, 533, 1769, 2479, 95, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0}},
		{{"m02ei"}, {261, 105, 438, 2196, 2480, 438, 2196, 2480, 435, 2130, 2610, 427, 2149, 2670, 418, 2163, 2717, 
			409, 2197, 2734, 385, 2199, 2744, 365, 2266, 2852, 328, 2268, 2819, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m02er"}, {296, 111, 457, 1406, 0, 526, 1449, 0, 498, 1383, 0, 452, 1422, 0, 457, 1406, 0, 
			449, 1423, 0, 438, 1426, 0, 428, 1442, 0, 432, 1493, 0, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m02ih"}, {183, 111, 384, 2039, 2523, 383, 2163, 2581, 384, 2039, 2523, 392, 2022, 2449, 403, 1984, 2447, 
			425, 1941, 2412, 431, 1894, 2412, 431, 1828, 2400, 433, 1798, 2392, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m02iy"}, {236, 108, 305, 2320, 3072, 296, 2325, 3079, 305, 2321, 3075, 304, 2319, 3061, 299, 2326, 3059, 
			293, 2339, 3054, 293, 2322, 3023, 299, 2307, 3016, 296, 2308, 3007, 95, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m02oa"}, {261, 105, 519, 891, 2532, 544, 888, 2577, 519, 891, 2532, 495, 913, 2689, 463, 887, 2580, 
			440, 866, 2570, 432, 865, 2544, 423, 870, 2441, 427, 946, 2483, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m02oo"}, {174, 112, 424, 992, 2249, 410, 936, 2251, 418, 949, 2247, 424, 993, 2247, 432, 1003, 2239, 
			435, 1055, 2240, 436, 1063, 2257, 437, 1101, 2258, 437, 1266, 2179, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m02uh"}, {173, 105, 642, 998, 2682, 682, 972, 2709, 642, 998, 2682, 636, 1017, 2677, 634, 1045, 2667, 
			642, 1046, 2662, 631, 1056, 2658, 633, 1106, 2619, 591, 1298, 2548, 65, 0, 0, 0, 0, 0, 0, 7, 0, 0, 0, 13, 0}},
		{{"m02uw"}, {257, 114, 319, 938, 2091, 326, 943, 2113, 320, 938, 2092, 319, 930, 2076, 319, 923, 2106, 
			320, 931, 2111, 325, 938, 2111, 327, 968, 2111, 331, 1038, 2073, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m03ae"}, {344, 99, 605, 1812, 2570, 605, 1812, 2570, 608, 1812, 2572, 612, 1795, 2560, 616, 1779, 2682, 
			618, 1789, 2618, 625, 1792, 2658, 630, 1761, 2703, 632, 1708, 2693, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"m03ah"}, {373, 97, 755, 1133, 2695, 748, 1101, 2696, 750, 1117, 2692, 754, 1125, 2694, 767, 1138, 2734, 
			766, 1161, 2740, 767, 1160, 2705, 758, 1165, 2698, 754, 1159, 2691, 70, 0, 0, 0, 0, 0, 14, 6, 0, 0, 0, 0, 0}},
		{{"m03aw"}, {355, 93, 647, 926, 2785, 647, 926, 2785, 647, 936, 2799, 642, 945, 2786, 668, 986, 2776, 
			672, 1002, 2771, 655, 1039, 2759, 660, 1047, 2751, 651, 1106, 2619, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"m03eh"}, {284, 95, 559, 1695, 2413, 508, 1807, 2405, 515, 1791, 2391, 549, 1724, 2391, 556, 1714, 2401, 
			559, 1693, 2417, 560, 1644, 2416, 561, 1631, 2439, 558, 1631, 2507, 65, 0, 0, 0, 13, 7, 0, 0, 0, 0, 0, 0, 0}},
		{{"m03ei"}, {353, 91, 438, 2026, 2476, 442, 2017, 2462, 439, 2029, 2476, 439, 2029, 2492, 435, 2105, 2479, 
			427, 2071, 2470, 414, 2085, 2478, 379, 2101, 2460, 371, 2134, 2418, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m03er"}, {368, 94, 479, 1252, 1621, 486, 1217, 1637, 469, 1299, 1621, 461, 1247, 1616, 449, 1250, 1619, 
			445, 1276, 1609, 455, 1275, 1612, 442, 1337, 1618, 436, 1350, 1619, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m03ih"}, {262, 101, 411, 1932, 2569, 429, 2024, 2575, 429, 1985, 2560, 411, 1943, 2558, 416, 1898, 2561, 
			420, 1889, 2568, 424, 1877, 2557, 427, 1874, 2562, 428, 1812, 2562, 95, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m03iy"}, {315, 96, 331, 2311, 2865, 311, 2356, 2815, 309, 2324, 2828, 345, 2334, 2835, 339, 2326, 2859, 
			344, 2341, 2790, 303, 2324, 2796, 327, 2289, 2668, 309, 2311, 2757, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m03oa"}, {417, 96, 496, 863, 2548, 535, 917, 2486, 509, 863, 2499, 518, 868, 2556, 506, 857, 2582, 
			492, 862, 2571, 453, 848, 2595, 480, 864, 2617, 422, 910, 2529, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m03oo"}, {275, 96, 489, 1059, 2567, 483, 1033, 2561, 489, 1092, 2567, 485, 1053, 2584, 484, 1061, 2526, 
			486, 1080, 2535, 489, 1126, 2557, 489, 1177, 2558, 484, 1321, 2475, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m03uh"}, {297, 96, 622, 1075, 2622, 631, 1043, 2691, 622, 1063, 2574, 622, 1075, 2622, 621, 1087, 2625, 
			620, 1080, 2642, 619, 1093, 2635, 613, 1137, 2645, 591, 1293, 2567, 50, 0, 0, 0, 0, 0, 4, 6, 0, 0, 0, 10, 0}},
		{{"m03uw"}, {420, 101, 360, 953, 2505, 393, 975, 2472, 378, 986, 2463, 383, 986, 2469, 355, 958, 2500, 
			356, 960, 2499, 359, 977, 2491, 355, 995, 2457, 350, 1013, 2434, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m04ae"}, {312, 124, 627, 1910, 2488, 627, 1910, 2488, 629, 1882, 2460, 672, 1884, 2482, 693, 1841, 2441, 
			720, 1750, 2435, 755, 1645, 2422, 764, 1572, 2462, 757, 1563, 2527, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m04ah"}, {302, 127, 832, 1222, 2624, 816, 1101, 2539, 831, 1183, 2555, 832, 1172, 2579, 832, 1222, 2624, 
			819, 1261, 2607, 815, 1329, 2575, 814, 1284, 2545, 804, 1320, 2592, 90, 0, 0, 0, 0, 0, 18, 2, 0, 0, 0, 0, 0}},
		{{"m04aw"}, {347, 131, 694, 970, 2746, 702, 964, 2806, 702, 985, 2746, 698, 939, 2746, 716, 976, 2707, 
			743, 998, 2739, 740, 1086, 2687, 744, 1258, 2701, 724, 1354, 2616, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"m04eh"}, {229, 127, 698, 1580, 2492, 643, 1783, 2521, 675, 1723, 2522, 694, 1640, 2516, 695, 1713, 2518, 
			691, 1675, 2586, 690, 1600, 2703, 645, 1517, 2670, 622, 1543, 2646, 30, 0, 0, 0, 6, 14, 0, 0, 0, 0, 0, 0, 0}},
		{{"m04ei"}, {253, 134, 493, 2102, 2839, 503, 2127, 2762, 489, 2125, 2802, 436, 2155, 2790, 415, 2175, 2865, 
			404, 2264, 2862, 413, 2247, 2997, 412, 2334, 2951, 415, 2345, 2903, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m04er"}, {332, 134, 460, 1473, 0, 484, 1471, 0, 501, 1361, 0, 479, 1449, 0, 459, 1493, 0, 
			454, 1492, 0, 462, 1444, 0, 446, 1480, 0, 436, 1493, 0, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m04ih"}, {227, 135, 441, 2045, 2694, 447, 1996, 2644, 480, 2037, 2637, 502, 1884, 2645, 508, 1905, 2595, 
			513, 1786, 2580, 511, 1776, 2680, 510, 1726, 2670, 504, 1716, 2715, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m04iy"}, {240, 134, 353, 2385, 3400, 361, 2459, 3515, 350, 2366, 3502, 352, 2332, 3580, 352, 2382, 3402, 
			353, 2385, 3400, 345, 2389, 3465, 332, 2426, 3343, 357, 2321, 3447, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m04oa"}, {300, 146, 441, 862, 2455, 497, 867, 2499, 484, 864, 2460, 454, 863, 2390, 444, 863, 2482, 
			440, 860, 2455, 438, 818, 2385, 435, 775, 2398, 434, 794, 2333, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m04oo"}, {210, 134, 522, 1123, 2538, 497, 1028, 2492, 522, 1123, 2538, 534, 1116, 2510, 543, 1148, 2484, 
			548, 1304, 2516, 553, 1315, 2631, 552, 1445, 2604, 548, 1494, 2470, 90, 0, 0, 0, 0, 0, 0, 0, 0, 18, 2, 0, 0}},
		{{"m04uh"}, {211, 131, 650, 1036, 2697, 638, 982, 2690, 686, 1046, 2705, 691, 1096, 2767, 691, 1159, 2664, 
			691, 1195, 2766, 691, 1394, 2763, 655, 1424, 2759, 617, 1475, 2756, 50, 0, 0, 0, 0, 0, 4, 6, 0, 0, 0, 10, 0}},
		{{"m04uw"}, {288, 144, 422, 1299, 2347, 426, 1293, 2460, 414, 1265, 2417, 389, 1266, 2527, 390, 1142, 2364, 
			387, 1148, 2319, 384, 1117, 2327, 383, 1098, 2332, 384, 1061, 2340, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m06ae"}, {254, 115, 647, 1864, 2561, 649, 1862, 2557, 642, 1866, 2557, 638, 1865, 2551, 640, 1857, 2522, 
			666, 1829, 2499, 675, 1795, 2512, 686, 1758, 2547, 689, 1696, 2556, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m06ah"}, {230, 112, 871, 1204, 2595, 860, 1219, 2563, 873, 1209, 2583, 864, 1198, 2600, 868, 1198, 2600, 
			852, 1211, 2607, 843, 1220, 2609, 829, 1238, 2599, 799, 1325, 2557, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"m06aw"}, {250, 115, 740, 999, 2691, 700, 965, 2621, 706, 965, 2636, 721, 980, 2659, 753, 1006, 2693, 
			811, 1025, 2696, 773, 1044, 2695, 754, 1063, 2690, 707, 1070, 2627, 85, 0, 0, 0, 0, 0, 3, 17, 0, 0, 0, 0, 0}},
		{{"m06eh"}, {177, 115, 555, 1851, 2624, 555, 1847, 2627, 554, 1859, 2635, 555, 1851, 2624, 561, 1838, 2620, 
			563, 1830, 2620, 560, 1822, 2617, 554, 1818, 2617, 524, 1824, 2635, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m06ei"}, {241, 120, 497, 2024, 2677, 501, 2024, 2675, 486, 2063, 2675, 476, 2107, 2677, 471, 2140, 2684, 
			469, 2158, 2685, 444, 2200, 2685, 426, 2208, 2686, 403, 2195, 2682, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m06er"}, {217, 117, 489, 1338, 1747, 503, 1308, 1788, 500, 1331, 1783, 495, 1329, 1783, 492, 1332, 1762, 
			488, 1338, 1738, 476, 1346, 1728, 453, 1383, 1751, 447, 1418, 1790, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m06ih"}, {159, 115, 441, 2080, 2721, 439, 2109, 2739, 446, 2025, 2705, 446, 2009, 2695, 441, 2002, 2691, 
			442, 1985, 2684, 447, 1957, 2684, 447, 1944, 2692, 441, 1934, 2699, 95, 0, 19, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0}},
		{{"m06iy"}, {241, 122, 367, 2390, 2777, 376, 2303, 2736, 375, 2332, 2792, 374, 2347, 2813, 371, 2370, 2791, 
			369, 2375, 2777, 367, 2391, 2762, 366, 2396, 2734, 365, 2391, 2714, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m06oa"}, {263, 114, 524, 1000, 2432, 578, 1055, 2439, 568, 1045, 2439, 561, 1009, 2437, 509, 990, 2434, 
			495, 946, 2433, 465, 914, 2431, 447, 873, 2434, 457, 888, 2434, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m06oo"}, {147, 117, 468, 1115, 2492, 465, 1124, 2489, 468, 1124, 2489, 468, 1111, 2494, 471, 1111, 2497, 
			474, 1146, 2499, 474, 1168, 2499, 460, 1215, 2498, 449, 1368, 2498, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m06uh"}, {169, 117, 629, 1146, 2643, 629, 1146, 2643, 629, 1152, 2644, 624, 1166, 2664, 614, 1170, 2664, 
			602, 1170, 2653, 597, 1216, 2653, 573, 1429, 2661, 556, 1480, 2659, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 19, 0}},
		{{"m06uw"}, {231, 120, 366, 919, 2378, 402, 1081, 2380, 387, 1026, 2380, 379, 995, 2381, 375, 961, 2381, 
			369, 934, 2412, 367, 929, 2390, 366, 917, 2378, 371, 920, 2378, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m07ae"}, {254, 96, 582, 1999, 2567, 621, 1983, 2681, 592, 1958, 2568, 575, 1991, 2563, 599, 1981, 2532, 
			624, 1925, 2569, 646, 1823, 2602, 649, 1784, 2536, 626, 1791, 2577, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m07ah"}, {265, 98, 786, 1341, 2403, 760, 1314, 2424, 761, 1330, 2422, 767, 1321, 2357, 760, 1319, 2416, 
			755, 1341, 2412, 752, 1371, 2439, 760, 1489, 2454, 695, 1581, 2522, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"m07aw"}, {292, 96, 642, 949, 2437, 651, 946, 2416, 642, 955, 2437, 650, 981, 2438, 667, 990, 2436, 
			679, 1051, 2440, 687, 1119, 2460, 681, 1422, 2619, 631, 1597, 2723, 90, 0, 0, 0, 0, 0, 2, 18, 0, 0, 0, 0, 0}},
		{{"m07eh"}, {197, 104, 571, 1926, 2615, 609, 1977, 2757, 571, 1926, 2615, 575, 1890, 2602, 570, 1878, 2505, 
			568, 1882, 2607, 560, 1877, 2680, 549, 1897, 2637, 538, 1873, 2740, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m07ei"}, {281, 100, 555, 2174, 2900, 555, 2174, 2907, 501, 2191, 2928, 495, 2253, 2962, 474, 2259, 2992, 
			431, 2285, 3032, 420, 2311, 3062, 398, 2421, 3061, 420, 2378, 3016, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m07er"}, {272, 96, 493, 1270, 1633, 513, 1329, 1692, 482, 1301, 1635, 464, 1225, 1579, 445, 1333, 0, 
			450, 1461, 0, 445, 1512, 0, 456, 1521, 0, 439, 1504, 0, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m07ih"}, {190, 108, 363, 2269, 2935, 363, 2269, 2935, 428, 2208, 2885, 430, 2125, 2840, 432, 2060, 2810, 
			442, 2052, 2776, 493, 1941, 2803, 494, 1907, 2853, 489, 1853, 2808, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m07iy"}, {259, 101, 346, 2379, 3217, 323, 2377, 3191, 346, 2379, 3217, 353, 2370, 3247, 354, 2370, 3257, 
			342, 2442, 3231, 329, 2435, 3212, 340, 2395, 3183, 340, 2377, 3209, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m07oa"}, {266, 94, 515, 1126, 2455, 564, 1162, 2504, 515, 1126, 2455, 498, 1032, 2488, 477, 1012, 2478, 
			452, 982, 2460, 439, 945, 2456, 411, 935, 2447, 379, 1011, 2447, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m07oo"}, {197, 92, 442, 1176, 2435, 440, 1113, 2472, 442, 1169, 2472, 443, 1181, 2435, 444, 1181, 2454, 
			444, 1250, 2465, 452, 1325, 2463, 444, 1500, 2438, 440, 1643, 2570, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"m07uh"}, {185, 90, 651, 1176, 2564, 689, 1165, 2549, 651, 1176, 2564, 629, 1169, 2584, 619, 1168, 2633, 
			623, 1285, 2619, 611, 1414, 2606, 557, 1530, 2640, 556, 1617, 2614, 95, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 19, 0}},
		{{"m07uw"}, {231, 113, 326, 997, 2384, 351, 1048, 2347, 348, 1020, 2358, 326, 997, 2369, 324, 999, 2384, 
			322, 999, 2373, 323, 997, 2370, 329, 989, 2334, 321, 1047, 2318, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m08ae"}, {289, 122, 602, 1880, 2539, 602, 1880, 2584, 606, 1873, 2539, 615, 1901, 2535, 615, 1871, 2535, 
			612, 1843, 2509, 615, 1765, 2493, 619, 1717, 2510, 620, 1677, 2553, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m08ah"}, {302, 115, 748, 1293, 2446, 732, 1302, 2446, 749, 1288, 2454, 754, 1300, 2424, 737, 1300, 2430, 
			734, 1318, 2457, 711, 1333, 2450, 722, 1377, 2426, 697, 1440, 2477, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"m08aw"}, {294, 120, 651, 1033, 2477, 651, 1033, 2477, 736, 1041, 2555, 689, 1045, 2528, 698, 1042, 2523, 
			692, 1061, 2507, 696, 1117, 2494, 706, 1237, 2473, 688, 1358, 2457, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"m08eh"}, {239, 126, 582, 1768, 2463, 582, 1768, 2463, 609, 1757, 2476, 615, 1755, 2489, 620, 1746, 2493, 
			620, 1732, 2560, 618, 1725, 2556, 615, 1695, 2591, 608, 1689, 2618, 65, 0, 0, 0, 13, 7, 0, 0, 0, 0, 0, 0, 0}},
		{{"m08ei"}, {328, 133, 529, 1974, 2638, 529, 1974, 2663, 516, 1985, 2690, 517, 2030, 2681, 501, 2060, 2675, 
			495, 2079, 2736, 487, 2080, 2799, 445, 2110, 2772, 409, 2122, 2750, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m08er"}, {282, 131, 501, 1334, 1611, 506, 1347, 1677, 501, 1334, 1611, 498, 1332, 1606, 489, 1376, 1614, 
			483, 1391, 1618, 478, 1424, 1642, 464, 1430, 1676, 443, 1449, 1666, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m08ih"}, {235, 131, 390, 1948, 2566, 381, 2001, 2604, 386, 1981, 2583, 418, 1946, 2546, 442, 1891, 2536, 
			486, 1866, 2545, 497, 1833, 2562, 498, 1783, 2566, 498, 1768, 2575, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m08iy"}, {298, 164, 312, 2402, 2849, 312, 2402, 2797, 314, 2413, 2857, 319, 2415, 2871, 317, 2419, 2925, 
			316, 2423, 2920, 317, 2427, 2932, 325, 2419, 2935, 344, 2443, 2971, 95, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m08oa"}, {285, 125, 500, 868, 2552, 507, 900, 2457, 503, 881, 2509, 500, 868, 2552, 499, 862, 2529, 
			496, 852, 2554, 496, 855, 2553, 495, 854, 2577, 481, 848, 2546, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m08oo"}, {241, 132, 486, 1116, 2563, 444, 1155, 2516, 463, 1122, 2563, 494, 1116, 2570, 499, 1160, 2619, 
			501, 1203, 2584, 500, 1300, 2558, 498, 1379, 2501, 499, 1499, 2516, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m08uh"}, {242, 132, 626, 1159, 2643, 626, 1171, 2642, 626, 1155, 2667, 637, 1128, 2687, 644, 1189, 2678, 
			636, 1260, 2663, 626, 1292, 2641, 618, 1373, 2626, 617, 1486, 2644, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 19, 0}},
		{{"m08uw"}, {239, 126, 382, 1058, 2334, 393, 1090, 2320, 393, 1080, 2310, 387, 1067, 2310, 382, 1058, 2334, 
			377, 1057, 2319, 376, 1061, 2330, 376, 1056, 2330, 377, 1095, 2278, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19, 0, 0}},
		{{"m09ae"}, {339, 120, 545, 1872, 2630, 541, 1842, 2665, 544, 1872, 2628, 550, 1806, 2624, 558, 1803, 2628, 
			562, 1694, 2614, 565, 1667, 2629, 565, 1624, 2657, 563, 1616, 2668, 75, 0, 0, 0, 5, 15, 0, 0, 0, 0, 0, 0, 0}},
		{{"m09ah"}, {330, 122, 738, 1394, 2522, 726, 1362, 2503, 738, 1372, 2542, 729, 1394, 2521, 721, 1390, 2539, 
			694, 1399, 2544, 678, 1449, 2547, 629, 1449, 2597, 622, 1501, 2602, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"m09aw"}, {342, 123, 620, 1068, 2634, 618, 1044, 2676, 616, 1061, 2643, 626, 1087, 2610, 629, 1103, 2578, 
			632, 1120, 2565, 623, 1227, 2559, 610, 1305, 2576, 580, 1421, 2618, 90, 0, 0, 0, 0, 0, 1, 18, 0, 0, 0, 1, 0}},
		{{"m09eh"}, {239, 120, 560, 1697, 2663, 560, 1746, 2665, 564, 1740, 2665, 561, 1702, 2665, 558, 1697, 2655, 
			553, 1680, 2667, 546, 1668, 2692, 528, 1673, 2710, 514, 1629, 2717, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m09ei"}, {318, 122, 446, 1918, 2674, 446, 1918, 2674, 434, 1944, 2673, 411, 2002, 2693, 378, 2056, 2705, 
			373, 2068, 2716, 372, 2081, 2683, 373, 2083, 2713, 372, 2083, 2759, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m09er"}, {342, 137, 466, 1377, 1616, 494, 1381, 1660, 448, 1385, 1620, 438, 1416, 1676, 439, 1449, 1684, 
			439, 1489, 1679, 446, 1519, 1672, 442, 1544, 1725, 390, 1579, 1740, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m09ih"}, {221, 132, 385, 1843, 2696, 390, 1859, 2695, 385, 1843, 2696, 388, 1836, 2701, 391, 1827, 2700, 
			406, 1815, 2700, 415, 1800, 2698, 415, 1752, 2703, 408, 1726, 2727, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m09iy"}, {341, 129, 355, 2049, 2921, 371, 1998, 2835, 364, 2042, 2912, 349, 2055, 2922, 340, 2057, 2925, 
			333, 2074, 2922, 318, 2106, 2921, 315, 2120, 2912, 314, 2122, 2916, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m09oa"}, {340, 134, 503, 950, 2680, 525, 965, 2635, 503, 963, 2687, 494, 934, 2672, 459, 937, 2680, 
			437, 921, 2638, 436, 869, 2686, 435, 879, 2651, 437, 948, 2616, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m09oo"}, {238, 125, 442, 1205, 2580, 442, 1205, 2580, 453, 1172, 2617, 463, 1238, 2617, 471, 1304, 2601, 
			472, 1371, 2619, 461, 1436, 2628, 453, 1454, 2633, 446, 1510, 2676, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"m09uh"}, {203, 121, 602, 1336, 2690, 602, 1336, 2690, 598, 1320, 2685, 592, 1334, 2685, 568, 1379, 2695, 
			561, 1416, 2694, 546, 1433, 2691, 520, 1447, 2704, 517, 1489, 2711, 90, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 18, 0}},
		{{"m09uw"}, {296, 136, 384, 1234, 2418, 383, 1261, 2433, 385, 1234, 2408, 396, 1221, 2360, 381, 1183, 2360, 
			380, 1124, 2368, 387, 1100, 2396, 379, 1102, 2402, 382, 1106, 2389, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m10ae"}, {282, 153, 552, 2027, 2737, 563, 2011, 2737, 533, 2027, 2721, 501, 2000, 2664, 506, 1991, 2664, 
			570, 1941, 2711, 613, 1830, 2707, 631, 1555, 2665, 637, 1517, 2718, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m10ah"}, {271, 152, 763, 1147, 2840, 762, 1144, 2809, 763, 1147, 2840, 767, 1155, 2720, 769, 1192, 2676, 
			762, 1228, 2548, 744, 1244, 2231, 710, 1310, 2261, 683, 1400, 2297, 85, 0, 0, 0, 0, 0, 17, 2, 0, 0, 0, 0, 1}},
		{{"m10aw"}, {296, 151, 637, 923, 2722, 678, 993, 2694, 645, 972, 2698, 634, 932, 2715, 636, 924, 2720, 
			633, 929, 2699, 631, 951, 2681, 629, 990, 2668, 613, 1020, 2574, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"m10eh"}, {186, 154, 594, 1716, 2678, 575, 1726, 2690, 594, 1716, 2678, 597, 1715, 2678, 602, 1697, 2693, 
			608, 1680, 2693, 607, 1656, 2692, 583, 1644, 2687, 567, 1644, 2687, 95, 0, 0, 0, 19, 0, 0, 0, 0, 1, 0, 0, 0}},
		{{"m10ei"}, {273, 151, 439, 2067, 2708, 445, 2041, 2691, 439, 2053, 2697, 439, 2067, 2705, 436, 2076, 2708, 
			435, 2093, 2691, 425, 2112, 2687, 389, 2122, 2685, 367, 2129, 2688, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m10er"}, {249, 154, 459, 1367, 1716, 497, 1320, 1697, 473, 1367, 1697, 446, 1382, 1727, 445, 1390, 1727, 
			442, 1426, 1761, 443, 1440, 1786, 443, 1482, 1792, 437, 1548, 1813, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m10ih"}, {186, 153, 440, 1954, 2587, 436, 1962, 2585, 442, 1954, 2597, 443, 1902, 2591, 444, 1873, 2584, 
			447, 1863, 2570, 449, 1767, 2568, 450, 1723, 2576, 452, 1723, 2576, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m10iy"}, {223, 163, 320, 2300, 3038, 332, 2360, 3051, 325, 2324, 3066, 319, 2291, 3019, 320, 2309, 3019, 
			318, 2309, 2986, 319, 2308, 2944, 320, 2288, 2905, 323, 2262, 2872, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m10oa"}, {269, 153, 476, 1016, 2502, 510, 1019, 2515, 498, 1013, 2524, 465, 1016, 2502, 449, 992, 2503, 
			444, 944, 2528, 444, 928, 2494, 441, 932, 2485, 427, 993, 2451, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m10oo"}, {196, 157, 485, 1086, 2556, 487, 1104, 2542, 486, 1100, 2556, 466, 1086, 2545, 462, 1093, 2544, 
			459, 1097, 2499, 459, 1107, 2491, 468, 1287, 2512, 468, 1480, 2590, 90, 0, 0, 0, 0, 0, 0, 0, 0, 18, 2, 0, 0}},
		{{"m10uh"}, {183, 157, 658, 1410, 2720, 658, 1410, 2720, 667, 1379, 2714, 656, 1332, 2694, 630, 1300, 2641, 
			615, 1285, 2628, 610, 1315, 2628, 571, 1480, 2621, 550, 1502, 2561, 90, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 18, 0}},
		{{"m10uw"}, {237, 156, 338, 1087, 2515, 400, 1308, 2529, 382, 1210, 2475, 359, 1122, 2453, 338, 1087, 2515, 
			328, 1080, 2513, 324, 1088, 2525, 320, 1072, 2462, 317, 1108, 2342, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m11ae"}, {319, 136, 629, 1871, 2574, 636, 1887, 2568, 635, 1879, 2612, 626, 1886, 2609, 619, 1849, 2644, 
			622, 1885, 2555, 633, 1847, 2554, 651, 1810, 2504, 679, 1736, 2582, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m11ah"}, {321, 148, 829, 1444, 2241, 784, 1418, 2185, 807, 1382, 2167, 816, 1432, 2209, 832, 1447, 2241, 
			834, 1470, 2244, 703, 1494, 2338, 682, 1504, 2432, 670, 1590, 2526, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"m11aw"}, {363, 162, 695, 1108, 2384, 724, 1052, 2373, 702, 1095, 2378, 693, 1106, 2384, 690, 1107, 2372, 
			682, 1104, 2349, 678, 1141, 2311, 673, 1156, 2312, 677, 1185, 2296, 55, 0, 0, 0, 0, 0, 9, 11, 0, 0, 0, 0, 0}},
		{{"m11eh"}, {261, 146, 569, 1709, 2502, 580, 1705, 2477, 572, 1704, 2497, 571, 1702, 2502, 569, 1708, 2493, 
			569, 1704, 2505, 568, 1709, 2510, 567, 1707, 2521, 564, 1702, 2546, 90, 0, 0, 0, 18, 2, 0, 0, 0, 0, 0, 0, 0}},
		{{"m11ei"}, {360, 141, 433, 2082, 2620, 437, 2062, 2624, 433, 2090, 2618, 434, 2113, 2596, 434, 2149, 2620, 
			435, 2115, 2601, 435, 2207, 2680, 430, 2207, 2723, 407, 2207, 2687, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m11er"}, {340, 140, 438, 1387, 1716, 493, 1384, 1717, 452, 1317, 1726, 446, 1342, 1725, 440, 1380, 1696, 
			438, 1387, 1716, 435, 1398, 1712, 431, 1420, 1667, 430, 1436, 1666, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m11ih"}, {301, 163, 470, 1959, 2697, 442, 2089, 2790, 458, 1992, 2717, 474, 1950, 2697, 480, 1944, 2663, 
			481, 1945, 2646, 471, 1954, 2654, 473, 1949, 2665, 478, 1902, 2660, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m11iy"}, {323, 160, 315, 2253, 2886, 317, 2259, 2864, 318, 2256, 2845, 317, 2259, 2920, 315, 2253, 2886, 
			314, 2270, 2886, 316, 2275, 2852, 318, 2288, 2762, 318, 2271, 2740, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m11oa"}, {374, 149, 496, 867, 2380, 494, 866, 2372, 488, 870, 2376, 470, 843, 2394, 453, 782, 2360, 
			449, 737, 2329, 435, 668, 2384, 388, 678, 2382, 374, 724, 2341, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m11oo"}, {275, 151, 472, 1296, 2273, 455, 1226, 2230, 466, 1248, 2281, 472, 1296, 2273, 478, 1318, 2270, 
			472, 1305, 2274, 474, 1307, 2333, 486, 1388, 2328, 491, 1441, 2360, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 0, 0, 1}},
		{{"m11uh"}, {295, 155, 619, 1241, 2455, 613, 1212, 2422, 620, 1232, 2437, 616, 1239, 2441, 618, 1241, 2446, 
			621, 1241, 2439, 623, 1250, 2431, 625, 1264, 2433, 613, 1332, 2413, 90, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 18, 0}},
		{{"m11uw"}, {327, 161, 319, 936, 2187, 300, 918, 2126, 317, 893, 2159, 314, 928, 2134, 313, 932, 2178, 
			314, 931, 2125, 319, 936, 2187, 321, 939, 2197, 323, 988, 2144, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m13ae"}, {279, 100, 685, 1807, 2611, 687, 1826, 2684, 687, 1803, 2642, 685, 1807, 2611, 687, 1815, 2608, 
			685, 1795, 2619, 693, 1751, 2658, 693, 1735, 2704, 710, 1748, 2790, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m13ah"}, {256, 104, 825, 1429, 2701, 759, 1348, 2682, 789, 1413, 2635, 794, 1411, 2683, 825, 1429, 2707, 
			802, 1413, 2703, 796, 1379, 2691, 737, 1441, 2705, 702, 1542, 2739, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"m13aw"}, {283, 104, 664, 1024, 2639, 654, 1022, 2564, 653, 1022, 2571, 677, 1038, 2619, 674, 1023, 2628, 
			673, 1013, 2634, 680, 1037, 2657, 674, 1049, 2678, 664, 1114, 2609, 90, 0, 0, 0, 0, 0, 2, 18, 0, 0, 0, 0, 0}},
		{{"m13eh"}, {203, 99, 577, 1811, 2559, 587, 1832, 2725, 587, 1833, 2660, 577, 1811, 2559, 611, 1791, 2573, 
			621, 1776, 2558, 626, 1750, 2603, 613, 1746, 2720, 563, 1747, 2764, 65, 0, 0, 0, 13, 7, 0, 0, 0, 0, 0, 0, 0}},
		{{"m13ei"}, {243, 95, 439, 2126, 2762, 491, 2003, 2723, 477, 2012, 2696, 464, 2090, 2759, 450, 2121, 2764, 
			439, 2126, 2762, 433, 2142, 2797, 415, 2136, 2815, 386, 2142, 2841, 95, 0, 0, 19, 0, 0, 1, 0, 0, 0, 0, 0, 0}},
		{{"m13er"}, {268, 110, 439, 1220, 1681, 453, 1240, 1800, 448, 1220, 1693, 440, 1217, 1691, 439, 1220, 1681, 
			437, 1199, 1677, 437, 1209, 1675, 437, 1247, 1700, 437, 1289, 1691, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m13ih"}, {164, 109, 411, 2034, 2795, 425, 2053, 2780, 432, 2161, 2824, 435, 2152, 2816, 434, 2093, 2787, 
			435, 2026, 2790, 436, 1988, 2768, 438, 1940, 2806, 438, 1893, 2767, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m13iy"}, {229, 106, 327, 2257, 2943, 348, 2202, 2932, 358, 2222, 2933, 353, 2250, 2930, 343, 2253, 2951, 
			348, 2242, 2945, 327, 2257, 2955, 333, 2255, 2965, 317, 2260, 2925, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m13oa"}, {268, 104, 486, 826, 2631, 486, 844, 2631, 485, 824, 2624, 491, 839, 2669, 462, 804, 2719, 
			440, 762, 2761, 435, 750, 2684, 423, 779, 2670, 420, 890, 2612, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m13oo"}, {208, 113, 459, 1036, 2625, 451, 1020, 2763, 452, 1038, 2727, 455, 1037, 2635, 465, 1044, 2643, 
			477, 1105, 2621, 498, 1179, 2580, 504, 1345, 2572, 499, 1481, 2606, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m13uh"}, {172, 107, 636, 1153, 2756, 635, 1135, 2730, 631, 1162, 2760, 631, 1171, 2752, 631, 1214, 2755, 
			629, 1236, 2765, 625, 1282, 2788, 610, 1393, 2754, 568, 1461, 2741, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"m13uw"}, {240, 116, 371, 793, 2719, 369, 748, 2717, 371, 791, 2690, 375, 780, 2730, 368, 755, 2628, 
			362, 746, 2772, 370, 775, 2754, 364, 760, 2814, 358, 803, 2756, 75, 0, 0, 0, 0, 0, 0, 0, 4, 1, 15, 0, 0}},
		{{"m14ae"}, {272, 106, 559, 1983, 2560, 557, 1993, 2582, 558, 1997, 2570, 559, 1983, 2560, 571, 1964, 2555, 
			608, 1877, 2520, 628, 1786, 2504, 629, 1718, 2537, 618, 1708, 2550, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m14ah"}, {278, 100, 737, 1298, 2323, 733, 1290, 2379, 737, 1298, 2323, 738, 1316, 2336, 733, 1358, 2328, 
			723, 1369, 2306, 706, 1372, 2336, 685, 1405, 2373, 657, 1492, 2397, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"m14aw"}, {248, 108, 664, 1043, 2575, 657, 1016, 2615, 656, 1037, 2579, 664, 1043, 2575, 657, 1037, 2570, 
			659, 1059, 2584, 657, 1073, 2563, 663, 1142, 2514, 629, 1251, 2494, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"m14eh"}, {231, 110, 570, 1824, 2627, 588, 1821, 2613, 570, 1824, 2627, 573, 1823, 2620, 577, 1825, 2612, 
			576, 1806, 2602, 565, 1801, 2578, 563, 1839, 2580, 556, 1872, 2619, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m14ei"}, {256, 109, 495, 2166, 2668, 507, 2142, 2665, 492, 2206, 2671, 469, 2220, 2684, 437, 2253, 2683, 
			436, 2285, 2699, 434, 2317, 2677, 434, 2372, 2677, 420, 2369, 2658, 95, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m14er"}, {287, 106, 506, 1464, 1672, 558, 1507, 1688, 520, 1471, 1672, 514, 1502, 1658, 498, 1459, 1675, 
			496, 1460, 1692, 477, 1469, 1707, 446, 1554, 1806, 437, 1654, 2057, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m14ih"}, {249, 121, 444, 2154, 2659, 448, 2139, 2622, 482, 2102, 2630, 491, 2071, 2642, 495, 1997, 2615, 
			497, 1931, 2590, 499, 1877, 2570, 501, 1876, 2524, 493, 1868, 2619, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m14iy"}, {256, 106, 329, 2514, 3073, 357, 2510, 3035, 329, 2504, 3085, 328, 2502, 3006, 323, 2499, 2998, 
			331, 2494, 2947, 326, 2492, 2977, 344, 2466, 2954, 338, 2424, 2949, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m14oa"}, {269, 107, 491, 1065, 2383, 498, 1055, 2346, 491, 1065, 2375, 489, 1059, 2375, 466, 1059, 2390, 
			447, 1015, 2374, 444, 1017, 2372, 440, 1046, 2362, 439, 1096, 2323, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m14oo"}, {203, 116, 446, 1298, 2275, 440, 1316, 2286, 448, 1298, 2302, 452, 1307, 2332, 481, 1335, 2332, 
			484, 1375, 2300, 485, 1428, 2268, 487, 1485, 2260, 486, 1578, 2220, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m14uh"}, {227, 105, 631, 1133, 2560, 628, 1146, 2556, 631, 1103, 2555, 654, 1092, 2554, 640, 1147, 2545, 
			626, 1257, 2516, 635, 1411, 2478, 620, 1489, 2442, 603, 1642, 2482, 85, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 17, 0}},
		{{"m14uw"}, {260, 115, 379, 1123, 2383, 435, 1296, 2422, 422, 1221, 2442, 398, 1163, 2418, 387, 1153, 2399, 
			382, 1123, 2384, 376, 1123, 2377, 374, 1126, 2344, 374, 1149, 2318, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m16ae"}, {212, 138, 544, 1819, 2511, 554, 1785, 2458, 549, 1816, 2511, 540, 1809, 2487, 541, 1798, 2450, 
			551, 1773, 2457, 557, 1740, 2467, 560, 1678, 2472, 564, 1633, 2516, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"m16ah"}, {192, 145, 679, 1208, 2630, 683, 1220, 2634, 679, 1208, 2630, 679, 1185, 2629, 649, 1182, 2580, 
			645, 1215, 2571, 632, 1274, 2552, 627, 1320, 2537, 634, 1369, 2553, 25, 0, 0, 0, 0, 0, 5, 1, 0, 0, 0, 14, 0}},
		{{"m16aw"}, {223, 142, 628, 1069, 2689, 630, 1075, 2679, 628, 1071, 2683, 625, 1067, 2688, 621, 1066, 2670, 
			619, 1068, 2646, 614, 1094, 2638, 610, 1101, 2633, 609, 1208, 2591, 65, 0, 0, 0, 0, 0, 1, 13, 0, 0, 0, 6, 0}},
		{{"m16eh"}, {156, 144, 546, 1750, 2509, 532, 1748, 2499, 546, 1750, 2509, 547, 1742, 2540, 541, 1738, 2533, 
			538, 1733, 2531, 547, 1701, 2535, 554, 1688, 2552, 553, 1621, 2559, 95, 0, 0, 0, 19, 0, 0, 1, 0, 0, 0, 0, 0}},
		{{"m16ei"}, {218, 143, 482, 1904, 2526, 482, 1904, 2526, 437, 1964, 2495, 434, 1999, 2497, 426, 2016, 2478, 
			413, 2084, 2478, 402, 2080, 2487, 395, 2087, 2504, 391, 2055, 2500, 95, 0, 0, 19, 0, 0, 0, 1, 0, 0, 0, 0, 0}},
		{{"m16er"}, {204, 147, 439, 1387, 1746, 452, 1397, 1992, 444, 1370, 1849, 443, 1376, 1832, 440, 1395, 1756, 
			439, 1386, 1746, 438, 1404, 1749, 437, 1398, 1749, 435, 1431, 1746, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m16ih"}, {134, 145, 424, 1913, 2556, 392, 1930, 2571, 422, 1921, 2552, 424, 1913, 2556, 426, 1896, 2561, 
			434, 1869, 2562, 436, 1834, 2552, 438, 1723, 2541, 437, 1627, 2556, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m16iy"}, {179, 142, 311, 2274, 2510, 314, 2215, 2560, 311, 2248, 2540, 310, 2257, 2533, 312, 2259, 2538, 
			311, 2254, 2518, 312, 2229, 2540, 316, 2203, 2544, 319, 2128, 2566, 95, 19, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0}},
		{{"m16oa"}, {214, 142, 434, 862, 2561, 486, 1003, 2600, 447, 945, 2573, 438, 875, 2566, 434, 862, 2566, 
			433, 861, 2551, 429, 868, 2519, 426, 932, 2506, 425, 997, 2493, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m16oo"}, {150, 145, 438, 1177, 2478, 440, 1177, 2483, 438, 1168, 2470, 437, 1157, 2463, 437, 1179, 2458, 
			438, 1222, 2452, 439, 1280, 2449, 443, 1364, 2472, 442, 1438, 2481, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m16uh"}, {146, 147, 573, 1152, 2604, 596, 1165, 2599, 573, 1152, 2604, 570, 1153, 2596, 575, 1174, 2578, 
			575, 1200, 2573, 565, 1276, 2545, 563, 1382, 2548, 550, 1435, 2560, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 19, 0}},
		{{"m16uw"}, {183, 148, 382, 994, 2372, 403, 1104, 2338, 398, 995, 2367, 392, 996, 2371, 382, 994, 2371, 
			382, 994, 2372, 377, 1004, 2364, 376, 1033, 2353, 376, 1060, 2352, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m17ae"}, {257, 135, 685, 1592, 2308, 695, 1641, 2286, 686, 1602, 2304, 684, 1561, 2300, 688, 1568, 2298, 
			682, 1695, 2295, 664, 1761, 2278, 634, 1690, 2306, 601, 1693, 2391, 75, 0, 0, 0, 5, 15, 0, 0, 0, 0, 0, 0, 0}},
		{{"m17ah"}, {243, 132, 689, 1064, 2303, 651, 1042, 2231, 654, 973, 2280, 678, 1012, 2281, 689, 1064, 2303, 
			686, 1001, 2274, 680, 988, 2275, 662, 1152, 2305, 641, 1376, 2324, 75, 0, 0, 0, 0, 0, 15, 4, 0, 0, 0, 1, 0}},
		{{"m17aw"}, {278, 142, 634, 987, 2305, 655, 978, 2176, 636, 987, 2286, 634, 985, 2302, 634, 979, 2332, 
			630, 975, 2307, 629, 961, 2303, 624, 1047, 2238, 599, 1224, 2310, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"m17eh"}, {181, 138, 554, 1790, 2469, 569, 1846, 2458, 558, 1806, 2450, 549, 1802, 2433, 554, 1799, 2421, 
			556, 1778, 2417, 554, 1790, 2469, 549, 1808, 2472, 501, 1814, 2525, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m17ei"}, {259, 148, 435, 2013, 2506, 490, 2027, 2553, 446, 2067, 2557, 435, 2018, 2532, 437, 2010, 2486, 
			436, 2044, 2500, 430, 2045, 2607, 411, 2043, 2656, 374, 2074, 2600, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m17er"}, {256, 140, 429, 1389, 1830, 431, 1451, 1943, 430, 1394, 1845, 427, 1389, 1829, 429, 1395, 1822, 
			431, 1426, 1870, 433, 1472, 1855, 433, 1532, 1817, 434, 1578, 1848, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m17ih"}, {169, 145, 419, 2076, 2602, 385, 2067, 2575, 419, 2076, 2602, 424, 2017, 2554, 433, 2017, 2561, 
			436, 2008, 2578, 437, 1952, 2555, 436, 1905, 2571, 432, 1892, 2568, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m17iy"}, {261, 165, 322, 2282, 2746, 316, 2249, 3162, 316, 2251, 2916, 320, 2284, 2808, 323, 2294, 2735, 
			321, 2290, 2709, 321, 2284, 2769, 322, 2292, 2821, 324, 2306, 2851, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m17oa"}, {272, 146, 439, 699, 2107, 490, 908, 2052, 447, 860, 2111, 439, 769, 2101, 439, 699, 2107, 
			436, 722, 2112, 432, 763, 2102, 420, 794, 2094, 407, 842, 2071, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m17oo"}, {178, 148, 438, 979, 2187, 443, 938, 2203, 441, 950, 2198, 439, 963, 2195, 438, 979, 2187, 
			438, 980, 2172, 440, 976, 2112, 435, 1174, 2103, 430, 1380, 2148, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"m17uh"}, {176, 152, 663, 1166, 2177, 684, 1164, 2175, 663, 1166, 2177, 642, 1157, 2218, 620, 1135, 2262, 
			617, 1166, 2306, 605, 1199, 2349, 587, 1366, 2415, 559, 1444, 2427, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"m17uw"}, {210, 174, 339, 860, 2013, 337, 878, 1983, 338, 860, 2000, 348, 866, 2029, 359, 867, 1999, 
			367, 837, 1999, 363, 817, 2016, 355, 839, 2077, 340, 868, 2084, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m18ae"}, {264, 133, 514, 2031, 2815, 511, 2013, 2800, 514, 2031, 2815, 514, 2083, 2828, 523, 2065, 2851, 
			545, 2045, 2800, 569, 1974, 2796, 622, 1898, 2807, 628, 1870, 2796, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"m18ah"}, {342, 135, 702, 1364, 2498, 702, 1364, 2498, 732, 1418, 2484, 719, 1423, 2574, 725, 1412, 2502, 
			707, 1404, 2493, 718, 1435, 2499, 727, 1421, 2510, 684, 1506, 2652, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"m18aw"}, {320, 146, 662, 1001, 2744, 636, 991, 2802, 651, 993, 2771, 662, 1001, 2744, 684, 1006, 2736, 
			698, 1051, 2780, 705, 1065, 2727, 720, 1101, 2714, 694, 1170, 2677, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"m18eh"}, {170, 131, 608, 1831, 2694, 608, 1832, 2694, 619, 1829, 2724, 624, 1824, 2748, 624, 1811, 2774, 
			624, 1812, 2794, 623, 1813, 2774, 611, 1808, 2781, 565, 1804, 2803, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m18ei"}, {338, 151, 445, 2268, 2798, 520, 2104, 2664, 488, 2249, 2765, 447, 2276, 2794, 441, 2307, 2773, 
			438, 2436, 2786, 436, 2492, 2799, 428, 2397, 2812, 393, 2397, 2907, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m18er"}, {298, 139, 504, 1386, 1813, 537, 1267, 1993, 537, 1272, 1865, 527, 1352, 1816, 510, 1379, 1812, 
			504, 1388, 1810, 505, 1442, 1741, 501, 1447, 1769, 495, 1505, 1796, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m18ih"}, {179, 142, 491, 1958, 2782, 471, 1942, 2757, 484, 1950, 2764, 491, 1958, 2782, 496, 1942, 2782, 
			499, 1934, 2762, 503, 1935, 2762, 506, 1935, 2787, 504, 1922, 2822, 90, 0, 18, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m18iy"}, {273, 143, 364, 2366, 3042, 376, 2347, 3047, 367, 2353, 3057, 364, 2366, 3042, 335, 2388, 3063, 
			348, 2395, 3100, 352, 2386, 3129, 349, 2382, 3130, 364, 2387, 3116, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m18oa"}, {337, 138, 454, 745, 2623, 518, 866, 2584, 504, 859, 2681, 499, 789, 2701, 486, 737, 2637, 
			469, 746, 2630, 442, 745, 2614, 442, 825, 2630, 443, 864, 2678, 95, 0, 0, 0, 0, 0, 0, 0, 19, 0, 0, 1, 0}},
		{{"m18oo"}, {166, 158, 481, 1094, 2623, 464, 1130, 2587, 469, 1106, 2604, 483, 1070, 2625, 489, 1116, 2629, 
			508, 1227, 2632, 518, 1371, 2614, 519, 1421, 2604, 512, 1534, 2628, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m18uh"}, {193, 147, 610, 1168, 2529, 598, 1237, 2684, 616, 1214, 2606, 608, 1128, 2491, 612, 1145, 2453, 
			630, 1214, 2534, 633, 1317, 2635, 628, 1518, 2747, 623, 1617, 2841, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"m18uw"}, {264, 147, 415, 1016, 2465, 434, 1101, 2414, 426, 1044, 2438, 415, 1016, 2465, 390, 1036, 2487, 
			395, 1051, 2458, 390, 1108, 2492, 388, 1105, 2491, 377, 1113, 2484, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m19ae"}, {244, 149, 612, 1937, 2605, 611, 1935, 2708, 613, 1941, 2648, 613, 1927, 2577, 620, 1887, 2599, 
			633, 1828, 2603, 665, 1771, 2660, 676, 1719, 2656, 646, 1701, 2665, 80, 0, 0, 0, 4, 16, 0, 0, 0, 0, 0, 0, 0}},
		{{"m19ah"}, {269, 148, 811, 1355, 2599, 739, 1324, 2710, 756, 1349, 2700, 788, 1343, 2657, 812, 1352, 2610, 
			811, 1356, 2581, 805, 1416, 2533, 772, 1437, 2703, 698, 1497, 2666, 90, 0, 0, 0, 0, 0, 18, 1, 0, 0, 0, 1, 0}},
		{{"m19aw"}, {274, 154, 653, 1020, 2810, 653, 1020, 2810, 654, 998, 2816, 663, 1012, 2781, 671, 1026, 2772, 
			681, 1060, 2753, 687, 1155, 2689, 679, 1228, 2616, 632, 1401, 2569, 70, 0, 0, 0, 0, 0, 2, 14, 0, 1, 0, 3, 0}},
		{{"m19eh"}, {185, 151, 562, 1816, 2707, 566, 1812, 2652, 573, 1805, 2614, 578, 1786, 2666, 598, 1767, 2673, 
			610, 1717, 2644, 609, 1678, 2641, 604, 1665, 2643, 579, 1669, 2666, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m19ei"}, {246, 156, 492, 2195, 2786, 494, 2183, 2769, 483, 2202, 2821, 464, 2220, 2800, 449, 2236, 2771, 
			443, 2246, 2795, 438, 2293, 2912, 433, 2316, 2898, 417, 2337, 2840, 95, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m19er"}, {243, 154, 505, 1330, 1929, 505, 1330, 1929, 497, 1347, 1903, 489, 1370, 1848, 478, 1382, 1823, 
			454, 1436, 1813, 441, 1441, 1811, 432, 1509, 1810, 423, 1540, 1816, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m19ih"}, {194, 159, 486, 2061, 2719, 465, 2122, 2776, 486, 2029, 2698, 484, 1993, 2724, 472, 1961, 2721, 
			474, 1925, 2669, 500, 1876, 2791, 516, 1898, 2791, 519, 1874, 2783, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m19iy"}, {232, 160, 350, 2462, 2921, 355, 2423, 2838, 353, 2391, 2972, 356, 2443, 2978, 346, 2510, 2876, 
			350, 2492, 2919, 355, 2457, 2882, 356, 2489, 2865, 364, 2435, 2886, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m19oa"}, {247, 157, 566, 997, 2748, 558, 999, 2748, 567, 997, 2751, 562, 996, 2744, 550, 991, 2741, 
			533, 985, 2738, 505, 969, 2720, 491, 981, 2723, 454, 1003, 2684, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m19oo"}, {222, 163, 499, 1127, 2684, 499, 1127, 2684, 498, 1152, 2696, 497, 1215, 2685, 497, 1268, 2638, 
			499, 1376, 2564, 504, 1458, 2549, 507, 1506, 2620, 506, 1586, 2673, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"m19uh"}, {183, 154, 637, 1295, 2757, 637, 1295, 2761, 639, 1313, 2748, 636, 1344, 2712, 630, 1377, 2687, 
			627, 1446, 2687, 618, 1483, 2658, 598, 1501, 2599, 565, 1573, 2612, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"m19uw"}, {226, 165, 455, 1181, 2616, 461, 1181, 2616, 453, 1167, 2609, 448, 1139, 2577, 445, 1125, 2582, 
			443, 1113, 2593, 439, 1084, 2566, 437, 1091, 2538, 434, 1135, 2459, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19, 0, 0}},
		{{"m20ae"}, {286, 140, 544, 2114, 2971, 537, 2098, 2944, 544, 2114, 2971, 544, 2136, 2989, 553, 2010, 3062, 
			556, 1991, 3050, 572, 1773, 2998, 583, 1732, 2991, 587, 1735, 2986, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m20ah"}, {272, 135, 744, 1489, 2586, 746, 1502, 2500, 745, 1489, 2586, 743, 1486, 2567, 745, 1499, 2540, 
			753, 1506, 2515, 753, 1532, 2510, 723, 1549, 2528, 661, 1586, 2504, 95, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0}},
		{{"m20aw"}, {287, 138, 648, 1013, 2473, 668, 997, 2475, 648, 1019, 2468, 643, 1040, 2471, 636, 1059, 2472, 
			632, 1062, 2455, 629, 1100, 2432, 625, 1191, 2435, 596, 1432, 2555, 80, 0, 0, 0, 0, 0, 2, 16, 0, 0, 0, 2, 0}},
		{{"m20eh"}, {199, 137, 583, 1757, 2852, 619, 1759, 2830, 618, 1762, 2825, 606, 1764, 2849, 583, 1757, 2852, 
			576, 1747, 2826, 574, 1746, 2814, 567, 1750, 2807, 561, 1763, 2811, 95, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0}},
		{{"m20ei"}, {237, 141, 535, 2004, 2939, 546, 2000, 2944, 504, 2027, 2937, 465, 2072, 2947, 436, 2112, 2955, 
			433, 2135, 2971, 433, 2149, 2958, 432, 2146, 2931, 430, 2167, 2913, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m20er"}, {356, 156, 457, 1437, 1648, 511, 1446, 1738, 471, 1450, 1686, 449, 1448, 1628, 453, 1437, 1639, 
			466, 1439, 1658, 470, 1444, 1607, 474, 1448, 1608, 464, 1459, 1608, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m20ih"}, {221, 139, 433, 2012, 2855, 433, 2012, 2855, 433, 1995, 2857, 436, 1958, 2824, 446, 1923, 2813, 
			481, 1838, 2787, 490, 1797, 2769, 486, 1769, 2786, 459, 1774, 2814, 90, 0, 18, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m20iy"}, {263, 161, 314, 2368, 3378, 322, 2258, 3285, 311, 2284, 3331, 313, 2324, 3350, 314, 2348, 3367, 
			315, 2369, 3379, 316, 2355, 3369, 318, 2335, 3326, 319, 2320, 3317, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m20oa"}, {319, 133, 534, 1002, 2293, 545, 1000, 2288, 509, 995, 2312, 488, 898, 2321, 443, 870, 2360, 
			438, 850, 2381, 432, 837, 2388, 432, 849, 2423, 431, 876, 2440, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m20oo"}, {237, 144, 438, 1181, 2381, 438, 1181, 2378, 441, 1191, 2435, 445, 1206, 2436, 461, 1222, 2449, 
			490, 1347, 2479, 501, 1412, 2478, 501, 1501, 2462, 493, 1658, 2661, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m20uh"}, {189, 138, 623, 1163, 2555, 624, 1163, 2559, 623, 1165, 2555, 622, 1179, 2556, 618, 1208, 2580, 
			617, 1236, 2587, 598, 1340, 2574, 569, 1436, 2583, 562, 1529, 2592, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"m20uw"}, {284, 156, 370, 889, 2324, 430, 1005, 2274, 416, 1008, 2262, 380, 933, 2314, 375, 917, 2305, 
			372, 887, 2322, 368, 897, 2336, 361, 920, 2364, 351, 946, 2320, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m21ae"}, {252, 108, 556, 1854, 2679, 568, 1878, 2749, 559, 1877, 2695, 559, 1854, 2630, 559, 1854, 2663, 
			565, 1817, 2566, 619, 1754, 2576, 630, 1724, 2547, 643, 1646, 2582, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"m21ah"}, {206, 114, 758, 1363, 2421, 755, 1362, 2450, 758, 1363, 2401, 784, 1377, 2394, 799, 1359, 2376, 
			771, 1347, 2382, 742, 1352, 2377, 696, 1395, 2384, 691, 1432, 2408, 95, 0, 0, 0, 0, 0, 19, 0, 0, 0, 0, 1, 0}},
		{{"m21aw"}, {253, 107, 631, 1019, 2219, 630, 1036, 2220, 629, 1013, 2206, 633, 1031, 2239, 633, 1042, 2245, 
			652, 1007, 2245, 645, 1096, 2222, 674, 1233, 2212, 667, 1350, 2168, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"m21eh"}, {138, 128, 615, 1624, 2265, 559, 1674, 2256, 578, 1633, 2258, 595, 1627, 2259, 615, 1624, 2265, 
			619, 1640, 2274, 619, 1641, 2272, 592, 1630, 2275, 558, 1611, 2300, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m21ei"}, {196, 126, 382, 2035, 0, 491, 1974, 2336, 486, 2006, 0, 471, 2015, 0, 448, 2023, 0, 
			434, 2036, 0, 422, 2050, 0, 392, 2036, 0, 382, 2035, 0, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m21er"}, {185, 122, 463, 1371, 1684, 491, 1365, 1679, 478, 1368, 1679, 466, 1367, 1683, 462, 1377, 1684, 
			462, 1409, 1699, 460, 1434, 1711, 446, 1457, 1725, 430, 1498, 1726, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m21ih"}, {145, 123, 418, 1964, 2432, 418, 1964, 2432, 409, 1943, 2423, 441, 1901, 2429, 472, 1853, 2433, 
			494, 1817, 2397, 496, 1788, 2418, 499, 1745, 2422, 499, 1742, 2436, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m21iy"}, {153, 129, 328, 2123, 3444, 333, 2119, 3528, 352, 2118, 3455, 331, 2121, 3457, 323, 2129, 3435, 
			313, 2139, 3524, 308, 2178, 3572, 314, 2192, 3592, 310, 2150, 3626, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m21oa"}, {204, 127, 499, 859, 2080, 530, 904, 2072, 524, 871, 2076, 503, 875, 2082, 499, 859, 2080, 
			500, 862, 2077, 501, 796, 2084, 446, 789, 2095, 429, 810, 2128, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m21oo"}, {147, 119, 451, 1106, 2195, 451, 1106, 2195, 448, 1122, 2198, 449, 1137, 2192, 455, 1158, 2189, 
			474, 1195, 2190, 484, 1291, 2188, 484, 1374, 2174, 483, 1474, 2126, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m21uh"}, {139, 120, 593, 1254, 2302, 593, 1254, 2308, 593, 1254, 2302, 598, 1267, 2290, 613, 1284, 2273, 
			619, 1294, 2270, 621, 1357, 2248, 615, 1399, 2242, 586, 1481, 2232, 90, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 18, 0}},
		{{"m21uw"}, {196, 116, 372, 1010, 2136, 376, 1067, 2133, 378, 1052, 2101, 377, 1038, 2131, 374, 1032, 2137, 
			372, 1010, 2136, 367, 1008, 2150, 366, 1006, 2092, 368, 1053, 2060, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m22ae"}, {249, 119, 619, 2023, 2716, 622, 2038, 2683, 619, 2023, 2716, 621, 2013, 2703, 621, 2004, 2723, 
			632, 1983, 2699, 667, 1927, 2671, 686, 1841, 2615, 682, 1798, 2621, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m22ah"}, {287, 115, 784, 1345, 2522, 812, 1329, 2585, 808, 1359, 2568, 796, 1347, 2527, 784, 1353, 2522, 
			804, 1395, 2506, 797, 1402, 2462, 780, 1445, 2462, 760, 1544, 2507, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"m22aw"}, {316, 120, 688, 1139, 2549, 688, 1143, 2557, 693, 1127, 2541, 709, 1127, 2482, 737, 1133, 2491, 
			751, 1174, 2526, 743, 1180, 2611, 758, 1333, 2435, 727, 1408, 2479, 45, 0, 0, 0, 0, 0, 11, 9, 0, 0, 0, 0, 0}},
		{{"m22eh"}, {184, 122, 630, 1786, 2637, 632, 1789, 2634, 627, 1786, 2635, 626, 1793, 2628, 624, 1770, 2626, 
			626, 1758, 2631, 625, 1750, 2644, 624, 1738, 2638, 617, 1746, 2663, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m22ei"}, {251, 121, 499, 2068, 2676, 507, 2069, 2696, 499, 2068, 2676, 495, 2068, 2673, 486, 2082, 2656, 
			480, 2111, 2674, 478, 2134, 2707, 471, 2135, 2675, 447, 2152, 2636, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m22er"}, {256, 120, 491, 1455, 1896, 500, 1469, 1914, 491, 1455, 1896, 485, 1447, 1871, 478, 1460, 1836, 
			469, 1493, 1843, 462, 1505, 1865, 457, 1501, 1856, 450, 1502, 1834, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m22ih"}, {220, 128, 452, 2062, 2736, 452, 2062, 2736, 458, 2063, 2709, 482, 2027, 2693, 484, 1997, 2723, 
			488, 1938, 2686, 491, 1882, 2668, 493, 1834, 2690, 492, 1783, 2708, 85, 0, 17, 0, 3, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m22iy"}, {251, 127, 372, 2280, 2710, 377, 2322, 2810, 372, 2298, 2786, 370, 2306, 2685, 371, 2303, 2812, 
			372, 2280, 2710, 372, 2291, 2723, 372, 2281, 2770, 373, 2273, 2729, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m22oa"}, {287, 119, 520, 987, 2573, 550, 1023, 2564, 534, 1009, 2555, 520, 987, 2573, 507, 942, 2571, 
			498, 936, 2597, 488, 933, 2591, 466, 928, 2594, 454, 923, 2608, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m22oo"}, {173, 124, 470, 1202, 2555, 449, 1178, 2511, 458, 1191, 2529, 470, 1202, 2555, 483, 1208, 2541, 
			494, 1240, 2497, 496, 1297, 2485, 497, 1398, 2467, 496, 1495, 2440, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m22uh"}, {226, 122, 622, 1155, 2606, 623, 1146, 2606, 623, 1179, 2611, 625, 1184, 2610, 635, 1204, 2610, 
			637, 1279, 2558, 638, 1315, 2560, 633, 1378, 2564, 622, 1444, 2593, 95, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 19, 0}},
		{{"m22uw"}, {275, 128, 387, 1123, 2426, 395, 1122, 2428, 381, 1119, 2419, 380, 1107, 2425, 380, 1082, 2430, 
			381, 1073, 2430, 382, 1064, 2424, 381, 1035, 2417, 378, 1054, 2390, 95, 0, 0, 0, 0, 0, 0, 0, 1, 0, 19, 0, 0}},
		{{"m23ae"}, {274, 112, 601, 1997, 2499, 602, 2008, 2458, 601, 1997, 2499, 593, 1997, 2516, 610, 1971, 2522, 
			628, 1904, 2515, 671, 1814, 2571, 685, 1749, 2634, 665, 1748, 2725, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m23ah"}, {278, 107, 802, 1297, 2765, 819, 1310, 2731, 790, 1295, 2771, 779, 1292, 2738, 764, 1265, 2745, 
			757, 1316, 2763, 743, 1365, 2769, 747, 1430, 2733, 703, 1542, 2809, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"m23aw"}, {292, 111, 696, 983, 2603, 695, 955, 2623, 687, 981, 2609, 685, 988, 2606, 697, 980, 2610, 
			693, 997, 2617, 701, 998, 2622, 701, 1045, 2626, 682, 1314, 2594, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"m23eh"}, {227, 111, 629, 1810, 2639, 616, 1827, 2666, 629, 1810, 2639, 633, 1790, 2635, 629, 1730, 2644, 
			630, 1654, 2604, 626, 1663, 2646, 615, 1674, 2672, 570, 1728, 2727, 80, 0, 0, 0, 16, 4, 0, 0, 0, 0, 0, 0, 0}},
		{{"m23ei"}, {311, 116, 477, 2110, 2563, 484, 2092, 2605, 480, 2102, 2563, 461, 2132, 2597, 444, 2138, 2561, 
			440, 2198, 2560, 441, 2251, 2525, 412, 2231, 2521, 377, 2197, 2533, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m23er"}, {274, 117, 504, 1173, 1595, 515, 1180, 1614, 500, 1179, 1577, 491, 1195, 1581, 485, 1226, 1571, 
			484, 1243, 1551, 484, 1268, 1540, 492, 1336, 1552, 485, 1387, 1641, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m23ih"}, {207, 113, 463, 2018, 2688, 470, 2029, 2697, 448, 2010, 2682, 442, 1998, 2693, 447, 1971, 2676, 
			449, 1953, 2680, 468, 1923, 2703, 486, 1905, 2684, 483, 1878, 2717, 95, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m23iy"}, {265, 121, 367, 2244, 3042, 367, 2244, 3042, 364, 2256, 3036, 357, 2256, 3018, 355, 2244, 3000, 
			360, 2230, 2955, 362, 2235, 2896, 365, 2228, 2892, 373, 2247, 2859, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m23oa"}, {279, 117, 461, 801, 2442, 549, 869, 2503, 503, 820, 2480, 483, 809, 2450, 463, 803, 2441, 
			466, 801, 2466, 462, 807, 2478, 448, 831, 2450, 444, 905, 2505, 95, 0, 0, 0, 1, 0, 0, 0, 19, 0, 0, 0, 0}},
		{{"m23oo"}, {204, 121, 503, 1058, 2518, 502, 1059, 2471, 502, 1056, 2499, 504, 1059, 2528, 507, 1069, 2552, 
			508, 1097, 2551, 506, 1141, 2598, 505, 1269, 2621, 503, 1369, 2637, 90, 0, 0, 0, 0, 0, 0, 0, 0, 18, 0, 2, 0}},
		{{"m23uh"}, {202, 114, 617, 1071, 2685, 624, 1046, 2708, 627, 1047, 2693, 617, 1071, 2685, 618, 1078, 2682, 
			624, 1131, 2659, 622, 1290, 2608, 610, 1387, 2593, 561, 1508, 2632, 85, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 17, 0}},
		{{"m23uw"}, {230, 132, 389, 959, 2441, 382, 1005, 2364, 382, 990, 2370, 387, 977, 2425, 399, 937, 2407, 
			396, 926, 2369, 393, 927, 2360, 391, 941, 2366, 408, 1023, 2278, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m24ae"}, {270, 120, 572, 2072, 2648, 572, 2072, 2648, 574, 2070, 2636, 597, 2053, 2622, 620, 1978, 2604, 
			625, 1880, 2576, 639, 1777, 2576, 639, 1697, 2602, 632, 1698, 2627, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m24ah"}, {258, 131, 803, 1234, 2430, 783, 1235, 2424, 810, 1227, 2443, 817, 1228, 2455, 822, 1245, 2457, 
			816, 1342, 2462, 808, 1409, 2471, 769, 1478, 2463, 721, 1571, 2501, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"m24aw"}, {248, 127, 691, 1111, 2439, 684, 1107, 2492, 691, 1111, 2439, 690, 1107, 2430, 700, 1099, 2424, 
			729, 1112, 2377, 746, 1175, 2365, 729, 1350, 2356, 684, 1484, 2393, 45, 0, 0, 0, 0, 0, 11, 9, 0, 0, 0, 0, 0}},
		{{"m24eh"}, {153, 119, 624, 1909, 2773, 616, 1877, 2721, 622, 1904, 2743, 624, 1909, 2773, 624, 1904, 2783, 
			620, 1911, 2796, 607, 1917, 2805, 584, 1905, 2816, 571, 1894, 2817, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m24ei"}, {274, 131, 529, 2114, 2731, 537, 2079, 2734, 529, 2117, 2733, 518, 2129, 2725, 508, 2185, 2709, 
			502, 2183, 2707, 492, 2190, 2693, 450, 2190, 2686, 436, 2191, 2684, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m24er"}, {258, 134, 513, 1342, 1965, 509, 1346, 1971, 517, 1345, 1950, 518, 1344, 1930, 511, 1390, 1916, 
			502, 1454, 1921, 496, 1494, 1934, 459, 1571, 1967, 445, 1595, 2004, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m24ih"}, {202, 138, 501, 1971, 2734, 499, 1929, 2714, 499, 1961, 2726, 506, 1996, 2744, 512, 2001, 2739, 
			514, 1986, 2735, 507, 1978, 2741, 500, 1971, 2739, 493, 1951, 2743, 50, 0, 10, 0, 10, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m24iy"}, {241, 133, 381, 2515, 3059, 379, 2510, 3053, 381, 2515, 3059, 379, 2515, 3040, 376, 2544, 2994, 
			373, 2540, 2942, 368, 2548, 2904, 358, 2525, 2882, 353, 2505, 2871, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m24oa"}, {233, 126, 543, 1009, 2326, 547, 1021, 2322, 534, 998, 2329, 529, 997, 2334, 518, 996, 2344, 
			505, 987, 2381, 500, 975, 2409, 499, 1024, 2432, 498, 1088, 2426, 95, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0}},
		{{"m24oo"}, {210, 131, 504, 1318, 2491, 505, 1315, 2459, 504, 1306, 2508, 506, 1306, 2519, 509, 1333, 2558, 
			516, 1401, 2556, 518, 1426, 2556, 511, 1524, 2550, 498, 1623, 2592, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m24uh"}, {201, 130, 628, 1223, 2609, 631, 1204, 2646, 627, 1230, 2611, 627, 1224, 2609, 631, 1233, 2605, 
			632, 1247, 2590, 629, 1355, 2576, 623, 1470, 2585, 571, 1650, 2623, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"m24uw"}, {239, 147, 435, 1137, 2395, 434, 1130, 2390, 437, 1128, 2394, 438, 1094, 2427, 433, 1112, 2465, 
			423, 1109, 2484, 397, 1102, 2501, 388, 1161, 2494, 393, 1307, 2477, 85, 0, 0, 0, 0, 0, 0, 0, 0, 3, 17, 0, 0}},
		{{"m25ae"}, {301, 115, 546, 1820, 2548, 554, 1784, 2573, 551, 1792, 2565, 555, 1810, 2543, 546, 1821, 2558, 
			554, 1819, 2565, 568, 1743, 2551, 577, 1690, 2558, 567, 1650, 2597, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m25ah"}, {256, 116, 683, 1238, 2428, 684, 1153, 2428, 682, 1176, 2399, 680, 1184, 2467, 684, 1243, 2434, 
			683, 1238, 2428, 672, 1244, 2331, 665, 1255, 2383, 634, 1342, 2404, 95, 0, 0, 0, 0, 0, 19, 0, 0, 0, 0, 1, 0}},
		{{"m25aw"}, {249, 109, 587, 941, 2371, 603, 931, 2369, 587, 941, 2371, 600, 926, 2376, 606, 934, 2372, 
			615, 974, 2345, 615, 1010, 2317, 613, 1030, 2329, 607, 1171, 2299, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"m25eh"}, {190, 109, 559, 1755, 2514, 565, 1752, 2506, 561, 1754, 2529, 563, 1737, 2514, 565, 1722, 2375, 
			562, 1708, 2418, 563, 1685, 2481, 562, 1699, 2560, 536, 1695, 2598, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m25ei"}, {267, 112, 494, 1962, 2608, 496, 1952, 2621, 461, 1967, 2655, 445, 2050, 2623, 446, 2072, 2609, 
			439, 2104, 2728, 420, 2291, 2794, 380, 2297, 2780, 375, 2248, 2721, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m25er"}, {260, 116, 474, 1505, 0, 478, 1542, 0, 474, 1505, 0, 474, 1467, 0, 486, 1437, 0, 
			487, 1473, 0, 480, 1535, 0, 445, 1517, 0, 429, 1540, 0, 95, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 19}},
		{{"m25ih"}, {176, 107, 427, 1974, 2608, 427, 1974, 2608, 429, 1952, 2587, 435, 1937, 2567, 439, 1890, 2582, 
			445, 1872, 2593, 459, 1842, 2606, 477, 1815, 2624, 482, 1768, 2627, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m25iy"}, {268, 119, 352, 2254, 2888, 333, 2233, 2891, 342, 2252, 2908, 348, 2253, 2894, 362, 2267, 2870, 
			368, 2298, 2867, 368, 2299, 2860, 366, 2287, 2870, 367, 2273, 2863, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m25oa"}, {261, 112, 494, 925, 2235, 498, 939, 2349, 485, 921, 2249, 474, 914, 2293, 465, 888, 2308, 
			484, 861, 2287, 476, 850, 2295, 443, 847, 2273, 429, 869, 2330, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m25oo"}, {216, 111, 507, 1183, 2309, 507, 1196, 2303, 507, 1183, 2309, 503, 1202, 2304, 504, 1204, 2301, 
			507, 1217, 2285, 511, 1287, 2284, 518, 1364, 2239, 515, 1464, 2298, 90, 0, 0, 0, 0, 0, 0, 0, 0, 18, 0, 2, 0}},
		{{"m25uh"}, {212, 113, 601, 1214, 2305, 601, 1214, 2305, 585, 1210, 2304, 578, 1245, 2281, 583, 1274, 2296, 
			594, 1303, 2263, 601, 1368, 2304, 595, 1433, 2311, 575, 1558, 2366, 80, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 16, 1}},
		{{"m25uw"}, {204, 123, 372, 972, 2316, 370, 1008, 2319, 372, 970, 2323, 372, 952, 2317, 371, 934, 2314, 
			371, 939, 2318, 372, 942, 2367, 372, 984, 2343, 372, 1008, 2379, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m26ae"}, {259, 120, 562, 1895, 2513, 570, 1857, 2546, 567, 1878, 2509, 564, 1887, 2510, 560, 1889, 2507, 
			561, 1858, 2483, 564, 1802, 2499, 565, 1723, 2506, 566, 1667, 2505, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"m26ah"}, {271, 124, 697, 1370, 2597, 711, 1398, 2431, 710, 1368, 2585, 697, 1370, 2597, 691, 1366, 2452, 
			690, 1351, 2461, 688, 1362, 2519, 684, 1367, 2433, 639, 1375, 2486, 95, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0}},
		{{"m26aw"}, {282, 111, 636, 978, 2220, 633, 1017, 2159, 629, 1007, 2195, 644, 983, 2240, 633, 1000, 2203, 
			635, 1046, 2273, 626, 1069, 2225, 622, 1111, 2251, 617, 1177, 2283, 80, 0, 0, 0, 0, 0, 4, 16, 0, 0, 0, 0, 0}},
		{{"m26eh"}, {207, 114, 609, 1675, 2495, 609, 1675, 2495, 613, 1671, 2495, 615, 1664, 2502, 621, 1641, 2500, 
			617, 1593, 2471, 609, 1576, 2501, 572, 1579, 2501, 567, 1599, 2548, 95, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0}},
		{{"m26ei"}, {262, 113, 504, 1878, 2549, 504, 1878, 2549, 489, 1900, 2543, 451, 1951, 2542, 437, 2026, 2527, 
			429, 2089, 2549, 418, 2117, 2515, 379, 2137, 2521, 373, 2148, 2555, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m26er"}, {252, 121, 505, 1450, 0, 513, 1448, 0, 500, 1450, 0, 493, 1434, 0, 479, 1465, 0, 
			445, 1436, 0, 440, 1460, 0, 439, 1477, 0, 437, 1460, 0, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m26ih"}, {197, 129, 439, 1951, 2622, 431, 1985, 2629, 435, 1965, 2622, 444, 1937, 2622, 493, 1878, 2589, 
			495, 1832, 2567, 463, 1757, 2570, 445, 1712, 2570, 438, 1691, 2596, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m26iy"}, {271, 115, 323, 2269, 2943, 343, 2290, 2994, 314, 2334, 2995, 311, 2300, 2995, 320, 2269, 2957, 
			327, 2286, 2943, 324, 2283, 2912, 322, 2256, 2900, 329, 2257, 2872, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m26oa"}, {275, 107, 436, 865, 2377, 544, 998, 2227, 499, 951, 2245, 482, 914, 2327, 442, 868, 2344, 
			437, 866, 2360, 434, 844, 2370, 433, 835, 2376, 427, 850, 2338, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m26oo"}, {208, 116, 492, 1157, 2454, 496, 1173, 2432, 489, 1157, 2448, 486, 1171, 2441, 496, 1186, 2417, 
			500, 1254, 2421, 504, 1320, 2429, 499, 1407, 2443, 496, 1515, 2474, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m26uh"}, {230, 116, 593, 1213, 2329, 605, 1253, 2331, 601, 1226, 2342, 593, 1211, 2324, 599, 1218, 2317, 
			599, 1231, 2315, 590, 1275, 2299, 597, 1328, 2276, 566, 1384, 2392, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"m26uw"}, {225, 123, 374, 921, 2288, 397, 958, 2273, 376, 937, 2280, 376, 932, 2288, 373, 914, 2297, 
			371, 917, 2313, 369, 928, 2305, 369, 930, 2309, 368, 973, 2371, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19, 0, 0}},
		{{"m27ae"}, {257, 105, 594, 2000, 2550, 581, 2052, 2582, 594, 2001, 2550, 616, 2002, 2537, 625, 1983, 2541, 
			638, 1863, 2513, 653, 1768, 2531, 663, 1753, 2511, 663, 1750, 2499, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m27ah"}, {212, 106, 743, 1423, 2494, 748, 1417, 2511, 720, 1423, 2477, 699, 1437, 2495, 720, 1446, 2492, 
			716, 1458, 2492, 710, 1489, 2482, 700, 1512, 2391, 680, 1611, 2433, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"m27aw"}, {171, 103, 686, 1064, 2432, 673, 988, 2526, 678, 1048, 2495, 687, 1053, 2440, 686, 1064, 2432, 
			692, 1060, 2438, 698, 1068, 2447, 703, 1072, 2449, 743, 1069, 2428, 90, 0, 0, 0, 0, 0, 2, 18, 0, 0, 0, 0, 0}},
		{{"m27eh"}, {215, 104, 634, 1874, 2563, 652, 1884, 2539, 634, 1871, 2566, 633, 1835, 2531, 632, 1806, 2514, 
			630, 1783, 2547, 625, 1719, 2567, 622, 1696, 2624, 588, 1749, 2634, 85, 0, 0, 0, 17, 2, 0, 1, 0, 0, 0, 0, 0}},
		{{"m27ei"}, {252, 115, 472, 2141, 2623, 509, 2176, 2635, 481, 2141, 2604, 440, 2130, 2595, 428, 2155, 2590, 
			391, 2196, 2604, 373, 2225, 2625, 372, 2254, 2608, 373, 2295, 2597, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m27er"}, {231, 116, 442, 1517, 1736, 500, 1542, 1804, 465, 1543, 1786, 447, 1515, 1764, 442, 1511, 1741, 
			442, 1546, 1734, 440, 1567, 1774, 428, 1582, 1823, 391, 1627, 1870, 95, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 19}},
		{{"m27ih"}, {188, 115, 453, 2013, 2660, 431, 2135, 2750, 436, 2045, 2697, 453, 2013, 2660, 465, 2002, 2656, 
			460, 1989, 2648, 460, 1955, 2642, 467, 1952, 2643, 472, 1942, 2693, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m27iy"}, {218, 117, 338, 2329, 2945, 358, 2323, 3087, 355, 2247, 3075, 350, 2268, 3081, 350, 2306, 3038, 
			346, 2317, 3030, 336, 2314, 3001, 337, 2323, 2965, 347, 2331, 2939, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m27oa"}, {246, 105, 533, 900, 2435, 533, 920, 2425, 506, 914, 2464, 495, 875, 2450, 470, 860, 2458, 
			442, 826, 2480, 440, 805, 2498, 434, 804, 2498, 433, 848, 2453, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m27oo"}, {173, 113, 487, 994, 2299, 487, 997, 2326, 494, 1001, 2353, 499, 1054, 2397, 502, 1108, 2430, 
			503, 1184, 2450, 504, 1287, 2437, 504, 1406, 2428, 499, 1521, 2435, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m27uh"}, {166, 105, 662, 1166, 2553, 679, 1167, 2526, 677, 1181, 2504, 656, 1218, 2523, 654, 1255, 2532, 
			644, 1287, 2525, 632, 1358, 2526, 621, 1431, 2512, 587, 1514, 2462, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"m27uw"}, {207, 123, 371, 950, 2264, 378, 972, 2292, 374, 1039, 2245, 373, 1009, 2259, 372, 963, 2283, 
			371, 950, 2264, 371, 968, 2292, 372, 995, 2284, 372, 1028, 2261, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m28ae"}, {212, 131, 662, 1934, 2649, 662, 1934, 2649, 640, 1940, 2698, 630, 1954, 2698, 626, 1941, 2663, 
			627, 1899, 2694, 628, 1821, 2727, 628, 1748, 2741, 625, 1728, 2711, 60, 0, 0, 0, 8, 12, 0, 0, 0, 0, 0, 0, 0}},
		{{"m28ah"}, {154, 173, 710, 1084, 2753, 733, 1150, 2758, 695, 1032, 2758, 692, 1016, 2774, 759, 1057, 2838, 
			762, 1069, 2861, 789, 1173, 2828, 813, 1237, 2814, 749, 1301, 2807, 85, 0, 0, 0, 0, 0, 17, 3, 0, 0, 0, 0, 0}},
		{{"m28aw"}, {222, 122, 681, 1060, 2863, 681, 1060, 2863, 703, 1060, 2828, 747, 1050, 2893, 774, 1120, 2827, 
			769, 1144, 2806, 786, 1281, 2826, 791, 1451, 2754, 637, 1501, 2752, 85, 0, 0, 0, 0, 0, 0, 17, 0, 0, 0, 3, 0}},
		{{"m28eh"}, {125, 128, 636, 1893, 2765, 644, 1960, 2767, 644, 1926, 2756, 636, 1893, 2765, 632, 1885, 2779, 
			629, 1865, 2761, 628, 1853, 2760, 620, 1816, 2762, 592, 1816, 2785, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m28ei"}, {231, 124, 490, 2187, 2722, 548, 2125, 2686, 497, 2134, 2774, 494, 2169, 2631, 492, 2224, 2790, 
			486, 2209, 2753, 445, 2200, 2695, 430, 2218, 2684, 394, 2243, 2661, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m28er"}, {201, 140, 496, 1500, 1711, 503, 1511, 1753, 487, 1491, 1689, 454, 1468, 1704, 439, 1468, 1694, 
			440, 1485, 1664, 444, 1506, 1685, 453, 1544, 1699, 480, 1503, 1744, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m28ih"}, {160, 189, 469, 2128, 2744, 469, 2128, 2744, 478, 2079, 2745, 497, 2060, 2814, 515, 1997, 2810, 
			509, 1965, 2760, 501, 1909, 2743, 501, 1894, 2727, 496, 1878, 2732, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m28iy"}, {194, 175, 368, 2509, 2893, 372, 2525, 3133, 373, 2482, 2840, 371, 2491, 2879, 357, 2509, 2920, 
			344, 2499, 2825, 375, 2485, 2821, 391, 2442, 2841, 374, 2415, 2776, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m28oa"}, {213, 130, 530, 1000, 2652, 544, 993, 2690, 508, 998, 2684, 502, 989, 2671, 496, 949, 2643, 
			486, 949, 2675, 484, 998, 2685, 481, 1084, 2624, 441, 1186, 2505, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m28oo"}, {146, 132, 435, 994, 2565, 447, 992, 2573, 435, 1016, 2565, 442, 1044, 2562, 481, 1144, 2560, 
			497, 1253, 2550, 497, 1401, 2506, 495, 1470, 2506, 493, 1672, 2516, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"m28uh"}, {168, 124, 682, 1192, 2774, 694, 1143, 2799, 657, 1210, 2769, 631, 1264, 2766, 627, 1325, 2765, 
			620, 1403, 2751, 597, 1499, 2738, 568, 1579, 2734, 532, 1674, 2729, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"m28uw"}, {197, 189, 377, 888, 2416, 379, 907, 2453, 374, 920, 2399, 372, 924, 2414, 358, 873, 2472, 
			362, 872, 2462, 382, 966, 2494, 375, 982, 2434, 370, 1014, 2438, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m29ae"}, {241, 114, 566, 2041, 2500, 606, 1975, 2605, 586, 2005, 2605, 568, 2027, 2529, 565, 2049, 2515, 
			565, 2040, 2546, 563, 1985, 2528, 565, 1943, 2554, 575, 1870, 2598, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m29ah"}, {220, 123, 825, 1438, 2434, 819, 1391, 2367, 822, 1432, 2382, 825, 1436, 2427, 825, 1437, 2465, 
			830, 1453, 2499, 819, 1491, 2497, 803, 1493, 2517, 787, 1538, 2596, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"m29aw"}, {226, 117, 720, 1063, 2369, 685, 1010, 2323, 711, 1063, 2368, 728, 1059, 2376, 738, 1050, 2374, 
			721, 1065, 2368, 699, 1112, 2384, 699, 1154, 2381, 696, 1228, 2364, 40, 0, 0, 0, 1, 0, 11, 8, 0, 0, 0, 0, 0}},
		{{"m29eh"}, {177, 118, 620, 1832, 2583, 618, 1824, 2585, 624, 1831, 2562, 622, 1832, 2576, 620, 1834, 2593, 
			595, 1838, 2614, 580, 1815, 2611, 578, 1799, 2625, 574, 1785, 2633, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m29ei"}, {211, 139, 393, 2305, 2694, 487, 2193, 2585, 447, 2203, 2589, 429, 2215, 2625, 392, 2254, 2670, 
			389, 2267, 2678, 391, 2301, 2696, 399, 2309, 2703, 412, 2282, 2722, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m29er"}, {206, 118, 453, 1432, 0, 510, 1330, 1671, 498, 1375, 1613, 488, 1394, 0, 468, 1413, 0, 
			452, 1426, 0, 452, 1422, 0, 449, 1403, 0, 446, 1431, 0, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m29ih"}, {145, 131, 404, 2107, 2677, 399, 2121, 2721, 397, 2114, 2677, 413, 2085, 2677, 419, 2067, 2677, 
			430, 2022, 2647, 436, 1979, 2621, 442, 1940, 2621, 444, 1879, 2620, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m29iy"}, {161, 132, 309, 2385, 3219, 351, 2370, 3090, 326, 2378, 3142, 315, 2384, 3191, 309, 2389, 3208, 
			309, 2376, 3162, 309, 2370, 3167, 311, 2373, 3149, 314, 2351, 3034, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m29oa"}, {238, 118, 509, 896, 2354, 555, 911, 2349, 544, 911, 2316, 509, 896, 2354, 501, 870, 2363, 
			490, 851, 2398, 475, 818, 2448, 453, 807, 2468, 441, 812, 2458, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m29oo"}, {188, 124, 491, 1246, 2336, 457, 1211, 2440, 489, 1239, 2336, 495, 1300, 2383, 494, 1298, 2404, 
			492, 1294, 2370, 491, 1309, 2368, 490, 1382, 2366, 487, 1442, 2368, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m29uh"}, {145, 133, 623, 1207, 2455, 627, 1218, 2427, 623, 1207, 2455, 620, 1230, 2436, 620, 1239, 2434, 
			619, 1288, 2432, 594, 1338, 2419, 570, 1410, 2416, 559, 1491, 2431, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"m29uw"}, {185, 143, 401, 980, 2460, 391, 961, 2497, 400, 963, 2430, 402, 905, 2424, 386, 868, 2421, 
			372, 867, 2445, 362, 872, 2391, 342, 867, 2416, 324, 919, 2437, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m30ae"}, {302, 96, 559, 1873, 2509, 559, 1787, 2523, 560, 1866, 2541, 559, 1870, 2498, 562, 1830, 2456, 
			567, 1786, 2494, 575, 1739, 2507, 588, 1691, 2532, 570, 1681, 2566, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"m30ah"}, {290, 101, 673, 1301, 2433, 680, 1315, 2418, 673, 1301, 2432, 678, 1305, 2439, 681, 1301, 2442, 
			678, 1342, 2422, 663, 1365, 2405, 644, 1394, 2404, 624, 1465, 2400, 90, 0, 0, 0, 0, 0, 18, 0, 0, 0, 0, 2, 0}},
		{{"m30aw"}, {344, 92, 619, 1012, 2460, 618, 1003, 2461, 620, 1035, 2457, 625, 1064, 2452, 625, 1099, 2444, 
			624, 1121, 2441, 622, 1131, 2429, 625, 1189, 2436, 622, 1323, 2450, 95, 0, 0, 0, 0, 0, 0, 19, 0, 0, 0, 1, 0}},
		{{"m30eh"}, {197, 98, 565, 1805, 2608, 552, 1765, 2605, 564, 1805, 2608, 566, 1800, 2608, 565, 1808, 2638, 
			564, 1813, 2673, 559, 1814, 2684, 553, 1802, 2693, 542, 1789, 2715, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m30ei"}, {338, 94, 374, 2436, 2803, 440, 2396, 2780, 432, 2348, 2761, 392, 2388, 2828, 376, 2427, 2829, 
			373, 2436, 2800, 369, 2438, 2802, 366, 2436, 2766, 363, 2417, 2787, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m30er"}, {310, 103, 430, 1294, 1487, 451, 1312, 1512, 437, 1315, 1491, 430, 1294, 1487, 405, 1374, 1483, 
			392, 1387, 1515, 384, 1411, 1549, 380, 1428, 1552, 379, 1445, 1602, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m30ih"}, {178, 105, 383, 2167, 2726, 381, 2208, 2778, 389, 2138, 2704, 406, 2100, 2701, 416, 2077, 2702, 
			425, 2035, 2697, 434, 1991, 2740, 438, 1951, 2741, 439, 1917, 2754, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m30iy"}, {282, 107, 307, 2405, 3442, 305, 2401, 3436, 301, 2437, 3429, 303, 2435, 3465, 289, 2438, 3466, 
			296, 2434, 3453, 286, 2429, 3398, 272, 2431, 3281, 271, 2404, 3125, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m30oa"}, {301, 95, 376, 659, 2572, 457, 867, 2421, 454, 853, 2472, 437, 794, 2511, 400, 725, 2567, 
			397, 685, 2543, 381, 680, 2572, 377, 672, 2567, 374, 729, 2532, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m30oo"}, {191, 109, 430, 1021, 2507, 427, 1038, 2505, 431, 1020, 2507, 432, 1042, 2547, 434, 1073, 2529, 
			434, 1142, 2499, 436, 1278, 2448, 442, 1355, 2434, 441, 1528, 2458, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m30uh"}, {199, 96, 560, 1118, 2503, 562, 1103, 2483, 562, 1113, 2484, 560, 1118, 2503, 557, 1123, 2514, 
			554, 1150, 2537, 555, 1176, 2522, 549, 1352, 2503, 533, 1430, 2463, 80, 0, 0, 0, 0, 0, 0, 0, 0, 4, 0, 16, 0}},
		{{"m30uw"}, {288, 105, 313, 861, 2374, 351, 936, 2369, 330, 915, 2390, 316, 896, 2366, 319, 870, 2373, 
			309, 860, 2383, 309, 818, 2415, 306, 854, 2386, 302, 870, 2388, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m31ae"}, {208, 143, 579, 1982, 2639, 574, 1971, 2622, 572, 1945, 2618, 582, 1935, 2616, 613, 1946, 2557, 
			624, 1874, 2604, 638, 1851, 2551, 648, 1818, 2524, 647, 1813, 2605, 25, 0, 0, 0, 15, 5, 0, 0, 0, 0, 0, 0, 0}},
		{{"m31ah"}, {184, 160, 707, 1421, 2403, 707, 1421, 2403, 700, 1430, 2378, 700, 1435, 2376, 694, 1420, 2425, 
			693, 1440, 2425, 692, 1451, 2604, 686, 1420, 2611, 681, 1431, 2648, 65, 0, 0, 0, 0, 1, 13, 0, 0, 0, 0, 6, 0}},
		{{"m31aw"}, {196, 155, 640, 1149, 2227, 640, 1158, 2222, 640, 1149, 2227, 627, 1149, 2229, 634, 1084, 2204, 
			662, 1069, 2195, 680, 1101, 2227, 682, 1263, 2345, 659, 1390, 2367, 30, 0, 0, 0, 0, 0, 3, 6, 0, 0, 0, 11, 0}},
		{{"m31eh"}, {146, 146, 571, 1752, 2668, 572, 1796, 2729, 573, 1769, 2697, 571, 1752, 2668, 565, 1758, 2654, 
			568, 1785, 2654, 572, 1787, 2666, 569, 1806, 2692, 560, 1809, 2721, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m31ei"}, {239, 158, 444, 2121, 2547, 494, 1985, 2560, 475, 1958, 2578, 445, 2093, 2615, 447, 2140, 2608, 
			449, 2193, 2668, 427, 2200, 2620, 367, 2210, 2618, 335, 2331, 2688, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m31er"}, {210, 146, 435, 1289, 1562, 437, 1349, 1652, 436, 1318, 1581, 436, 1301, 1602, 438, 1364, 0, 
			441, 1413, 0, 443, 1448, 0, 433, 1443, 0, 381, 1444, 1661, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m31ih"}, {134, 165, 440, 2119, 2694, 411, 2187, 2729, 440, 2119, 2694, 459, 2072, 2690, 471, 2067, 2790, 
			490, 2009, 2707, 491, 1973, 2701, 490, 1973, 2701, 461, 1966, 2750, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m31iy"}, {202, 158, 312, 2319, 2865, 372, 2245, 2748, 360, 2287, 2831, 315, 2332, 2911, 312, 2319, 2865, 
			320, 2358, 2861, 345, 2394, 2752, 368, 2374, 2803, 371, 2378, 2795, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m31oa"}, {175, 150, 457, 870, 2566, 457, 870, 2566, 447, 869, 2592, 442, 772, 2613, 442, 729, 2599, 
			439, 774, 2628, 406, 806, 2621, 377, 838, 2519, 372, 854, 2458, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m31oo"}, {111, 156, 463, 1086, 2453, 451, 1067, 2493, 463, 1086, 2453, 477, 1113, 2448, 484, 1148, 2449, 
			495, 1271, 2471, 497, 1333, 2471, 497, 1364, 2471, 475, 1500, 2482, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 0, 1, 0}},
		{{"m31uh"}, {128, 152, 622, 1285, 2472, 622, 1285, 2472, 619, 1210, 2417, 626, 1239, 2431, 627, 1343, 2477, 
			625, 1362, 2498, 622, 1420, 2556, 570, 1541, 2589, 528, 1578, 2611, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"m31uw"}, {171, 199, 379, 982, 2407, 418, 1115, 2435, 375, 1061, 2390, 368, 1034, 2415, 374, 1063, 2434, 
			377, 1066, 2429, 382, 1010, 2410, 390, 1047, 2445, 396, 1056, 2465, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m32ae"}, {208, 149, 623, 1937, 2526, 610, 1938, 2554, 627, 1948, 2571, 633, 2052, 2547, 660, 2012, 2531, 
			680, 1867, 2486, 685, 1817, 2492, 688, 1756, 2491, 637, 1759, 2559, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"m32ah"}, {216, 146, 697, 1293, 2461, 690, 1276, 2453, 702, 1303, 2487, 722, 1286, 2562, 738, 1290, 2620, 
			740, 1259, 2557, 746, 1232, 2570, 746, 1370, 2576, 715, 1441, 2545, 95, 0, 0, 0, 0, 0, 19, 0, 0, 0, 0, 1, 0}},
		{{"m32aw"}, {195, 144, 633, 995, 2500, 616, 994, 2438, 626, 987, 2501, 627, 993, 2501, 633, 995, 2500, 
			638, 1004, 2480, 640, 1065, 2460, 659, 1237, 2435, 668, 1449, 2431, 70, 0, 0, 0, 0, 0, 1, 14, 0, 0, 0, 5, 0}},
		{{"m32eh"}, {166, 149, 608, 1871, 2572, 590, 1864, 2570, 608, 1871, 2572, 620, 1839, 2590, 623, 1770, 2587, 
			626, 1759, 2586, 623, 1743, 2617, 619, 1746, 2604, 594, 1753, 2615, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m32ei"}, {219, 148, 500, 2159, 2634, 501, 2173, 2631, 500, 2159, 2611, 497, 2193, 2646, 467, 2249, 2718, 
			446, 2269, 2698, 442, 2296, 2717, 445, 2310, 2735, 439, 2304, 2781, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m32er"}, {202, 151, 491, 1370, 1679, 512, 1335, 1708, 508, 1363, 1680, 502, 1366, 1676, 494, 1370, 1679, 
			491, 1370, 1679, 486, 1370, 1680, 475, 1390, 1685, 461, 1420, 1694, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m32ih"}, {154, 148, 438, 2049, 2609, 438, 2047, 2603, 438, 2045, 2609, 439, 2009, 2606, 442, 2001, 2604, 
			455, 1959, 2566, 484, 1896, 2560, 491, 1839, 2585, 482, 1802, 2619, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m32iy"}, {205, 166, 327, 2378, 2999, 366, 2311, 2924, 360, 2284, 3019, 358, 2311, 3093, 340, 2364, 3074, 
			332, 2374, 3020, 327, 2378, 2999, 327, 2392, 2937, 330, 2373, 2872, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m32oa"}, {198, 160, 487, 804, 2508, 566, 936, 2454, 561, 900, 2447, 530, 841, 2458, 505, 814, 2521, 
			487, 804, 2508, 478, 802, 2485, 480, 907, 2439, 474, 979, 2407, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m32oo"}, {134, 160, 500, 1067, 2435, 498, 1068, 2416, 499, 1067, 2425, 500, 1076, 2437, 501, 1109, 2441, 
			503, 1119, 2435, 506, 1200, 2413, 506, 1348, 2380, 503, 1443, 2361, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m32uh"}, {143, 147, 666, 1135, 2471, 646, 1122, 2474, 649, 1122, 2469, 678, 1138, 2473, 681, 1142, 2490, 
			684, 1178, 2498, 685, 1309, 2472, 683, 1343, 2449, 631, 1476, 2474, 95, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 19, 0}},
		{{"m32uw"}, {185, 165, 402, 809, 2466, 428, 856, 2472, 421, 811, 2484, 407, 808, 2476, 400, 809, 2466, 
			392, 813, 2442, 393, 842, 2439, 394, 932, 2408, 392, 1029, 2343, 85, 0, 0, 0, 0, 0, 0, 0, 1, 2, 17, 0, 0}},
		{{"m33ae"}, {328, 136, 555, 1996, 2609, 561, 1965, 2678, 558, 1989, 2637, 552, 1996, 2621, 555, 1976, 2637, 
			560, 1945, 2572, 563, 1897, 2555, 576, 1835, 2435, 590, 1753, 2443, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"m33ah"}, {272, 135, 816, 1361, 2493, 819, 1366, 2522, 835, 1360, 2491, 826, 1347, 2493, 811, 1387, 2483, 
			808, 1422, 2496, 799, 1485, 2498, 778, 1533, 2451, 747, 1572, 2498, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"m33aw"}, {268, 133, 649, 1048, 2441, 639, 1059, 2440, 643, 1048, 2438, 655, 1056, 2440, 674, 1102, 2430, 
			682, 1125, 2426, 681, 1141, 2440, 675, 1246, 2474, 638, 1357, 2506, 85, 0, 0, 0, 0, 0, 3, 17, 0, 0, 0, 0, 0}},
		{{"m33eh"}, {205, 131, 567, 1718, 2557, 566, 1709, 2518, 568, 1689, 2534, 566, 1726, 2561, 565, 1731, 2566, 
			565, 1750, 2566, 564, 1752, 2563, 560, 1775, 2570, 553, 1798, 2577, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m33ei"}, {311, 142, 480, 2100, 2576, 453, 2048, 2542, 441, 2114, 2555, 438, 2189, 2624, 438, 2312, 2576, 
			433, 2315, 2616, 421, 2361, 2625, 399, 2380, 2663, 381, 2390, 2684, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m33er"}, {270, 129, 477, 1376, 1679, 494, 1448, 1731, 492, 1434, 1731, 485, 1423, 1708, 479, 1381, 1681, 
			474, 1376, 1677, 461, 1410, 1676, 461, 1430, 1664, 449, 1445, 1683, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m33ih"}, {204, 147, 437, 1960, 2596, 437, 1960, 2596, 440, 2013, 2615, 444, 1996, 2605, 456, 1951, 2613, 
			464, 1948, 2619, 477, 1938, 2631, 479, 1909, 2642, 476, 1887, 2639, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m33iy"}, {254, 159, 314, 2506, 2963, 319, 2411, 2955, 314, 2453, 2973, 314, 2494, 2947, 313, 2488, 2919, 
			313, 2482, 2921, 311, 2438, 2928, 312, 2384, 2905, 326, 2352, 2821, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m33oa"}, {260, 138, 502, 879, 2465, 507, 926, 2465, 506, 913, 2452, 505, 883, 2479, 500, 877, 2463, 
			496, 874, 2479, 486, 860, 2487, 450, 857, 2450, 439, 869, 2426, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m33oo"}, {180, 141, 495, 1093, 2463, 483, 1080, 2483, 492, 1079, 2463, 496, 1109, 2449, 494, 1122, 2443, 
			491, 1172, 2446, 494, 1266, 2446, 493, 1347, 2431, 487, 1495, 2470, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m33uh"}, {204, 136, 575, 1130, 2489, 584, 1117, 2509, 575, 1130, 2489, 574, 1152, 2501, 575, 1171, 2518, 
			571, 1231, 2551, 566, 1273, 2545, 562, 1327, 2516, 556, 1409, 2526, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"m33uw"}, {241, 160, 373, 931, 2381, 382, 942, 2438, 373, 931, 2381, 364, 925, 2387, 358, 913, 2380, 
			357, 911, 2386, 353, 933, 2401, 348, 968, 2408, 342, 1019, 2406, 90, 0, 0, 0, 0, 0, 0, 0, 1, 0, 18, 0, 1}},
		{{"m34ae"}, {267, 129, 555, 1849, 2502, 555, 1827, 2505, 555, 1849, 2502, 559, 1828, 2516, 567, 1791, 2441, 
			569, 1767, 2371, 575, 1791, 2394, 571, 1761, 2386, 565, 1711, 2536, 35, 0, 0, 0, 13, 7, 0, 0, 0, 0, 0, 0, 0}},
		{{"m34ah"}, {250, 115, 722, 1411, 2264, 695, 1417, 2236, 735, 1423, 2249, 706, 1406, 2239, 696, 1407, 2249, 
			693, 1420, 2208, 693, 1423, 2244, 680, 1431, 2249, 644, 1443, 2206, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"m34aw"}, {239, 116, 623, 1012, 2506, 620, 1012, 2485, 626, 1012, 2506, 626, 1008, 2489, 624, 1003, 2496, 
			628, 1054, 2516, 628, 1098, 2523, 626, 1128, 2593, 626, 1340, 2574, 95, 0, 0, 0, 0, 0, 0, 19, 0, 0, 0, 1, 0}},
		{{"m34eh"}, {185, 121, 541, 1743, 2563, 542, 1743, 2564, 538, 1738, 2561, 535, 1718, 2543, 536, 1712, 2548, 
			543, 1702, 2557, 549, 1693, 2564, 532, 1694, 2576, 503, 1697, 2563, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m34ei"}, {261, 116, 469, 2035, 2660, 469, 2035, 2660, 444, 2065, 2692, 438, 2081, 2701, 438, 2114, 2729, 
			432, 2139, 2737, 433, 2152, 2719, 417, 2174, 2717, 380, 2186, 2659, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m34er"}, {252, 115, 447, 1400, 1791, 484, 1406, 1725, 478, 1389, 1745, 458, 1387, 1723, 447, 1383, 1766, 
			445, 1387, 1742, 447, 1411, 1797, 442, 1414, 1799, 438, 1436, 1858, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m34ih"}, {172, 128, 379, 1942, 2658, 379, 1942, 2658, 380, 1919, 2646, 384, 1908, 2636, 397, 1881, 2624, 
			422, 1854, 2619, 424, 1816, 2616, 419, 1778, 2615, 398, 1761, 2590, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m34iy"}, {200, 130, 365, 2119, 2883, 366, 2096, 2875, 366, 2124, 2887, 361, 2123, 2870, 356, 2126, 2869, 
			352, 2130, 2869, 347, 2129, 2868, 341, 2127, 2845, 335, 2124, 2821, 95, 19, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0}},
		{{"m34oa"}, {243, 124, 499, 883, 2618, 501, 891, 2619, 497, 881, 2610, 484, 876, 2624, 449, 878, 2611, 
			441, 872, 2609, 441, 868, 2621, 438, 872, 2632, 430, 907, 2529, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m34oo"}, {160, 125, 442, 1092, 2459, 442, 1092, 2459, 446, 1088, 2454, 454, 1093, 2477, 471, 1113, 2488, 
			485, 1207, 2471, 486, 1323, 2450, 466, 1398, 2441, 438, 1469, 2444, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m34uh"}, {177, 124, 583, 1187, 2627, 583, 1187, 2627, 573, 1183, 2638, 571, 1180, 2668, 570, 1185, 2696, 
			571, 1206, 2654, 569, 1283, 2569, 559, 1390, 2590, 536, 1439, 2636, 95, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19, 0}},
		{{"m34uw"}, {244, 131, 377, 1008, 2308, 393, 1003, 2317, 378, 1007, 2307, 376, 1011, 2330, 374, 1000, 2309, 
			373, 995, 2290, 373, 996, 2323, 372, 995, 2332, 370, 1028, 2313, 95, 0, 0, 0, 0, 0, 0, 0, 1, 0, 19, 0, 0}},
		{{"m35ae"}, {326, 110, 634, 1730, 2456, 631, 1732, 2481, 638, 1739, 2462, 640, 1705, 2443, 645, 1698, 2438, 
			651, 1683, 2413, 643, 1673, 2382, 641, 1659, 2408, 629, 1635, 2421, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"m35ah"}, {269, 109, 748, 1274, 2406, 758, 1283, 2407, 748, 1274, 2406, 760, 1276, 2410, 751, 1299, 2401, 
			771, 1308, 2423, 758, 1346, 2432, 750, 1362, 2436, 730, 1457, 2441, 90, 0, 0, 0, 0, 1, 18, 1, 0, 0, 0, 0, 0}},
		{{"m35aw"}, {304, 106, 645, 1049, 2507, 643, 1047, 2503, 660, 1055, 2502, 648, 1055, 2474, 669, 1059, 2434, 
			649, 1063, 2403, 649, 1067, 2379, 638, 1068, 2326, 638, 1097, 2318, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"m35eh"}, {244, 112, 543, 1840, 2574, 503, 1827, 2528, 533, 1840, 2574, 548, 1837, 2567, 536, 1794, 2488, 
			548, 1763, 2501, 546, 1765, 2492, 520, 1766, 2505, 503, 1757, 2547, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m35ei"}, {312, 108, 438, 2003, 2552, 473, 1969, 2569, 451, 1969, 2618, 440, 1985, 2582, 437, 2012, 2581, 
			434, 2040, 2660, 429, 2044, 2658, 433, 2097, 2634, 378, 2122, 2716, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m35er"}, {271, 112, 440, 1447, 1701, 480, 1430, 1798, 448, 1441, 1754, 445, 1456, 1714, 441, 1449, 1698, 
			441, 1451, 1697, 439, 1446, 1713, 438, 1446, 1698, 439, 1460, 1689, 95, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 19}},
		{{"m35ih"}, {226, 111, 397, 1985, 2615, 396, 1996, 2659, 422, 1952, 2582, 439, 1901, 2498, 437, 1880, 2503, 
			439, 1838, 2504, 457, 1805, 2478, 477, 1759, 2461, 484, 1748, 2498, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m35iy"}, {268, 113, 327, 2143, 2881, 347, 2109, 2866, 337, 2138, 2878, 326, 2143, 2884, 321, 2132, 2875, 
			323, 2135, 2867, 325, 2144, 2864, 323, 2182, 2862, 316, 2185, 2870, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m35oa"}, {317, 114, 465, 969, 2364, 472, 977, 2335, 457, 940, 2373, 458, 927, 2375, 451, 924, 2403, 
			445, 912, 2410, 447, 888, 2407, 440, 868, 2404, 437, 823, 2391, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m35oo"}, {230, 106, 466, 1375, 2478, 439, 1403, 2437, 439, 1402, 2450, 446, 1410, 2493, 464, 1401, 2478, 
			489, 1383, 2493, 494, 1388, 2493, 490, 1385, 2480, 495, 1440, 2488, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"m35uh"}, {219, 109, 561, 1343, 2498, 556, 1371, 2487, 559, 1364, 2504, 562, 1343, 2495, 564, 1360, 2494, 
			568, 1368, 2522, 566, 1367, 2491, 563, 1379, 2442, 555, 1436, 2453, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"m35uw"}, {254, 114, 357, 1130, 2300, 336, 1113, 2262, 360, 1130, 2294, 360, 1133, 2300, 356, 1123, 2297, 
			354, 1099, 2301, 350, 1052, 2297, 346, 1035, 2277, 351, 1052, 2292, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m36ae"}, {270, 121, 570, 1856, 2630, 571, 1842, 2598, 569, 1865, 2638, 591, 1843, 2624, 616, 1812, 2618, 
			621, 1790, 2612, 624, 1771, 2627, 625, 1745, 2627, 626, 1677, 2617, 90, 0, 0, 0, 2, 18, 0, 0, 0, 0, 0, 0, 0}},
		{{"m36ah"}, {252, 116, 744, 1388, 2541, 713, 1387, 2541, 762, 1393, 2541, 793, 1407, 2538, 745, 1422, 2505, 
			727, 1433, 2491, 714, 1446, 2487, 692, 1467, 2464, 656, 1509, 2502, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"m36aw"}, {300, 112, 645, 1063, 2565, 639, 1063, 2557, 646, 1062, 2562, 634, 1065, 2563, 646, 1074, 2558, 
			637, 1088, 2533, 639, 1119, 2513, 645, 1152, 2473, 640, 1228, 2400, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"m36eh"}, {217, 114, 579, 1866, 2799, 584, 1885, 2805, 569, 1853, 2791, 569, 1827, 2795, 577, 1808, 2758, 
			582, 1768, 2731, 588, 1726, 2718, 603, 1713, 2714, 593, 1656, 2765, 90, 0, 0, 0, 18, 2, 0, 0, 0, 0, 0, 0, 0}},
		{{"m36ei"}, {269, 121, 478, 2049, 2866, 481, 2021, 2863, 481, 2042, 2872, 477, 2055, 2860, 483, 2065, 2848, 
			473, 2068, 2834, 456, 2071, 2819, 442, 2070, 2806, 430, 2071, 2813, 90, 0, 1, 18, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m36er"}, {290, 119, 493, 1428, 1741, 490, 1469, 1733, 490, 1433, 1743, 494, 1432, 1748, 493, 1436, 1747, 
			489, 1453, 1742, 488, 1490, 1728, 487, 1507, 1769, 488, 1537, 1840, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m36ih"}, {174, 130, 428, 1999, 2849, 428, 1999, 2849, 434, 1994, 2826, 434, 1933, 2806, 446, 1911, 2780, 
			469, 1885, 2757, 484, 1848, 2746, 490, 1810, 2743, 490, 1774, 2744, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m36iy"}, {219, 135, 362, 2103, 3074, 337, 2174, 3066, 341, 2153, 3111, 353, 2121, 3113, 355, 2113, 3100, 
			355, 2122, 3086, 362, 2103, 3074, 368, 2091, 3060, 371, 2077, 2985, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m36oa"}, {230, 122, 496, 908, 2435, 556, 1035, 2426, 545, 1002, 2467, 516, 919, 2488, 500, 914, 2480, 
			496, 908, 2435, 496, 913, 2456, 498, 926, 2435, 500, 1012, 2365, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m36oo"}, {185, 120, 496, 1101, 2396, 497, 1108, 2382, 496, 1101, 2396, 496, 1097, 2409, 499, 1117, 2430, 
			502, 1172, 2431, 502, 1211, 2422, 502, 1270, 2368, 501, 1434, 2341, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m36uh"}, {171, 118, 630, 1328, 2554, 627, 1316, 2563, 634, 1318, 2555, 624, 1346, 2554, 615, 1357, 2556, 
			614, 1382, 2559, 609, 1444, 2553, 587, 1500, 2553, 562, 1561, 2588, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"m36uw"}, {265, 124, 388, 982, 2316, 400, 982, 2318, 388, 982, 2316, 378, 992, 2311, 376, 996, 2297, 
			376, 994, 2288, 377, 989, 2272, 378, 997, 2259, 379, 1059, 2192, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m37ae"}, {207, 130, 614, 1768, 2393, 617, 1756, 2357, 615, 1762, 2384, 616, 1768, 2357, 615, 1751, 2349, 
			617, 1721, 2327, 619, 1703, 2292, 619, 1680, 2319, 599, 1655, 2366, 60, 0, 0, 0, 8, 12, 0, 0, 0, 0, 0, 0, 0}},
		{{"m37ah"}, {219, 127, 700, 1295, 2310, 700, 1299, 2329, 700, 1287, 2310, 709, 1287, 2302, 701, 1268, 2323, 
			700, 1296, 2320, 704, 1328, 2286, 692, 1354, 2282, 679, 1394, 2258, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"m37aw"}, {202, 130, 606, 991, 2135, 591, 968, 2168, 610, 991, 2135, 615, 983, 2207, 615, 977, 2255, 
			615, 963, 2277, 617, 960, 2242, 626, 1016, 2225, 622, 1183, 2273, 90, 0, 0, 0, 0, 0, 0, 18, 1, 0, 0, 1, 0}},
		{{"m37eh"}, {139, 130, 568, 1687, 2419, 568, 1692, 2430, 568, 1687, 2419, 568, 1686, 2397, 568, 1687, 2405, 
			567, 1686, 2428, 557, 1681, 2451, 544, 1678, 2487, 515, 1685, 2552, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m37ei"}, {193, 140, 497, 1903, 2455, 495, 1896, 2441, 492, 1908, 0, 466, 1919, 0, 441, 1933, 0, 
			426, 1957, 0, 419, 1969, 0, 414, 1985, 0, 408, 1991, 0, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m37er"}, {212, 130, 478, 1440, 1665, 501, 1378, 1680, 496, 1382, 1650, 486, 1432, 1671, 443, 1447, 1671, 
			438, 1453, 1668, 433, 1459, 1664, 428, 1462, 1735, 424, 1459, 1739, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m37ih"}, {147, 133, 434, 1872, 2543, 436, 1853, 2549, 433, 1872, 2545, 434, 1868, 2519, 436, 1847, 2465, 
			437, 1825, 2455, 437, 1820, 2515, 437, 1813, 2539, 436, 1805, 2542, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m37iy"}, {195, 137, 392, 2127, 2874, 408, 2090, 2803, 415, 2101, 2801, 410, 2104, 2806, 390, 2134, 2874, 
			394, 2140, 2900, 399, 2137, 2941, 399, 2118, 2758, 385, 2090, 2718, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m37oa"}, {211, 142, 437, 861, 2160, 536, 1026, 2188, 531, 990, 2229, 509, 892, 2283, 492, 864, 2263, 
			441, 863, 2189, 435, 854, 2141, 435, 849, 2111, 434, 912, 2072, 95, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0}},
		{{"m37oo"}, {159, 132, 497, 1164, 2245, 497, 1163, 2222, 501, 1178, 2211, 502, 1263, 2190, 502, 1302, 2194, 
			502, 1390, 2235, 499, 1484, 2273, 493, 1525, 2314, 461, 1559, 2354, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m37uh"}, {169, 134, 596, 1053, 2282, 596, 1053, 2282, 593, 1087, 2294, 590, 1110, 2358, 581, 1137, 2383, 
			576, 1173, 2375, 563, 1227, 2363, 552, 1355, 2361, 523, 1469, 2409, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 19, 0}},
		{{"m37uw"}, {193, 143, 409, 846, 2073, 418, 842, 2055, 387, 859, 2061, 404, 855, 2069, 398, 831, 2076, 
			384, 824, 2096, 384, 823, 2088, 385, 849, 2086, 380, 855, 2076, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19, 0, 0}},
		{{"m38ae"}, {213, 131, 553, 2140, 2327, 553, 2140, 2327, 552, 2062, 2332, 557, 2036, 2335, 562, 1988, 2338, 
			569, 1953, 2345, 622, 1802, 2335, 627, 1723, 2319, 624, 1707, 2317, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m38ah"}, {192, 148, 705, 1323, 2643, 693, 1319, 2631, 703, 1324, 2653, 705, 1325, 2630, 702, 1326, 2596, 
			701, 1328, 2595, 696, 1357, 2597, 689, 1388, 2584, 679, 1471, 2591, 45, 0, 0, 0, 0, 0, 9, 0, 0, 0, 0, 11, 0}},
		{{"m38aw"}, {229, 142, 636, 1108, 2623, 627, 1078, 2592, 639, 1093, 2620, 636, 1108, 2619, 638, 1112, 2624, 
			648, 1117, 2624, 671, 1159, 2611, 661, 1195, 2591, 629, 1324, 2564, 75, 0, 0, 0, 0, 0, 3, 15, 0, 0, 0, 2, 0}},
		{{"m38eh"}, {158, 135, 583, 1760, 2497, 569, 1752, 2479, 578, 1755, 2462, 589, 1755, 2524, 603, 1746, 2534, 
			605, 1738, 2544, 583, 1740, 2563, 565, 1742, 2570, 552, 1738, 2578, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m38ei"}, {255, 155, 477, 2073, 2617, 480, 2073, 2603, 457, 2069, 2610, 437, 2107, 2601, 438, 2151, 2573, 
			435, 2204, 2632, 429, 2206, 2629, 385, 2251, 2656, 362, 2251, 2659, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m38er"}, {206, 142, 451, 1314, 1731, 450, 1356, 1756, 455, 1331, 1744, 451, 1314, 1731, 446, 1321, 1722, 
			445, 1337, 1726, 440, 1417, 1763, 436, 1494, 1795, 429, 1540, 1800, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m38ih"}, {142, 133, 427, 2140, 2626, 427, 2140, 2626, 427, 2130, 2613, 418, 2096, 2580, 423, 2073, 2550, 
			435, 1990, 2546, 440, 1917, 2551, 441, 1886, 2555, 439, 1844, 2553, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m38iy"}, {187, 145, 308, 2190, 2819, 308, 2218, 2827, 308, 2190, 2819, 306, 2198, 2796, 304, 2199, 2783, 
			302, 2239, 2822, 297, 2249, 2783, 295, 2249, 2739, 302, 2245, 2680, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m38oa"}, {209, 136, 548, 1067, 2480, 550, 1067, 2482, 547, 1060, 2475, 512, 1041, 2462, 501, 981, 2466, 
			480, 934, 2480, 436, 924, 2489, 430, 931, 2491, 425, 995, 2419, 95, 0, 0, 0, 0, 0, 1, 0, 19, 0, 0, 0, 0}},
		{{"m38oo"}, {153, 146, 436, 1154, 2300, 436, 1154, 2300, 439, 1163, 2310, 438, 1188, 2308, 438, 1221, 2298, 
			438, 1247, 2298, 438, 1373, 2302, 438, 1586, 2356, 438, 1701, 2438, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"m38uh"}, {155, 141, 602, 1176, 2495, 602, 1172, 2481, 608, 1183, 2502, 610, 1195, 2504, 614, 1225, 2502, 
			611, 1259, 2497, 609, 1321, 2506, 578, 1477, 2536, 563, 1639, 2586, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 19, 0}},
		{{"m38uw"}, {195, 153, 377, 1063, 2238, 390, 1092, 2244, 377, 1063, 2238, 374, 1060, 2245, 367, 1064, 2256, 
			363, 1066, 2257, 345, 1066, 2250, 332, 1066, 2244, 318, 1078, 2243, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m39ae"}, {311, 153, 605, 2002, 2666, 605, 2002, 2666, 591, 2005, 2606, 587, 2016, 2653, 595, 2021, 2694, 
			642, 1870, 2488, 673, 1832, 2418, 667, 1767, 2316, 658, 1744, 2519, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m39ah"}, {287, 139, 963, 1524, 2552, 968, 1410, 2506, 970, 1490, 2491, 953, 1554, 2493, 963, 1524, 2552, 
			954, 1565, 2460, 942, 1595, 2513, 945, 1570, 2442, 946, 1586, 2484, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"m39aw"}, {196, 124, 670, 919, 2241, 634, 957, 2246, 670, 919, 2241, 639, 899, 2179, 638, 887, 2173, 
			640, 932, 2199, 664, 930, 2260, 683, 932, 2343, 695, 982, 2394, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"m39eh"}, {201, 150, 641, 1814, 2642, 562, 1947, 2692, 589, 1886, 2692, 620, 1872, 2644, 629, 1839, 2614, 
			641, 1814, 2642, 645, 1811, 2590, 631, 1818, 2583, 623, 1792, 2584, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m39ei"}, {286, 151, 440, 2245, 3044, 490, 2086, 2859, 463, 2164, 2873, 447, 2215, 2981, 442, 2243, 3044, 
			439, 2245, 3043, 438, 2262, 2966, 437, 2301, 3061, 426, 2316, 3021, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m39er"}, {260, 138, 467, 1432, 1742, 475, 1452, 1738, 468, 1436, 1733, 472, 1431, 1742, 482, 1441, 1697, 
			493, 1472, 1729, 487, 1532, 1762, 471, 1607, 1794, 452, 1576, 1838, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m39ih"}, {168, 149, 440, 2082, 2767, 439, 2108, 2787, 440, 2072, 2753, 438, 2061, 2741, 439, 2036, 2725, 
			442, 2006, 2684, 457, 1968, 2655, 491, 1927, 2637, 491, 1877, 2626, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m39iy"}, {235, 184, 373, 2207, 3191, 373, 2204, 3137, 374, 2212, 3172, 372, 2242, 3175, 371, 2267, 3149, 
			371, 2281, 3206, 371, 2256, 3165, 371, 2286, 3168, 373, 2294, 3162, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m39oa"}, {252, 151, 444, 815, 2375, 507, 895, 2364, 500, 810, 2324, 472, 809, 2350, 446, 823, 2372, 
			444, 815, 2375, 448, 770, 2437, 453, 790, 2439, 444, 827, 2505, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m39oo"}, {196, 151, 498, 1048, 2391, 492, 1122, 2387, 493, 1101, 2388, 495, 1045, 2396, 501, 1064, 2393, 
			508, 1079, 2397, 546, 1214, 2391, 550, 1273, 2381, 522, 1481, 2373, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m39uh"}, {193, 149, 608, 1167, 2523, 569, 1227, 2503, 616, 1107, 2540, 615, 1108, 2529, 615, 1119, 2502, 
			620, 1157, 2508, 619, 1245, 2529, 620, 1356, 2509, 605, 1516, 2501, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"m39uw"}, {250, 192, 387, 967, 2257, 384, 972, 2245, 376, 939, 2326, 373, 899, 2291, 400, 860, 2310, 
			427, 868, 2290, 394, 835, 2289, 392, 815, 2284, 407, 864, 2281, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m40ae"}, {273, 140, 597, 1989, 2677, 615, 2039, 2689, 608, 1973, 2647, 582, 1997, 2647, 610, 1947, 2624, 
			623, 1929, 2618, 651, 1871, 2609, 683, 1795, 2572, 684, 1745, 2616, 70, 0, 0, 0, 6, 14, 0, 0, 0, 0, 0, 0, 0}},
		{{"m40ah"}, {254, 126, 711, 1428, 2413, 690, 1497, 2413, 705, 1446, 2407, 721, 1428, 2405, 703, 1428, 2390, 
			686, 1438, 2378, 669, 1441, 2334, 664, 1464, 2411, 683, 1515, 2489, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"m40aw"}, {281, 131, 631, 1167, 2344, 622, 1170, 2337, 630, 1169, 2350, 632, 1170, 2333, 636, 1166, 2332, 
			642, 1195, 2323, 660, 1244, 2365, 675, 1301, 2386, 665, 1387, 2434, 75, 0, 0, 0, 0, 0, 2, 15, 0, 0, 0, 3, 0}},
		{{"m40eh"}, {204, 138, 623, 1937, 2643, 608, 1920, 2623, 583, 1929, 2630, 615, 1953, 2641, 622, 1941, 2652, 
			622, 1937, 2636, 622, 1930, 2639, 606, 1814, 2665, 581, 1779, 2671, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m40ei"}, {230, 142, 502, 2025, 2642, 502, 2043, 2637, 508, 2083, 2642, 495, 2150, 2674, 441, 2193, 2680, 
			425, 2235, 2680, 414, 2241, 2675, 422, 2267, 2670, 428, 2270, 2682, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m40er"}, {266, 127, 505, 1447, 1823, 505, 1447, 1823, 498, 1438, 1802, 474, 1445, 1767, 439, 1450, 1740, 
			434, 1428, 1727, 437, 1448, 1729, 441, 1496, 1733, 452, 1537, 1737, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m40ih"}, {194, 145, 431, 2071, 2671, 436, 2057, 2677, 433, 2069, 2677, 430, 2070, 2657, 431, 2077, 2639, 
			432, 2088, 2641, 436, 2048, 2646, 441, 2026, 2617, 448, 1975, 2629, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m40iy"}, {231, 154, 343, 2502, 3035, 370, 2490, 3004, 327, 2502, 3035, 315, 2493, 2995, 312, 2497, 2986, 
			313, 2497, 2944, 313, 2501, 2922, 314, 2500, 2900, 321, 2491, 2867, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m40oa"}, {256, 134, 522, 911, 2461, 550, 1008, 2439, 522, 911, 2461, 512, 924, 2498, 493, 892, 2502, 
			440, 822, 2564, 438, 866, 2575, 447, 851, 2619, 442, 795, 2589, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m40oo"}, {168, 142, 435, 1263, 2559, 435, 1263, 2559, 434, 1237, 2553, 434, 1221, 2535, 436, 1243, 2535, 
			437, 1257, 2530, 439, 1292, 2524, 450, 1375, 2500, 455, 1518, 2501, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m40uh"}, {201, 138, 612, 1317, 2608, 623, 1338, 2617, 612, 1309, 2607, 621, 1289, 2614, 625, 1197, 2612, 
			624, 1203, 2604, 609, 1215, 2566, 577, 1317, 2599, 566, 1522, 2586, 95, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 19, 0}},
		{{"m40uw"}, {205, 151, 316, 893, 2385, 377, 1011, 2380, 365, 975, 2378, 347, 974, 2374, 320, 893, 2367, 
			316, 893, 2385, 313, 912, 2370, 314, 936, 2357, 323, 967, 2357, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m41ae"}, {230, 139, 561, 1772, 2597, 557, 1808, 2610, 559, 1781, 2602, 567, 1738, 2574, 589, 1649, 2547, 
			615, 1624, 2511, 612, 1570, 2500, 591, 1569, 2504, 566, 1596, 2514, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m41ah"}, {213, 138, 712, 1325, 2624, 699, 1330, 2609, 712, 1328, 2619, 723, 1323, 2634, 734, 1308, 2626, 
			718, 1314, 2627, 666, 1374, 2625, 631, 1431, 2620, 598, 1532, 2606, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"m41aw"}, {249, 134, 687, 1062, 2524, 632, 1081, 2474, 661, 1056, 2487, 687, 1060, 2518, 685, 1065, 2531, 
			685, 1085, 2546, 683, 1115, 2551, 655, 1186, 2522, 612, 1341, 2513, 90, 0, 0, 0, 0, 0, 1, 18, 0, 0, 0, 1, 0}},
		{{"m41eh"}, {188, 137, 562, 1591, 2556, 557, 1627, 2559, 560, 1624, 2552, 560, 1614, 2554, 562, 1591, 2556, 
			562, 1590, 2556, 559, 1591, 2555, 551, 1621, 2567, 510, 1628, 2578, 90, 0, 0, 0, 18, 0, 0, 0, 0, 1, 0, 1, 0}},
		{{"m41ei"}, {217, 139, 501, 1944, 2643, 501, 1903, 2633, 500, 1957, 2648, 480, 1994, 2657, 459, 2000, 2653, 
			439, 2072, 2657, 429, 2143, 2685, 428, 2150, 2695, 431, 2160, 2741, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m41er"}, {214, 150, 461, 1297, 1586, 496, 1262, 1677, 489, 1243, 1638, 482, 1326, 1606, 461, 1297, 1586, 
			455, 1321, 1582, 453, 1324, 1601, 452, 1371, 1633, 450, 1394, 1683, 95, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 19}},
		{{"m41ih"}, {185, 143, 437, 1915, 2687, 439, 1922, 2684, 437, 1915, 2687, 437, 1876, 2681, 444, 1857, 2646, 
			451, 1833, 2629, 450, 1782, 2626, 447, 1775, 2632, 444, 1763, 2631, 95, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m41iy"}, {221, 141, 394, 2300, 2827, 417, 2247, 2820, 404, 2245, 2830, 395, 2241, 2825, 396, 2295, 2825, 
			392, 2297, 2827, 390, 2281, 2828, 392, 2260, 2854, 389, 2210, 2866, 95, 19, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m41oa"}, {229, 143, 554, 1005, 2369, 558, 1017, 2372, 541, 990, 2382, 505, 975, 2429, 483, 928, 2442, 
			446, 891, 2434, 438, 880, 2420, 438, 918, 2371, 439, 1031, 2357, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m41oo"}, {176, 151, 468, 1064, 2383, 470, 1072, 2379, 468, 1064, 2383, 477, 1083, 2377, 483, 1120, 2380, 
			493, 1158, 2379, 497, 1332, 2373, 496, 1446, 2386, 492, 1503, 2431, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m41uh"}, {181, 144, 628, 1172, 2626, 628, 1172, 2626, 624, 1176, 2626, 624, 1183, 2617, 627, 1217, 2606, 
			624, 1260, 2589, 602, 1357, 2542, 566, 1459, 2500, 555, 1557, 2535, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"m41uw"}, {200, 154, 437, 1085, 2305, 438, 1185, 2287, 436, 1181, 2290, 435, 1153, 2289, 435, 1131, 2297, 
			437, 1085, 2305, 437, 1075, 2309, 435, 1139, 2308, 421, 1300, 2293, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m42ae"}, {334, 128, 615, 1885, 2709, 618, 1874, 2647, 617, 1875, 2717, 622, 1877, 2735, 627, 1901, 2753, 
			646, 1876, 2767, 689, 1933, 2553, 700, 1886, 2671, 702, 1859, 2704, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m42ah"}, {292, 140, 821, 1283, 2498, 800, 1309, 2465, 821, 1283, 2498, 827, 1262, 2496, 839, 1307, 2385, 
			850, 1347, 2438, 781, 1400, 2465, 805, 1421, 2477, 816, 1468, 2479, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"m42aw"}, {328, 131, 684, 1000, 2576, 639, 1001, 2443, 666, 979, 2591, 684, 1000, 2576, 689, 921, 2586, 
			690, 948, 2548, 693, 1046, 2553, 681, 1066, 2606, 691, 1069, 2525, 95, 0, 0, 0, 0, 0, 0, 19, 0, 0, 0, 1, 0}},
		{{"m42eh"}, {207, 161, 622, 1799, 2726, 579, 1782, 2694, 593, 1772, 2678, 606, 1782, 2706, 617, 1801, 2748, 
			622, 1799, 2726, 628, 1790, 2765, 631, 1811, 2808, 623, 1840, 2796, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m42ei"}, {319, 138, 549, 1921, 2686, 552, 1922, 2760, 554, 1953, 2803, 559, 1999, 2843, 549, 2060, 2918, 
			500, 2076, 2978, 497, 2114, 2932, 500, 2167, 2922, 479, 2218, 2945, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m42er"}, {326, 143, 557, 1344, 1610, 562, 1221, 1713, 559, 1358, 1646, 555, 1444, 1621, 558, 1322, 1608, 
			555, 1309, 1624, 528, 1386, 1621, 502, 1401, 1629, 501, 1419, 1669, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m42ih"}, {191, 162, 474, 1956, 2762, 448, 1988, 2798, 463, 1979, 2790, 481, 1939, 2748, 491, 1921, 2748, 
			498, 1906, 2651, 501, 1856, 2618, 510, 1814, 2641, 534, 1834, 2641, 95, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m42iy"}, {286, 157, 373, 2214, 3134, 389, 2176, 3059, 377, 2204, 3129, 371, 2214, 3114, 367, 2197, 3026, 
			353, 2276, 3108, 356, 2297, 3190, 367, 2254, 3113, 372, 2310, 3110, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m42oa"}, {289, 140, 528, 792, 2481, 561, 895, 2267, 551, 788, 2390, 528, 792, 2481, 505, 789, 2450, 
			446, 724, 2589, 463, 709, 2562, 459, 656, 2627, 422, 664, 2633, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m42oo"}, {227, 128, 510, 1238, 2250, 508, 1230, 2243, 510, 1238, 2250, 509, 1247, 2283, 512, 1249, 2328, 
			519, 1268, 2296, 539, 1349, 2316, 555, 1407, 2380, 558, 1492, 2371, 90, 0, 0, 0, 0, 0, 0, 0, 0, 18, 0, 2, 0}},
		{{"m42uh"}, {242, 128, 631, 1241, 2276, 628, 1241, 2274, 633, 1243, 2300, 656, 1258, 2320, 677, 1313, 2306, 
			644, 1332, 2299, 640, 1343, 2269, 633, 1469, 2235, 645, 1534, 2381, 75, 0, 0, 0, 0, 0, 4, 1, 0, 0, 0, 15, 0}},
		{{"m42uw"}, {256, 154, 435, 1070, 2416, 433, 1134, 2384, 435, 1070, 2416, 438, 1033, 2433, 441, 1078, 2456, 
			438, 1032, 2472, 431, 970, 2500, 419, 980, 2506, 388, 1055, 2519, 90, 0, 0, 0, 0, 0, 0, 0, 0, 2, 18, 0, 0}},
		{{"m44ae"}, {276, 98, 577, 1952, 2588, 578, 1961, 2590, 570, 1937, 2555, 570, 1910, 2562, 587, 1854, 2630, 
			614, 1788, 2650, 624, 1730, 2659, 629, 1604, 2705, 624, 1561, 2741, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m44ah"}, {286, 101, 822, 1315, 2708, 825, 1355, 2685, 837, 1343, 2694, 824, 1303, 2696, 810, 1315, 2708, 
			791, 1303, 2715, 781, 1304, 2709, 748, 1332, 2685, 675, 1432, 2624, 95, 0, 0, 0, 0, 0, 19, 0, 0, 0, 1, 0, 0}},
		{{"m44aw"}, {301, 100, 684, 1001, 2833, 684, 1001, 2833, 694, 996, 2837, 681, 1010, 2855, 687, 1037, 2849, 
			684, 1049, 2829, 685, 1105, 2754, 652, 1175, 2699, 640, 1281, 2702, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"m44eh"}, {200, 98, 561, 1786, 2603, 561, 1797, 2599, 564, 1786, 2621, 577, 1771, 2627, 577, 1758, 2629, 
			579, 1699, 2624, 579, 1692, 2648, 571, 1677, 2665, 563, 1677, 2690, 95, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0}},
		{{"m44ei"}, {269, 101, 497, 2083, 2628, 497, 2051, 2623, 497, 2089, 2671, 498, 2112, 2669, 488, 2133, 2671, 
			442, 2182, 2684, 434, 2195, 2687, 433, 2210, 2678, 435, 2225, 2679, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m44er"}, {252, 105, 491, 1300, 1744, 493, 1343, 1765, 491, 1300, 1744, 474, 1288, 1703, 461, 1283, 1696, 
			442, 1307, 1696, 437, 1359, 1706, 437, 1431, 1704, 437, 1457, 1754, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m44ih"}, {204, 111, 433, 2012, 2708, 433, 2012, 2708, 437, 2015, 2739, 437, 1988, 2726, 437, 1937, 2737, 
			438, 1910, 2747, 446, 1881, 2737, 451, 1824, 2750, 448, 1803, 2770, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m44iy"}, {291, 100, 312, 2307, 2834, 312, 2307, 2834, 306, 2326, 2869, 307, 2313, 2824, 309, 2287, 2829, 
			310, 2409, 2962, 309, 2456, 2972, 311, 2422, 3036, 313, 2402, 3029, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m44oa"}, {282, 106, 555, 1019, 2611, 553, 1013, 2611, 536, 987, 2608, 505, 937, 2586, 495, 908, 2576, 
			441, 873, 2558, 434, 872, 2541, 430, 872, 2499, 430, 879, 2462, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m44oo"}, {220, 112, 409, 965, 2490, 432, 978, 2488, 436, 1012, 2484, 440, 1072, 2439, 444, 1153, 2427, 
			459, 1223, 2425, 467, 1280, 2419, 460, 1417, 2422, 452, 1512, 2532, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"m44uh"}, {208, 107, 650, 1154, 2640, 654, 1158, 2636, 650, 1154, 2637, 630, 1169, 2629, 628, 1168, 2636, 
			622, 1202, 2624, 608, 1266, 2619, 600, 1340, 2627, 564, 1431, 2623, 95, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 19, 0}},
		{{"m44uw"}, {219, 113, 365, 999, 2270, 383, 1054, 2237, 384, 1054, 2258, 383, 989, 2255, 371, 992, 2286, 
			366, 997, 2275, 363, 1007, 2270, 362, 1047, 2274, 367, 1091, 2197, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m45ae"}, {283, 187, 629, 2436, 3022, 681, 2391, 2973, 626, 2368, 2987, 629, 2436, 3022, 625, 2361, 2922, 
			657, 2324, 2887, 740, 2274, 2794, 799, 2185, 2799, 802, 2165, 2848, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m45ah"}, {285, 180, 753, 1200, 2677, 829, 1159, 2680, 772, 1203, 2712, 801, 1191, 2751, 796, 1156, 2930, 
			932, 1161, 2955, 896, 1250, 2784, 813, 1426, 2729, 870, 1691, 2869, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"m45aw"}, {293, 184, 704, 1052, 2749, 683, 917, 2664, 686, 955, 2680, 697, 1044, 2730, 711, 1056, 2740, 
			747, 1061, 2753, 764, 1062, 2775, 961, 1165, 2841, 900, 1397, 2855, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"m45eh"}, {200, 187, 723, 2208, 3057, 720, 2233, 3017, 704, 2451, 3038, 714, 2602, 3034, 685, 2332, 2968, 
			687, 2225, 2932, 672, 2267, 2881, 636, 2069, 2852, 625, 2007, 2831, 50, 0, 0, 0, 10, 10, 0, 0, 0, 0, 0, 0, 0}},
		{{"m45ei"}, {255, 181, 537, 2493, 3151, 537, 2493, 3151, 551, 2532, 3178, 517, 2581, 3215, 497, 2567, 3140, 
			446, 2569, 3162, 428, 2576, 3236, 422, 2625, 3223, 427, 2630, 3325, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m45er"}, {255, 196, 486, 1372, 1797, 508, 1335, 1847, 486, 1372, 1797, 428, 1375, 1740, 429, 1432, 1741, 
			424, 1442, 1706, 428, 1460, 1666, 429, 1494, 0, 434, 1504, 0, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m45ih"}, {224, 196, 432, 2494, 3137, 375, 2539, 3151, 426, 2506, 3178, 442, 2463, 2988, 504, 2410, 3021, 
			547, 2383, 2972, 571, 2325, 2929, 575, 2359, 2946, 537, 2276, 2936, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m45iy"}, {234, 216, 420, 2600, 0, 418, 2632, 0, 421, 2587, 0, 425, 2610, 0, 430, 2626, 0, 
			432, 2640, 0, 434, 2659, 0, 437, 2683, 0, 448, 2571, 3178, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m45oa"}, {247, 181, 552, 924, 2565, 552, 934, 2568, 550, 917, 2561, 554, 921, 2564, 554, 923, 2604, 
			514, 862, 2614, 445, 734, 2580, 431, 750, 2571, 419, 768, 2602, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m45oo"}, {186, 195, 497, 1107, 2501, 463, 1081, 2502, 500, 1053, 2548, 500, 1070, 2525, 500, 1144, 2514, 
			522, 1169, 2551, 550, 1209, 2573, 565, 1414, 2552, 560, 1582, 2518, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m45uh"}, {194, 180, 670, 1061, 2703, 632, 1099, 2723, 670, 1061, 2703, 683, 1063, 2742, 687, 1071, 2752, 
			690, 1105, 2754, 692, 1198, 2744, 672, 1456, 2737, 625, 1570, 2745, 90, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 18, 0}},
		{{"m45uw"}, {245, 224, 437, 871, 2393, 375, 978, 2324, 377, 951, 2327, 418, 947, 2329, 429, 831, 2313, 
			434, 840, 2363, 438, 875, 2395, 442, 932, 2386, 457, 1110, 2349, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m47ae"}, {292, 129, 511, 1811, 2529, 511, 1810, 2532, 514, 1813, 2515, 522, 1813, 2517, 545, 1801, 2545, 
			563, 1750, 2516, 575, 1692, 2505, 580, 1633, 2521, 563, 1603, 2544, 90, 0, 0, 0, 2, 18, 0, 0, 0, 0, 0, 0, 0}},
		{{"m47ah"}, {301, 120, 662, 1183, 2374, 662, 1183, 2374, 662, 1190, 2343, 674, 1199, 2358, 672, 1223, 2359, 
			661, 1234, 2349, 653, 1251, 2321, 636, 1280, 2283, 628, 1302, 2279, 85, 0, 0, 0, 0, 0, 17, 2, 0, 0, 0, 1, 0}},
		{{"m47aw"}, {300, 121, 591, 974, 2325, 586, 1004, 2327, 590, 982, 2326, 596, 970, 2327, 612, 985, 2349, 
			618, 998, 2335, 621, 1024, 2295, 623, 1083, 2265, 617, 1164, 2219, 80, 0, 0, 0, 0, 0, 2, 16, 0, 0, 0, 2, 0}},
		{{"m47eh"}, {209, 127, 534, 1608, 2509, 513, 1628, 2532, 520, 1617, 2518, 538, 1606, 2511, 534, 1608, 2509, 
			523, 1619, 2511, 515, 1619, 2522, 511, 1617, 2545, 505, 1615, 2556, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m47ei"}, {282, 123, 482, 1804, 2551, 486, 1837, 2574, 458, 2025, 2604, 436, 2069, 2639, 406, 2114, 2649, 
			389, 2094, 2658, 387, 2077, 2666, 384, 2081, 2662, 383, 2069, 2677, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m47er"}, {259, 133, 443, 1262, 1538, 484, 1252, 1621, 461, 1260, 1585, 449, 1281, 1535, 441, 1277, 1554, 
			436, 1283, 1563, 431, 1287, 1547, 430, 1332, 1571, 427, 1376, 1614, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m47ih"}, {244, 131, 396, 1810, 2504, 396, 1810, 2504, 400, 1802, 2508, 413, 1771, 2501, 424, 1753, 2489, 
			433, 1745, 2497, 435, 1696, 2503, 435, 1681, 2514, 436, 1678, 2521, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m47iy"}, {228, 159, 315, 2052, 3005, 310, 2054, 2999, 310, 2068, 3010, 312, 2070, 3010, 315, 2052, 3005, 
			317, 2061, 3020, 320, 2079, 3016, 329, 2117, 3032, 340, 2119, 2992, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m47oa"}, {293, 143, 473, 820, 2308, 505, 972, 2300, 504, 929, 2314, 503, 896, 2317, 500, 853, 2346, 
			473, 820, 2308, 446, 852, 2292, 439, 919, 2234, 436, 1008, 2160, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m47oo"}, {219, 141, 465, 1087, 2217, 441, 1079, 2178, 451, 1068, 2196, 454, 1070, 2206, 473, 1098, 2225, 
			481, 1118, 2231, 486, 1141, 2236, 489, 1211, 2242, 454, 1326, 2265, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"m47uh"}, {198, 124, 566, 1040, 2476, 560, 1001, 2444, 570, 1050, 2491, 572, 1051, 2492, 569, 1067, 2432, 
			562, 1112, 2407, 554, 1173, 2397, 515, 1331, 2406, 508, 1403, 2398, 80, 0, 0, 0, 0, 0, 0, 1, 0, 3, 0, 16, 0}},
		{{"m47uw"}, {250, 147, 413, 1193, 2164, 408, 1179, 2132, 413, 1195, 2225, 397, 1188, 2154, 389, 1159, 2131, 
			382, 1110, 2180, 369, 1115, 2242, 359, 1137, 2190, 351, 1185, 2180, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m48ae"}, {245, 119, 590, 1830, 2512, 595, 1827, 2482, 580, 1832, 2476, 593, 1848, 2465, 582, 1832, 2397, 
			577, 1812, 2352, 593, 1791, 2410, 622, 1774, 2427, 619, 1749, 2511, 80, 0, 0, 0, 4, 16, 0, 0, 0, 0, 0, 0, 0}},
		{{"m48ah"}, {207, 122, 682, 1259, 2618, 662, 1247, 2623, 679, 1263, 2618, 675, 1282, 2618, 677, 1287, 2611, 
			677, 1281, 2607, 673, 1287, 2560, 653, 1322, 2615, 642, 1372, 2664, 90, 0, 0, 0, 0, 0, 18, 1, 0, 0, 0, 1, 0}},
		{{"m48aw"}, {242, 118, 611, 1067, 2491, 609, 1066, 2495, 613, 1071, 2491, 624, 1090, 2460, 628, 1101, 2450, 
			629, 1118, 2441, 629, 1123, 2420, 629, 1171, 2427, 628, 1250, 2435, 90, 0, 0, 0, 0, 0, 0, 18, 0, 1, 0, 1, 0}},
		{{"m48eh"}, {198, 124, 574, 1813, 2476, 563, 1877, 2536, 566, 1868, 2524, 573, 1847, 2493, 575, 1820, 2470, 
			575, 1813, 2488, 573, 1812, 2456, 569, 1797, 2484, 564, 1769, 2507, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m48ei"}, {247, 120, 501, 2042, 2617, 496, 2044, 2616, 492, 2058, 2604, 488, 2078, 2503, 492, 2115, 2492, 
			487, 2149, 2502, 449, 2155, 2528, 434, 2200, 2505, 424, 2230, 2537, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m48er"}, {248, 116, 489, 1493, 1836, 494, 1498, 1838, 490, 1492, 1838, 486, 1500, 1822, 480, 1505, 1823, 
			475, 1530, 1814, 489, 1569, 1789, 493, 1564, 1794, 463, 1567, 1801, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m48ih"}, {182, 137, 429, 1962, 2554, 419, 1967, 2575, 429, 1962, 2554, 432, 1936, 2507, 436, 1889, 2515, 
			443, 1828, 2551, 455, 1829, 2551, 458, 1846, 2565, 452, 1783, 2605, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m48iy"}, {216, 130, 366, 2504, 0, 369, 2438, 0, 368, 2442, 0, 365, 2462, 0, 365, 2462, 0, 
			366, 2504, 0, 363, 2503, 0, 369, 2505, 0, 372, 2446, 2623, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m48oa"}, {232, 133, 499, 927, 2493, 504, 1003, 2561, 501, 989, 2557, 500, 935, 2555, 500, 910, 2511, 
			499, 919, 2491, 498, 928, 2495, 496, 933, 2510, 494, 949, 2482, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m48oo"}, {167, 136, 441, 1181, 2326, 430, 1154, 2341, 434, 1134, 2341, 439, 1149, 2331, 441, 1181, 2326, 
			453, 1191, 2326, 490, 1345, 2320, 489, 1398, 2329, 468, 1480, 2304, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"m48uh"}, {188, 119, 600, 1182, 2471, 575, 1184, 2531, 606, 1183, 2524, 594, 1182, 2538, 605, 1188, 2444, 
			607, 1213, 2444, 612, 1261, 2443, 610, 1353, 2441, 600, 1397, 2462, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"m48uw"}, {212, 132, 378, 893, 2455, 379, 939, 2319, 377, 856, 2401, 377, 862, 2389, 380, 883, 2513, 
			378, 901, 2455, 379, 933, 2417, 382, 954, 2403, 391, 994, 2364, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"m49ae"}, {266, 123, 622, 1957, 2841, 623, 1953, 2870, 622, 1956, 2832, 618, 1978, 2852, 610, 1990, 2847, 
			615, 1966, 2831, 623, 1885, 2786, 628, 1821, 2759, 631, 1814, 2774, 75, 0, 0, 0, 5, 15, 0, 0, 0, 0, 0, 0, 0}},
		{{"m49ah"}, {270, 124, 818, 1315, 2697, 819, 1303, 2716, 821, 1337, 2695, 801, 1333, 2685, 806, 1355, 2680, 
			806, 1315, 2677, 811, 1332, 2659, 805, 1362, 2635, 753, 1514, 2624, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"m49aw"}, {293, 121, 643, 931, 2630, 629, 896, 2584, 631, 903, 2643, 633, 905, 2684, 648, 936, 2589, 
			668, 958, 2562, 670, 989, 2569, 668, 1033, 2537, 683, 1147, 2510, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"m49eh"}, {172, 114, 557, 1996, 2876, 509, 2057, 2899, 545, 2004, 2887, 557, 1996, 2876, 560, 2000, 2874, 
			563, 1954, 2871, 562, 1916, 2901, 555, 1893, 2871, 522, 1876, 2823, 90, 0, 2, 0, 18, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m49ei"}, {229, 138, 441, 2348, 2935, 501, 2257, 2939, 499, 2288, 2886, 497, 2313, 2951, 463, 2341, 2935, 
			422, 2347, 2888, 423, 2348, 2873, 430, 2333, 2852, 432, 2334, 2838, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m49er"}, {238, 156, 444, 1377, 1543, 507, 1268, 1637, 499, 1301, 1604, 453, 1317, 1557, 443, 1331, 1555, 
			442, 1331, 1589, 445, 1383, 1571, 455, 1422, 1628, 483, 1432, 1651, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m49ih"}, {177, 136, 416, 2296, 2995, 428, 2368, 3055, 416, 2296, 2995, 407, 2255, 2938, 416, 2247, 2917, 
			433, 2246, 2908, 448, 2175, 2899, 459, 2058, 2890, 443, 2012, 2873, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m49iy"}, {246, 133, 370, 2536, 3332, 381, 2616, 3307, 376, 2584, 3268, 374, 2579, 3253, 370, 2579, 3290, 
			369, 2569, 3349, 370, 2561, 3348, 367, 2521, 3317, 363, 2509, 3258, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m49oa"}, {188, 136, 495, 938, 2519, 474, 972, 2562, 500, 954, 2560, 498, 943, 2542, 489, 935, 2520, 
			452, 934, 2526, 441, 928, 2536, 436, 903, 2511, 428, 836, 2516, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m49oo"}, {178, 164, 456, 1002, 2518, 453, 1013, 2518, 453, 996, 2523, 452, 996, 2532, 450, 1024, 2528, 
			450, 1078, 2509, 452, 1120, 2499, 458, 1277, 2460, 455, 1402, 2408, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"m49uh"}, {166, 138, 618, 1228, 2718, 628, 1225, 2683, 631, 1290, 2651, 633, 1308, 2640, 635, 1336, 2668, 
			635, 1368, 2712, 631, 1376, 2685, 622, 1418, 2607, 594, 1513, 2586, 95, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 19, 0}},
		{{"m49uw"}, {125, 171, 374, 887, 2478, 374, 887, 2478, 372, 874, 2478, 373, 892, 2474, 373, 904, 2469, 
			374, 877, 2476, 381, 861, 2501, 381, 837, 2501, 374, 824, 2492, 95, 0, 0, 0, 0, 0, 0, 0, 1, 0, 19, 0, 0}},
		{{"m50ae"}, {289, 116, 565, 2055, 2575, 565, 2055, 2575, 566, 2043, 2537, 573, 2026, 2533, 598, 2006, 2548, 
			617, 1918, 2491, 626, 1809, 2525, 625, 1751, 2525, 619, 1676, 2542, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"m50ah"}, {250, 110, 710, 1483, 2575, 706, 1476, 2580, 713, 1511, 2561, 713, 1516, 2525, 727, 1447, 2499, 
			721, 1489, 2517, 717, 1540, 2495, 720, 1547, 2523, 710, 1557, 2542, 90, 0, 0, 0, 0, 1, 18, 1, 0, 0, 0, 0, 0}},
		{{"m50aw"}, {277, 109, 679, 1102, 2424, 677, 1176, 2421, 679, 1102, 2424, 655, 1111, 2397, 669, 1086, 2409, 
			646, 1070, 2354, 667, 1112, 2354, 668, 1233, 2406, 662, 1304, 2424, 10, 0, 0, 0, 0, 0, 18, 2, 0, 0, 0, 0, 0}},
		{{"m50eh"}, {178, 115, 565, 1952, 2619, 565, 1952, 2619, 565, 1943, 2620, 568, 1913, 2622, 578, 1906, 2612, 
			582, 1900, 2615, 582, 1857, 2599, 576, 1807, 2600, 563, 1798, 2613, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m50ei"}, {256, 109, 492, 2108, 2974, 491, 2111, 2840, 487, 2122, 2841, 443, 2138, 2894, 437, 2190, 2898, 
			436, 2211, 2840, 432, 2255, 2815, 416, 2263, 2829, 417, 2274, 2776, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m50er"}, {258, 123, 516, 1399, 1756, 555, 1353, 1805, 543, 1372, 1787, 505, 1409, 1691, 500, 1389, 1679, 
			499, 1406, 1673, 499, 1432, 1659, 498, 1438, 1727, 477, 1516, 1747, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"m50ih"}, {198, 130, 441, 2068, 2594, 441, 2069, 2594, 482, 2064, 2587, 497, 2037, 2589, 499, 1951, 2494, 
			499, 1927, 2503, 499, 1864, 2531, 500, 1827, 2564, 499, 1814, 2637, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m50iy"}, {228, 114, 354, 2375, 3150, 370, 2348, 3165, 363, 2363, 3207, 351, 2378, 3218, 334, 2375, 3241, 
			341, 2375, 3215, 357, 2375, 3133, 348, 2377, 3120, 324, 2378, 3158, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"m50oa"}, {251, 119, 601, 1045, 2396, 566, 1031, 2376, 563, 995, 2343, 556, 995, 2379, 513, 934, 2385, 
			486, 871, 2436, 461, 809, 2442, 441, 746, 2357, 435, 736, 2449, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"m50oo"}, {205, 120, 495, 1106, 2444, 500, 1113, 2404, 495, 1106, 2444, 493, 1104, 2449, 489, 1140, 2483, 
			488, 1161, 2486, 492, 1239, 2467, 489, 1341, 2423, 483, 1419, 2386, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"m50uh"}, {176, 119, 622, 1163, 2370, 620, 1161, 2363, 622, 1161, 2373, 622, 1165, 2375, 623, 1188, 2387, 
			624, 1224, 2394, 624, 1248, 2395, 618, 1305, 2404, 584, 1390, 2415, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"m50uw"}, {222, 130, 389, 1033, 2375, 417, 1054, 2321, 395, 1034, 2342, 382, 996, 2380, 378, 0, 2386, 
			378, 0, 2437, 377, 0, 2424, 377, 0, 2424, 378, 1019, 2411, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w01ae"}, {305, 225, 678, 2293, 2861, 674, 2243, 2853, 681, 2295, 2868, 679, 2279, 2877, 679, 2234, 2867, 
			711, 2160, 2867, 760, 1999, 2857, 775, 1887, 2912, 705, 1968, 2906, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"w01ah"}, {265, 211, 1012, 1603, 2767, 1012, 1603, 2767, 1001, 1637, 2762, 1017, 1663, 2762, 1053, 1672, 2781, 
			1058, 1692, 2801, 1056, 1723, 2814, 1019, 1726, 2853, 886, 1784, 2807, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"w01aw"}, {305, 218, 815, 1200, 2783, 788, 1168, 2710, 815, 1200, 2783, 815, 1171, 2787, 816, 1170, 2812, 
			844, 1193, 2833, 839, 1248, 2865, 831, 1284, 2857, 831, 1414, 2825, 90, 0, 0, 0, 0, 0, 2, 18, 0, 0, 0, 0, 0}},
		{{"w01eh"}, {217, 228, 734, 1893, 2820, 734, 1893, 2820, 733, 1905, 2820, 713, 1929, 2885, 697, 1962, 2904, 
			682, 1994, 2909, 673, 2007, 2921, 627, 2052, 2936, 511, 2091, 2948, 75, 0, 0, 1, 15, 0, 0, 0, 0, 0, 0, 3, 1}},
		{{"w01ei"}, {239, 227, 539, 2476, 2868, 456, 2500, 2892, 441, 2682, 2919, 445, 2763, 2934, 468, 2702, 2950, 
			484, 2715, 2953, 492, 2692, 2963, 492, 2739, 2972, 493, 2747, 2982, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w01er"}, {300, 224, 470, 1437, 1672, 478, 1437, 1707, 464, 1437, 1659, 452, 1437, 1659, 457, 1390, 1693, 
			460, 1398, 1741, 472, 1403, 1784, 491, 1451, 1930, 499, 1508, 1982, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w01ih"}, {176, 246, 496, 2260, 3027, 492, 2340, 2976, 494, 2266, 2976, 496, 2260, 3027, 497, 2257, 3015, 
			497, 2259, 3044, 498, 2270, 3069, 499, 2289, 3082, 500, 2308, 3084, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w01iy"}, {232, 234, 441, 2806, 3260, 443, 2767, 3217, 446, 2767, 3193, 442, 2872, 3295, 439, 2840, 3265, 
			441, 2815, 3260, 441, 2806, 3260, 441, 2810, 3249, 439, 2822, 3264, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w01oa"}, {302, 234, 557, 1135, 2961, 564, 1157, 2979, 554, 1139, 2935, 529, 1133, 2930, 501, 1131, 2981, 
			489, 1054, 3009, 481, 979, 2995, 491, 1000, 2913, 496, 990, 2945, 95, 0, 0, 0, 0, 0, 1, 0, 19, 0, 0, 0, 0}},
		{{"w01oo"}, {238, 228, 489, 1357, 2937, 488, 1357, 2896, 489, 1357, 2926, 490, 1365, 2944, 492, 1382, 2971, 
			496, 1416, 3016, 497, 1426, 3071, 500, 1448, 3071, 500, 1467, 2979, 75, 0, 0, 0, 0, 0, 0, 0, 0, 15, 5, 0, 0}},
		{{"w01uh"}, {246, 222, 803, 1501, 2989, 803, 1501, 2989, 801, 1458, 2989, 784, 1474, 3000, 741, 1530, 2970, 
			698, 1558, 2955, 676, 1594, 2945, 658, 1733, 2958, 584, 1836, 2922, 95, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 19, 0}},
		{{"w01uw"}, {281, 260, 503, 1068, 2810, 486, 1044, 2761, 513, 1122, 2860, 510, 1187, 2825, 503, 1143, 2843, 
			504, 1075, 2801, 504, 1061, 2869, 500, 1064, 2851, 487, 1125, 2865, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w02ae"}, {486, 214, 624, 2442, 3091, 626, 2442, 3101, 621, 2470, 3042, 631, 2415, 3004, 701, 2313, 3041, 
			755, 2237, 3033, 765, 2091, 3052, 753, 2047, 3072, 712, 2000, 3069, 95, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0}},
		{{"w02ah"}, {443, 209, 883, 1682, 2962, 939, 1655, 2839, 908, 1677, 2903, 883, 1682, 2962, 883, 1687, 2993, 
			885, 1706, 3048, 871, 1764, 3057, 825, 1791, 3068, 783, 1863, 3081, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"w02aw"}, {464, 211, 760, 1225, 2796, 755, 1236, 2811, 762, 1227, 2818, 764, 1220, 2790, 768, 1265, 2766, 
			774, 1375, 2842, 783, 1429, 2860, 799, 1624, 2887, 770, 1772, 2843, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"w02eh"}, {432, 197, 740, 2186, 3086, 779, 2140, 3070, 751, 2186, 3066, 746, 2182, 3020, 731, 2185, 3091, 
			704, 2162, 3019, 690, 2136, 3028, 675, 2155, 3033, 630, 2132, 3014, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w02ei"}, {453, 220, 477, 2704, 0, 550, 2780, 0, 486, 2733, 0, 446, 2678, 0, 439, 2683, 0, 
			436, 2808, 0, 435, 2833, 0, 435, 2824, 0, 435, 2817, 0, 95, 0, 0, 19, 0, 0, 1, 0, 0, 0, 0, 0, 0}},
		{{"w02er"}, {418, 218, 500, 1441, 1914, 507, 1428, 1914, 493, 1456, 1914, 484, 1461, 1882, 464, 1578, 1940, 
			449, 1596, 1998, 442, 1644, 2021, 440, 1680, 2042, 439, 1694, 2068, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w02ih"}, {356, 219, 446, 2444, 2956, 484, 2477, 3116, 444, 2443, 2934, 442, 2444, 2973, 443, 2444, 2993, 
			448, 2418, 3010, 454, 2375, 3069, 479, 2354, 3048, 487, 2354, 3051, 85, 0, 17, 2, 0, 1, 0, 0, 0, 0, 0, 0, 0}},
		{{"w02iy"}, {380, 222, 435, 2890, 3667, 442, 2857, 3609, 437, 2939, 3679, 436, 2896, 3647, 435, 2890, 3667, 
			435, 2884, 3663, 434, 2920, 3617, 433, 2942, 3575, 434, 2879, 3593, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w02oa"}, {422, 212, 432, 1013, 2690, 576, 1105, 2664, 519, 1052, 2734, 479, 1041, 2751, 444, 1043, 2805, 
			435, 1006, 2755, 432, 1020, 2729, 433, 1012, 2664, 434, 1036, 2615, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w02oo"}, {390, 225, 503, 1332, 2903, 492, 1120, 2869, 498, 1193, 2865, 503, 1332, 2903, 524, 1333, 2888, 
			568, 1543, 2883, 607, 1634, 2884, 623, 1774, 2893, 624, 1825, 2901, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w02uh"}, {345, 226, 788, 1547, 2917, 788, 1547, 2917, 770, 1516, 2913, 766, 1521, 2925, 766, 1607, 2979, 
			766, 1721, 2955, 754, 1736, 3040, 695, 1803, 3061, 667, 1893, 3061, 65, 0, 0, 0, 0, 0, 7, 0, 0, 0, 0, 13, 0}},
		{{"w02uw"}, {409, 218, 435, 1384, 2804, 445, 1408, 2771, 439, 1417, 2867, 436, 1368, 2840, 436, 1356, 2828, 
			435, 1376, 2812, 435, 1384, 2834, 436, 1364, 2818, 436, 1439, 2858, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w03ae"}, {293, 192, 666, 2370, 2814, 666, 2370, 2814, 701, 2328, 2768, 701, 2322, 2603, 700, 2247, 2607, 
			799, 2074, 2595, 920, 2009, 2671, 931, 1947, 2671, 925, 1983, 2695, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w03ah"}, {257, 192, 1025, 1548, 2748, 1020, 1548, 2639, 1053, 1574, 2748, 1056, 1598, 2745, 1020, 1619, 2738, 
			1014, 1619, 2745, 1020, 1600, 2788, 1009, 1658, 2828, 970, 1787, 2900, 95, 0, 1, 0, 0, 0, 19, 0, 0, 0, 0, 0, 0}},
		{{"w03aw"}, {326, 186, 772, 1051, 2870, 748, 1060, 3056, 766, 1053, 2936, 772, 1051, 2870, 796, 1047, 2813, 
			788, 1067, 2732, 821, 1125, 2705, 855, 1228, 2669, 843, 1415, 2621, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"w03eh"}, {202, 218, 707, 2110, 3028, 732, 2138, 3333, 728, 2098, 3128, 709, 2098, 2966, 708, 2135, 2977, 
			709, 2135, 2995, 709, 2124, 2995, 689, 2120, 2988, 638, 2121, 3036, 95, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0}},
		{{"w03ei"}, {263, 208, 672, 2457, 3190, 615, 2457, 3133, 556, 2529, 3048, 513, 2556, 2963, 501, 2600, 2990, 
			486, 2630, 2990, 441, 2678, 2993, 387, 2718, 3037, 371, 2758, 3046, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w03er"}, {278, 182, 523, 1413, 0, 559, 1381, 1665, 559, 1434, 0, 559, 1395, 0, 524, 1381, 0, 
			530, 1439, 0, 508, 1381, 0, 504, 1430, 0, 504, 1427, 0, 95, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19}},
		{{"w03ih"}, {197, 188, 486, 2332, 2925, 486, 2361, 2950, 497, 2332, 2925, 553, 2332, 2925, 561, 2243, 2927, 
			560, 2209, 2924, 560, 2184, 2924, 559, 2184, 2965, 557, 2188, 2975, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w03iy"}, {289, 192, 380, 2703, 3087, 435, 2673, 3055, 433, 2716, 3060, 421, 2697, 3103, 396, 2688, 3021, 
			380, 2703, 3087, 375, 2695, 3087, 373, 2709, 3042, 369, 2700, 2999, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w03oa"}, {274, 176, 624, 1130, 2881, 635, 1188, 2884, 620, 1101, 2879, 567, 995, 2825, 557, 963, 2896, 
			526, 896, 2903, 495, 822, 2901, 486, 802, 2894, 440, 869, 2949, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w03oo"}, {265, 215, 494, 1102, 2863, 494, 1121, 2854, 501, 1072, 2882, 541, 1036, 2885, 562, 1140, 2935, 
			567, 1137, 2887, 575, 1268, 2828, 606, 1449, 2794, 587, 1609, 2732, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w03uh"}, {206, 182, 789, 1291, 2877, 789, 1305, 2881, 799, 1340, 2863, 811, 1346, 2864, 822, 1346, 2875, 
			808, 1418, 2828, 783, 1503, 2753, 706, 1626, 2741, 669, 1824, 2773, 80, 0, 0, 0, 0, 0, 2, 1, 0, 1, 0, 16, 0}},
		{{"w03uw"}, {271, 211, 431, 788, 2717, 500, 980, 2733, 482, 943, 2657, 451, 927, 2749, 441, 883, 2702, 
			437, 846, 2723, 434, 814, 2746, 429, 794, 2737, 427, 765, 2733, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w04ae"}, {353, 233, 743, 2230, 3055, 743, 2201, 3055, 759, 2193, 3073, 800, 2118, 3053, 803, 2030, 3010, 
			812, 1948, 2951, 822, 1886, 2986, 817, 1873, 3019, 805, 1923, 3119, 95, 0, 0, 1, 0, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"w04ah"}, {350, 216, 804, 1484, 2789, 792, 1450, 2757, 840, 1474, 2764, 811, 1474, 2811, 804, 1484, 2789, 
			798, 1498, 2803, 783, 1499, 2775, 774, 1524, 2751, 759, 1574, 2758, 85, 0, 0, 0, 0, 0, 17, 2, 0, 0, 0, 1, 0}},
		{{"w04aw"}, {379, 203, 770, 1245, 2811, 745, 1197, 2801, 747, 1242, 2785, 748, 1218, 2725, 760, 1225, 2758, 
			771, 1239, 2811, 783, 1254, 2831, 768, 1298, 2745, 759, 1423, 2812, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"w04eh"}, {315, 216, 693, 1991, 3121, 715, 2075, 3133, 697, 2089, 3058, 696, 2028, 3054, 702, 2002, 3056, 
			701, 1992, 3081, 697, 1991, 3106, 690, 2009, 3137, 682, 2011, 3178, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w04ei"}, {437, 226, 601, 2577, 3091, 599, 2365, 3077, 578, 2454, 3056, 559, 2518, 3028, 546, 2539, 3112, 
			516, 2552, 3131, 499, 2546, 3135, 481, 2552, 3120, 446, 2585, 3134, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w04er"}, {350, 216, 619, 1716, 2138, 627, 1709, 2115, 620, 1714, 2138, 619, 1731, 2137, 615, 1743, 2127, 
			612, 1779, 2117, 603, 1817, 2110, 598, 1833, 2128, 575, 1863, 2176, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w04ih"}, {356, 224, 502, 2391, 3071, 497, 2460, 3118, 500, 2410, 3084, 518, 2360, 3062, 547, 2316, 3062, 
			557, 2289, 3102, 559, 2266, 3109, 561, 2229, 3127, 545, 2201, 3170, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w04iy"}, {364, 215, 428, 2767, 3627, 428, 2745, 3679, 430, 2771, 3728, 430, 2795, 3679, 430, 2795, 3625, 
			428, 2785, 3619, 428, 2767, 3627, 429, 2760, 3627, 430, 2774, 3540, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w04oa"}, {410, 225, 627, 1314, 2911, 631, 1335, 2900, 624, 1292, 2922, 614, 1234, 2899, 603, 1191, 2838, 
			570, 1163, 2855, 563, 1147, 2841, 553, 1110, 2804, 508, 1104, 2828, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w04oo"}, {279, 222, 509, 1323, 3122, 509, 1323, 3122, 523, 1353, 2949, 549, 1367, 2928, 559, 1461, 2890, 
			571, 1578, 2845, 572, 1622, 2867, 579, 1746, 2841, 578, 1820, 2822, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 0, 1, 0}},
		{{"w04uh"}, {331, 216, 731, 1552, 3111, 726, 1538, 3034, 726, 1542, 2993, 731, 1552, 3111, 741, 1602, 3105, 
			741, 1662, 3147, 735, 1693, 3134, 709, 1738, 3070, 670, 1933, 3065, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w04uw"}, {430, 216, 436, 1415, 2843, 458, 1560, 2836, 447, 1488, 2869, 441, 1436, 2817, 437, 1419, 2839, 
			436, 1397, 2851, 436, 1402, 2840, 435, 1436, 2809, 435, 1472, 2809, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w05ae"}, {338, 223, 677, 2320, 2987, 678, 2329, 3065, 678, 2263, 2987, 696, 2226, 3003, 744, 2183, 2910, 
			804, 2056, 2808, 821, 1994, 2832, 821, 1986, 2808, 806, 1988, 2810, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"w05ah"}, {365, 226, 935, 1377, 2598, 938, 1468, 2598, 938, 1423, 2626, 949, 1439, 2624, 950, 1481, 2602, 
			961, 1499, 2615, 941, 1520, 2613, 902, 1614, 2632, 864, 1697, 2640, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"w05aw"}, {363, 231, 923, 1368, 2503, 924, 1318, 2488, 919, 1355, 2498, 923, 1368, 2503, 923, 1379, 2499, 
			931, 1426, 2504, 933, 1441, 2467, 932, 1504, 2484, 738, 1646, 2540, 65, 0, 0, 0, 0, 0, 7, 13, 0, 0, 0, 0, 0}},
		{{"w05eh"}, {260, 233, 743, 1981, 2868, 713, 2176, 2900, 729, 2124, 2870, 746, 2052, 2842, 773, 2026, 2854, 
			774, 2004, 2854, 757, 1983, 2851, 743, 1981, 2868, 698, 2010, 2871, 90, 0, 0, 0, 18, 2, 0, 0, 0, 0, 0, 0, 0}},
		{{"w05ei"}, {302, 241, 487, 2600, 3130, 496, 2465, 3185, 491, 2487, 3185, 488, 2562, 3222, 486, 2600, 3169, 
			486, 2620, 3118, 487, 2624, 3155, 486, 2634, 3166, 486, 2623, 3065, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w05er"}, {317, 242, 494, 1485, 1863, 498, 1477, 1994, 503, 1469, 1911, 493, 1515, 1858, 494, 1482, 1873, 
			496, 1529, 1938, 498, 1548, 1975, 499, 1566, 1989, 499, 1680, 1990, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w05ih"}, {266, 238, 497, 2379, 3074, 505, 2362, 3194, 490, 2357, 3036, 496, 2327, 3026, 506, 2188, 2976, 
			542, 2181, 3010, 558, 2187, 2924, 556, 2093, 3000, 548, 2070, 3038, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w05iy"}, {305, 249, 346, 2703, 3510, 317, 2677, 3448, 341, 2697, 3471, 347, 2697, 3510, 339, 2700, 3490, 
			325, 2727, 3510, 328, 2744, 3470, 329, 2744, 3427, 328, 2707, 3416, 95, 19, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0}},
		{{"w05oa"}, {376, 224, 443, 879, 2532, 571, 1033, 2293, 530, 962, 2371, 489, 924, 2436, 460, 883, 2482, 
			441, 879, 2539, 438, 870, 2487, 437, 861, 2540, 436, 889, 2514, 95, 0, 0, 0, 0, 0, 0, 0, 19, 0, 0, 0, 1}},
		{{"w05oo"}, {269, 244, 506, 1188, 2485, 498, 1179, 2517, 506, 1188, 2485, 516, 1209, 2558, 540, 1254, 2574, 
			547, 1345, 2589, 546, 1494, 2624, 540, 1566, 2572, 529, 1750, 2643, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w05uh"}, {290, 224, 693, 1381, 2560, 717, 1370, 2628, 686, 1403, 2584, 673, 1447, 2626, 644, 1480, 2679, 
			659, 1524, 2547, 670, 1606, 2535, 669, 1674, 2607, 643, 1840, 2678, 80, 0, 0, 0, 0, 0, 3, 1, 0, 0, 0, 16, 0}},
		{{"w05uw"}, {328, 223, 455, 994, 2612, 458, 991, 2563, 453, 990, 2611, 455, 994, 2612, 449, 994, 2606, 
			446, 997, 2676, 445, 1001, 2658, 435, 1024, 2624, 406, 1064, 2651, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w06ae"}, {362, 223, 627, 2266, 2875, 631, 2274, 2878, 628, 2224, 2918, 635, 2251, 2889, 677, 2158, 2820, 
			730, 1983, 2791, 773, 1867, 2762, 781, 1758, 2796, 763, 1788, 2760, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"w06ah"}, {329, 211, 804, 1363, 2803, 795, 1455, 2903, 794, 1408, 2756, 804, 1363, 2803, 810, 1370, 2722, 
			827, 1382, 2811, 818, 1457, 2827, 782, 1470, 2777, 752, 1603, 2786, 90, 0, 0, 0, 0, 0, 18, 1, 0, 0, 0, 1, 0}},
		{{"w06aw"}, {363, 212, 778, 1220, 2677, 855, 1314, 2829, 817, 1275, 2903, 806, 1243, 2913, 785, 1230, 2849, 
			778, 1211, 2675, 793, 1207, 2828, 790, 1244, 2725, 783, 1374, 2795, 80, 0, 0, 0, 0, 0, 4, 16, 0, 0, 0, 0, 0}},
		{{"w06eh"}, {280, 211, 712, 1881, 2777, 692, 1856, 2808, 690, 1907, 2842, 712, 1881, 2777, 699, 1826, 2816, 
			690, 1823, 2812, 678, 1828, 2836, 670, 1825, 2787, 641, 1879, 2816, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w06ei"}, {318, 214, 492, 2437, 3001, 513, 2377, 2920, 492, 2437, 3001, 449, 2438, 2996, 439, 2494, 3054, 
			435, 2488, 3097, 433, 2489, 3059, 430, 2496, 3047, 437, 2640, 3127, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w06er"}, {342, 225, 527, 1636, 1974, 523, 1642, 1996, 518, 1628, 1991, 501, 1619, 1973, 498, 1622, 1934, 
			498, 1623, 1913, 485, 1681, 0, 459, 1704, 0, 444, 1704, 0, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w06ih"}, {266, 216, 531, 2174, 2984, 474, 2138, 2951, 476, 2158, 3005, 496, 2185, 2974, 531, 2174, 2984, 
			517, 2125, 2989, 506, 1955, 2939, 496, 1941, 2938, 492, 1964, 2947, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w06iy"}, {332, 221, 439, 2676, 3291, 447, 2550, 3249, 446, 2532, 3287, 444, 2516, 3312, 439, 2666, 3250, 
			437, 2690, 3283, 438, 2677, 3302, 438, 2670, 3200, 439, 2585, 3068, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w06oa"}, {353, 203, 580, 995, 2637, 620, 1004, 2575, 593, 999, 2647, 574, 995, 2632, 562, 990, 2629, 
			550, 985, 2634, 507, 990, 2665, 473, 994, 2697, 443, 1009, 2717, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w06oo"}, {283, 207, 483, 987, 2679, 472, 1042, 2645, 494, 1020, 2696, 512, 1136, 2697, 561, 1180, 2663, 
			581, 1174, 2672, 606, 1382, 2634, 610, 1489, 2690, 593, 1582, 2691, 90, 0, 0, 0, 0, 0, 0, 0, 0, 18, 2, 0, 0}},
		{{"w06uh"}, {233, 226, 795, 1362, 2671, 817, 1537, 2764, 778, 1434, 2685, 795, 1362, 2671, 790, 1319, 2713, 
			779, 1362, 2868, 755, 1452, 2864, 719, 1540, 2851, 666, 1700, 2858, 90, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 18, 0}},
		{{"w06uw"}, {311, 219, 437, 1113, 2676, 444, 1101, 2566, 438, 1115, 2682, 437, 1108, 2668, 435, 1097, 2675, 
			434, 1088, 2685, 435, 1098, 2727, 436, 1105, 2647, 440, 1143, 2673, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w07ae"}, {313, 176, 690, 2327, 2771, 725, 2374, 2916, 695, 2277, 2837, 692, 2313, 2755, 734, 2370, 2728, 
			805, 2088, 2709, 815, 1975, 2691, 829, 1936, 2676, 816, 1938, 2753, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w07ah"}, {306, 173, 939, 1233, 2564, 994, 1227, 2695, 920, 1191, 2670, 914, 1164, 2651, 934, 1222, 2576, 
			942, 1237, 2560, 940, 1228, 2635, 928, 1290, 2695, 884, 1476, 2721, 85, 0, 0, 0, 0, 0, 17, 3, 0, 0, 0, 0, 0}},
		{{"w07aw"}, {314, 167, 809, 1097, 2646, 734, 1110, 2699, 736, 1085, 2621, 782, 1073, 2684, 815, 1096, 2690, 
			814, 1164, 2653, 812, 1235, 2643, 814, 1292, 2577, 820, 1395, 2547, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"w07eh"}, {238, 198, 734, 2141, 2746, 680, 2193, 2928, 689, 2235, 2801, 723, 2185, 2667, 742, 2145, 2772, 
			740, 2105, 2723, 721, 2095, 2712, 692, 2078, 2725, 643, 2150, 2803, 85, 0, 0, 0, 17, 3, 0, 0, 0, 0, 0, 0, 0}},
		{{"w07ei"}, {330, 186, 553, 2458, 2788, 565, 2397, 2990, 561, 2485, 2851, 555, 2451, 2839, 539, 2523, 2789, 
			510, 2550, 2868, 498, 2640, 2917, 475, 2665, 2921, 444, 2708, 2907, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w07er"}, {320, 174, 518, 1604, 1912, 515, 1741, 2499, 568, 1645, 2429, 564, 1624, 2285, 564, 1567, 2158, 
			559, 1569, 2019, 544, 1628, 1942, 507, 1632, 1888, 505, 1586, 1863, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w07ih"}, {285, 192, 535, 2380, 2930, 497, 2447, 2987, 535, 2380, 2930, 562, 2423, 2865, 561, 2364, 2816, 
			556, 2238, 2812, 561, 2187, 2823, 564, 2114, 2914, 556, 2111, 2971, 95, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w07iy"}, {281, 200, 394, 2689, 3058, 411, 2689, 3109, 410, 2710, 3064, 402, 2704, 3022, 390, 2758, 3058, 
			389, 2801, 3058, 386, 2709, 3022, 385, 2695, 3071, 392, 2688, 3068, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w07oa"}, {262, 196, 621, 1025, 2704, 617, 1049, 2860, 619, 1009, 2668, 608, 1007, 2716, 572, 1019, 2764, 
			560, 995, 2819, 524, 952, 2821, 510, 940, 2882, 502, 998, 2816, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w07oo"}, {227, 182, 563, 1233, 2758, 573, 1176, 2803, 566, 1231, 2750, 563, 1238, 2755, 562, 1185, 2741, 
			569, 1221, 2720, 588, 1286, 2702, 611, 1455, 2695, 586, 1634, 2719, 95, 0, 0, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0}},
		{{"w07uh"}, {209, 189, 854, 1401, 2756, 824, 1430, 2789, 819, 1387, 2737, 815, 1352, 2690, 816, 1351, 2690, 
			817, 1371, 2714, 815, 1430, 2748, 791, 1556, 2820, 695, 1715, 2882, 65, 0, 0, 0, 0, 0, 7, 0, 0, 0, 0, 13, 0}},
		{{"w07uw"}, {384, 211, 384, 1084, 2929, 437, 1078, 2854, 427, 1032, 2907, 396, 1037, 2924, 384, 1098, 2959, 
			384, 1137, 2939, 392, 1177, 3014, 415, 1194, 2908, 432, 1298, 2962, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w08ae"}, {284, 238, 658, 2650, 3471, 646, 2791, 3480, 668, 2608, 3520, 646, 2706, 3458, 684, 2570, 3430, 
			719, 2268, 3435, 724, 2204, 3437, 733, 2224, 3448, 734, 2119, 3525, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"w08ah"}, {317, 243, 827, 1701, 3056, 857, 1695, 3004, 845, 1702, 3010, 827, 1701, 3056, 853, 1731, 2980, 
			817, 1734, 3092, 822, 1735, 3111, 834, 1807, 3135, 814, 1933, 3153, 95, 0, 0, 0, 0, 0, 19, 0, 0, 0, 0, 1, 0}},
		{{"w08aw"}, {320, 242, 815, 1426, 2990, 855, 1432, 3098, 842, 1435, 3037, 820, 1417, 3002, 815, 1426, 2990, 
			819, 1429, 2955, 817, 1355, 2941, 821, 1535, 2932, 800, 1667, 3106, 70, 0, 0, 0, 0, 0, 5, 14, 0, 0, 0, 1, 0}},
		{{"w08eh"}, {199, 244, 732, 2189, 3366, 693, 2361, 3318, 713, 2300, 3273, 740, 2233, 3370, 732, 2189, 3366, 
			738, 2192, 3369, 736, 2192, 3433, 734, 2200, 3420, 706, 2225, 3483, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w08ei"}, {274, 238, 496, 2665, 3173, 496, 2665, 3173, 493, 2668, 3210, 490, 2749, 3100, 484, 2807, 3209, 
			470, 2816, 3181, 468, 2817, 3236, 480, 2822, 3121, 489, 2844, 3308, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w08er"}, {272, 250, 516, 1983, 0, 543, 1979, 0, 525, 1978, 0, 510, 1994, 0, 506, 1999, 0, 
			504, 2005, 0, 505, 2042, 0, 505, 2054, 0, 508, 1886, 2194, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w08ih"}, {180, 247, 498, 2654, 3391, 498, 2634, 3308, 497, 2666, 3424, 497, 2644, 3371, 498, 2620, 3377, 
			498, 2532, 3375, 499, 2460, 3365, 501, 2435, 3351, 501, 2425, 3380, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w08iy"}, {297, 264, 493, 2967, 3496, 499, 2871, 3640, 499, 2891, 3614, 499, 2898, 3543, 495, 2916, 3490, 
			493, 2967, 3496, 494, 2899, 3417, 496, 2889, 3388, 496, 2887, 3404, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w08oa"}, {304, 253, 506, 1257, 2957, 662, 1369, 2924, 639, 1412, 2995, 609, 1364, 3060, 572, 1313, 3026, 
			535, 1249, 2951, 506, 1257, 2957, 502, 1254, 2858, 505, 1264, 2806, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w08oo"}, {210, 254, 546, 1534, 2880, 520, 1517, 2751, 539, 1514, 2823, 552, 1551, 2916, 557, 1587, 2924, 
			560, 1674, 2962, 562, 1765, 2991, 561, 1805, 2978, 559, 2000, 3059, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w08uh"}, {207, 243, 716, 1634, 3119, 716, 1611, 3010, 735, 1667, 3246, 741, 1668, 3224, 723, 1687, 3229, 
			726, 1737, 3396, 734, 1754, 3274, 725, 1885, 3275, 695, 1985, 3341, 95, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19, 0}},
		{{"w08uw"}, {287, 258, 503, 1514, 2816, 512, 1550, 2769, 504, 1517, 2806, 502, 1516, 2849, 502, 1518, 2836, 
			501, 1538, 2814, 500, 1482, 2802, 500, 1348, 2750, 500, 1249, 2787, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w09ae"}, {253, 251, 685, 2299, 2930, 694, 2270, 2793, 688, 2330, 2808, 684, 2317, 2902, 688, 2282, 2942, 
			714, 2245, 2975, 752, 2104, 2906, 785, 2015, 2840, 774, 1870, 2741, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"w09ah"}, {282, 240, 913, 1436, 2589, 900, 1465, 2564, 913, 1436, 2589, 927, 1452, 2629, 954, 1487, 2658, 
			961, 1496, 2624, 969, 1492, 2538, 972, 1510, 2558, 905, 1597, 2604, 90, 0, 0, 0, 0, 0, 18, 2, 0, 0, 0, 0, 0}},
		{{"w09aw"}, {309, 241, 890, 1063, 2523, 746, 992, 2504, 840, 1020, 2491, 880, 1040, 2576, 882, 1070, 2539, 
			878, 1097, 2508, 848, 1127, 2486, 877, 1209, 2512, 862, 1422, 2563, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"w09eh"}, {233, 246, 710, 2138, 2818, 698, 2126, 2768, 710, 2138, 2818, 720, 2176, 2837, 732, 2143, 2725, 
			736, 2081, 2748, 730, 2049, 2805, 712, 2047, 2811, 683, 1945, 2777, 95, 0, 0, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0}},
		{{"w09ei"}, {222, 265, 505, 2777, 3044, 503, 2502, 3017, 502, 2615, 3195, 502, 2644, 3129, 505, 2777, 3044, 
			522, 2690, 3021, 521, 2699, 3122, 524, 2697, 3251, 521, 2747, 3100, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w09er"}, {261, 248, 503, 1480, 1735, 523, 1430, 1863, 524, 1457, 1854, 506, 1472, 1805, 503, 1486, 1734, 
			503, 1488, 1682, 506, 1503, 1727, 512, 1521, 1794, 537, 1577, 1863, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w09ih"}, {218, 253, 500, 2272, 3028, 499, 2337, 3004, 499, 2281, 2927, 500, 2272, 3028, 500, 2251, 3037, 
			501, 2246, 3024, 504, 2249, 3018, 512, 2241, 3000, 515, 2167, 3041, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w09iy"}, {246, 266, 442, 2850, 3157, 463, 2866, 3284, 474, 2744, 3311, 465, 2794, 3341, 442, 2850, 3157, 
			426, 2815, 3069, 412, 2820, 3229, 339, 2838, 3185, 321, 2876, 3072, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w09oa"}, {257, 252, 538, 998, 2796, 570, 938, 2676, 562, 977, 2727, 549, 995, 2766, 525, 998, 2808, 
			507, 999, 2905, 505, 992, 2878, 511, 804, 2893, 526, 793, 2798, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w09oo"}, {209, 247, 500, 1200, 2783, 504, 1244, 2756, 499, 1201, 2817, 500, 1230, 2853, 502, 1245, 2785, 
			511, 1297, 2819, 527, 1311, 2844, 551, 1359, 2857, 558, 1624, 2819, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 0, 1, 0}},
		{{"w09uh"}, {193, 236, 809, 1391, 2927, 694, 1413, 2854, 886, 1402, 2798, 818, 1365, 2845, 809, 1391, 2927, 
			776, 1405, 2943, 749, 1430, 2948, 719, 1478, 2891, 703, 1743, 2809, 95, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 19, 0}},
		{{"w09uw"}, {265, 265, 504, 1096, 2665, 500, 1189, 2722, 500, 1163, 2751, 503, 1125, 2715, 504, 1096, 2665, 
			500, 1066, 2679, 492, 1079, 2700, 464, 1115, 2756, 443, 1136, 2729, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w10ae"}, {356, 227, 621, 2249, 2873, 613, 2260, 2919, 609, 2261, 2917, 614, 2244, 2999, 623, 2198, 2997, 
			649, 2148, 2934, 694, 1972, 2874, 728, 1846, 2844, 734, 1826, 2858, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w10ah"}, {367, 223, 856, 1540, 2667, 878, 1428, 2639, 855, 1467, 2625, 848, 1533, 2658, 861, 1546, 2679, 
			840, 1541, 2687, 836, 1560, 2714, 824, 1604, 2745, 804, 1701, 2831, 95, 0, 0, 0, 0, 0, 19, 0, 1, 0, 0, 0, 0}},
		{{"w10aw"}, {416, 225, 771, 1170, 2760, 702, 1148, 2611, 758, 1182, 2733, 786, 1188, 2809, 790, 1237, 2848, 
			791, 1265, 2868, 782, 1312, 2870, 796, 1422, 2868, 789, 1601, 2820, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"w10eh"}, {271, 227, 704, 2023, 2927, 680, 1994, 3003, 719, 2043, 2938, 697, 1995, 3063, 707, 1909, 3072, 
			711, 1844, 3058, 701, 1813, 3052, 710, 1805, 2966, 693, 1829, 2992, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w10ei"}, {347, 220, 455, 2433, 3023, 475, 2465, 2979, 445, 2441, 3033, 438, 2479, 3004, 436, 2513, 2977, 
			434, 2541, 2923, 433, 2547, 2912, 432, 2567, 2919, 436, 2640, 2976, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w10er"}, {342, 230, 495, 1596, 1973, 517, 1466, 1798, 510, 1559, 1840, 503, 1576, 1953, 495, 1582, 1971, 
			489, 1601, 1978, 483, 1618, 1968, 482, 1614, 1925, 482, 1603, 1985, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w10ih"}, {249, 227, 471, 2277, 2884, 498, 2311, 2881, 484, 2317, 2954, 473, 2275, 2872, 481, 2248, 2920, 
			493, 2199, 2919, 502, 2064, 2936, 530, 2012, 3046, 507, 2039, 3006, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w10iy"}, {324, 239, 477, 2866, 3236, 492, 2770, 3161, 486, 2847, 3099, 481, 2872, 3283, 474, 2863, 3230, 
			477, 2871, 3238, 483, 2870, 3288, 476, 2838, 3230, 468, 2820, 3220, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w10oa"}, {363, 219, 453, 980, 2854, 570, 1129, 2659, 506, 1070, 2759, 490, 1026, 2875, 476, 996, 2876, 
			472, 992, 2863, 453, 980, 2854, 447, 984, 2848, 442, 1004, 2828, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w10oo"}, {253, 231, 491, 1179, 2636, 496, 1188, 2619, 490, 1183, 2637, 494, 1244, 2672, 507, 1384, 2761, 
			535, 1434, 2792, 561, 1601, 2813, 565, 1690, 2816, 558, 1874, 2828, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w10uh"}, {256, 217, 680, 1499, 2885, 670, 1307, 2802, 687, 1302, 2744, 692, 1372, 2849, 682, 1474, 2864, 
			678, 1503, 2907, 673, 1543, 2890, 668, 1668, 2894, 642, 1748, 2963, 90, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 18, 0}},
		{{"w10uw"}, {351, 238, 473, 1113, 2626, 500, 1179, 2632, 493, 1161, 2627, 491, 1125, 2645, 487, 1104, 2632, 
			472, 1116, 2644, 476, 1126, 2688, 468, 1149, 2701, 477, 1177, 2678, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w11ae"}, {385, 188, 868, 2004, 2797, 751, 2020, 2719, 801, 2012, 2772, 820, 1990, 2754, 875, 1996, 2816, 
			878, 1919, 2746, 891, 1778, 2717, 922, 1727, 2662, 928, 1707, 2691, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w11ah"}, {380, 191, 882, 1380, 2834, 1051, 1548, 2886, 933, 1458, 2872, 891, 1388, 2899, 879, 1376, 2848, 
			877, 1421, 2827, 882, 1488, 2831, 830, 1528, 2835, 838, 1571, 2784, 90, 0, 0, 0, 0, 0, 18, 2, 0, 0, 0, 0, 0}},
		{{"w11aw"}, {380, 197, 755, 1114, 2766, 694, 1109, 2779, 739, 1116, 2802, 755, 1114, 2766, 761, 1117, 2760, 
			792, 1135, 2752, 814, 1171, 2720, 830, 1171, 2671, 843, 1253, 2680, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"w11eh"}, {317, 195, 766, 1922, 2908, 671, 1989, 2916, 701, 1982, 2968, 744, 1952, 2952, 759, 1928, 2912, 
			778, 1900, 2849, 790, 1892, 2848, 775, 1884, 2886, 751, 1886, 2909, 65, 0, 0, 0, 13, 6, 1, 0, 0, 0, 0, 0, 0}},
		{{"w11ei"}, {413, 214, 574, 2316, 2854, 587, 2262, 2923, 574, 2316, 2854, 562, 2319, 2892, 553, 2314, 2869, 
			561, 2347, 2879, 539, 2380, 2935, 486, 2372, 2993, 427, 2460, 2955, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w11er"}, {389, 195, 567, 1496, 1803, 574, 1473, 1818, 569, 1486, 1853, 562, 1514, 1742, 565, 1486, 1786, 
			562, 1511, 1867, 560, 1518, 1888, 558, 1617, 1915, 552, 1759, 2016, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w11ih"}, {320, 207, 453, 2427, 2932, 470, 2385, 2936, 542, 2304, 2932, 564, 2217, 2912, 578, 2153, 2913, 
			588, 2092, 2887, 606, 2037, 2929, 610, 2012, 2987, 595, 2030, 2985, 90, 0, 18, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w11iy"}, {433, 203, 408, 2627, 3064, 441, 2643, 3148, 435, 2597, 3118, 433, 2624, 3070, 430, 2686, 3040, 
			419, 2656, 3028, 408, 2626, 3065, 409, 2629, 3045, 410, 2630, 3008, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w11oa"}, {372, 196, 597, 1150, 2751, 630, 1187, 2744, 622, 1178, 2746, 613, 1164, 2747, 583, 1136, 2747, 
			568, 1123, 2745, 565, 1119, 2751, 553, 1095, 2780, 504, 1075, 2797, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w11oo"}, {332, 200, 575, 1619, 2737, 509, 1628, 2718, 560, 1635, 2729, 577, 1619, 2743, 581, 1623, 2736, 
			586, 1664, 2743, 599, 1703, 2756, 611, 1746, 2749, 601, 1795, 2809, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"w11uh"}, {266, 205, 773, 1614, 2813, 743, 1578, 2831, 757, 1598, 2795, 773, 1614, 2813, 777, 1613, 2836, 
			770, 1665, 2826, 759, 1721, 2850, 746, 1809, 2861, 668, 1842, 2893, 90, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 18, 0}},
		{{"w11uw"}, {377, 232, 461, 1627, 2761, 467, 1634, 2740, 456, 1625, 2760, 449, 1619, 2758, 445, 1604, 2805, 
			445, 1611, 2804, 448, 1600, 2814, 455, 1448, 2760, 475, 1394, 2784, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w12ae"}, {355, 229, 682, 2486, 3207, 682, 2476, 3185, 674, 2484, 3140, 658, 2528, 3105, 654, 2497, 3077, 
			694, 2442, 2994, 771, 2281, 2831, 803, 2100, 2739, 803, 2051, 2762, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w12ah"}, {343, 216, 887, 1743, 2557, 877, 1769, 2538, 885, 1711, 2504, 879, 1739, 2538, 883, 1757, 2558, 
			902, 1738, 2571, 918, 1711, 2564, 885, 1760, 2601, 836, 1867, 2666, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"w12aw"}, {397, 227, 930, 1422, 2489, 834, 1379, 2542, 928, 1431, 2470, 936, 1414, 2487, 932, 1384, 2488, 
			928, 1424, 2488, 930, 1518, 2497, 920, 1596, 2515, 904, 1754, 2649, 30, 0, 0, 0, 0, 0, 14, 6, 0, 0, 0, 0, 0}},
		{{"w12eh"}, {234, 210, 673, 2389, 2989, 683, 2407, 2988, 678, 2400, 3002, 685, 2384, 2963, 678, 2413, 2969, 
			667, 2381, 3002, 665, 2369, 3036, 648, 2363, 3043, 633, 2346, 3094, 80, 0, 0, 0, 16, 4, 0, 0, 0, 0, 0, 0, 0}},
		{{"w12ei"}, {329, 238, 502, 2674, 3394, 513, 2693, 3435, 498, 2651, 3440, 485, 2770, 0, 462, 3023, 0, 
			458, 3044, 0, 472, 3052, 0, 474, 3062, 0, 489, 3106, 0, 95, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w12er"}, {348, 219, 501, 1555, 1938, 513, 1621, 1982, 511, 1570, 1932, 497, 1548, 1938, 490, 1551, 1958, 
			484, 1554, 1943, 484, 1553, 1943, 478, 1554, 1937, 467, 1589, 1937, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w12ih"}, {246, 237, 489, 2603, 3353, 495, 2630, 3328, 486, 2585, 3350, 480, 2559, 3306, 484, 2507, 3274, 
			492, 2492, 3222, 499, 2451, 3114, 504, 2416, 3104, 500, 2389, 3091, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w12iy"}, {359, 223, 435, 2868, 3625, 456, 2876, 3631, 446, 2861, 3662, 440, 2819, 3816, 435, 2877, 3716, 
			435, 2863, 3703, 435, 2865, 3607, 434, 2862, 3631, 433, 2828, 3605, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w12oa"}, {319, 233, 688, 1185, 2748, 703, 1287, 2794, 686, 1185, 2726, 658, 1164, 2741, 607, 1125, 2840, 
			561, 1123, 2918, 516, 1115, 2912, 492, 1068, 2917, 458, 1074, 2926, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w12oo"}, {289, 225, 502, 1322, 2951, 500, 1312, 2941, 502, 1322, 2951, 531, 1338, 2979, 558, 1426, 2960, 
			568, 1512, 2877, 599, 1583, 2920, 610, 1709, 2903, 610, 1934, 2915, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w12uh"}, {248, 241, 817, 1604, 2649, 809, 1625, 2641, 819, 1604, 2649, 816, 1625, 2718, 800, 1638, 2721, 
			772, 1745, 2755, 748, 1822, 2831, 725, 1959, 2884, 688, 2068, 2912, 95, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 19, 0}},
		{{"w12uw"}, {300, 242, 477, 998, 2690, 498, 1223, 2962, 494, 1116, 2978, 489, 994, 2941, 483, 977, 2862, 
			477, 998, 2690, 478, 974, 2858, 467, 1030, 2758, 480, 1035, 2802, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w13ae"}, {315, 237, 668, 2252, 2790, 664, 2260, 2781, 676, 2246, 2776, 687, 2187, 2755, 711, 2056, 2739, 
			727, 1921, 2727, 807, 1842, 2756, 841, 1865, 2773, 781, 1875, 2757, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w13ah"}, {329, 226, 869, 1495, 2731, 816, 1495, 2731, 897, 1525, 2748, 890, 1491, 2721, 891, 1498, 2726, 
			929, 1545, 2716, 905, 1558, 2708, 887, 1673, 2732, 817, 1736, 2753, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"w13aw"}, {363, 234, 697, 1142, 2507, 697, 1142, 2515, 699, 1154, 2538, 709, 1190, 2560, 723, 1314, 2623, 
			735, 1419, 2641, 740, 1534, 2655, 747, 1593, 2629, 711, 1738, 2639, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"w13eh"}, {231, 214, 651, 2157, 2794, 695, 2168, 2818, 694, 2116, 2736, 679, 2047, 2693, 675, 2131, 2757, 
			653, 2168, 2784, 646, 2144, 2805, 636, 2134, 2819, 629, 2133, 2852, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w13ei"}, {318, 236, 647, 2539, 0, 673, 2430, 0, 617, 2530, 0, 547, 2521, 0, 490, 2553, 0, 
			457, 2621, 0, 447, 2669, 0, 443, 2720, 0, 440, 2701, 0, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w13er"}, {275, 244, 531, 1688, 1971, 562, 1605, 1966, 553, 1683, 1944, 531, 1688, 1971, 518, 1691, 2017, 
			511, 1691, 2013, 508, 1710, 2002, 505, 1735, 2021, 507, 1759, 2045, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w13ih"}, {219, 233, 555, 2216, 2812, 560, 2401, 2924, 558, 2270, 2838, 557, 2304, 2871, 555, 2227, 2858, 
			557, 2203, 2815, 537, 2266, 2819, 505, 2299, 2849, 492, 2308, 2862, 95, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w13iy"}, {336, 229, 438, 2813, 3532, 487, 2926, 3424, 476, 2895, 3509, 447, 2824, 3523, 439, 2794, 3603, 
			438, 2822, 3472, 439, 2846, 3472, 437, 2823, 3459, 439, 2761, 3439, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w13oa"}, {315, 236, 631, 1181, 2458, 637, 1189, 2458, 616, 1168, 2505, 565, 1142, 2514, 519, 1130, 2542, 
			501, 1050, 2552, 489, 973, 2543, 483, 949, 2510, 484, 974, 2486, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w13oo"}, {227, 240, 552, 1193, 2559, 552, 1193, 2559, 553, 1172, 2530, 561, 1174, 2522, 573, 1357, 2541, 
			591, 1480, 2522, 622, 1691, 2526, 629, 1768, 2471, 617, 1998, 2569, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w13uh"}, {221, 235, 739, 1320, 2596, 779, 1347, 2579, 738, 1292, 2636, 723, 1312, 2573, 705, 1348, 2610, 
			711, 1465, 2583, 684, 1552, 2541, 675, 1617, 2438, 646, 1755, 2517, 80, 0, 0, 0, 0, 0, 3, 1, 0, 0, 0, 16, 0}},
		{{"w13uw"}, {298, 231, 445, 1136, 2692, 499, 1277, 2670, 494, 1211, 2645, 484, 1190, 2632, 472, 1184, 2621, 
			457, 1177, 2641, 447, 1147, 2706, 445, 1136, 2692, 445, 1161, 2714, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w14ae"}, {272, 214, 726, 2350, 2996, 693, 2376, 3120, 697, 2358, 3047, 730, 2350, 2993, 745, 2358, 3076, 
			789, 2326, 3055, 816, 2206, 2972, 822, 2095, 2935, 812, 2035, 2826, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w14ah"}, {275, 211, 994, 1609, 2930, 999, 1641, 2916, 997, 1613, 2934, 1016, 1627, 2949, 1018, 1645, 2929, 
			996, 1663, 2867, 976, 1668, 2765, 946, 1681, 2725, 927, 1745, 2786, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"w14aw"}, {266, 223, 748, 1063, 2876, 740, 1063, 2870, 765, 1101, 2870, 766, 1062, 2821, 770, 1044, 2811, 
			777, 1053, 2826, 817, 1177, 2809, 862, 1303, 2749, 878, 1441, 2747, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"w14eh"}, {241, 226, 817, 2242, 3139, 816, 2319, 3116, 820, 2220, 3166, 822, 2312, 3200, 810, 2210, 3192, 
			782, 2218, 3132, 771, 2224, 3173, 777, 2260, 3195, 756, 2298, 3159, 95, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0}},
		{{"w14ei"}, {230, 206, 596, 2478, 2924, 581, 2509, 3044, 596, 2478, 2924, 560, 2485, 2785, 500, 2518, 0, 
			449, 2614, 0, 435, 2753, 0, 411, 2987, 0, 383, 2992, 0, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w14er"}, {267, 225, 506, 1604, 1915, 506, 1563, 1923, 514, 1617, 1925, 507, 1605, 1918, 505, 1604, 1913, 
			504, 1618, 1917, 498, 1635, 1914, 491, 1620, 1955, 444, 1685, 1840, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w14ih"}, {209, 221, 491, 2316, 2934, 495, 2509, 3083, 495, 2429, 3063, 490, 2344, 3032, 491, 2316, 2934, 
			505, 2317, 2863, 557, 2308, 2920, 570, 2259, 2935, 562, 2208, 2972, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w14iy"}, {261, 225, 440, 2706, 3404, 440, 2717, 3426, 438, 2694, 3389, 439, 2692, 3355, 438, 2707, 3320, 
			436, 2751, 3285, 435, 2761, 3250, 431, 2757, 3203, 421, 2749, 3168, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w14oa"}, {264, 216, 636, 1041, 2862, 605, 949, 2730, 644, 1002, 2905, 637, 1042, 2832, 623, 1002, 2860, 
			595, 994, 2847, 537, 968, 2843, 454, 897, 2844, 423, 924, 2793, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w14oo"}, {227, 224, 499, 1103, 2900, 499, 1092, 2823, 500, 1110, 2900, 510, 1185, 2878, 537, 1229, 2873, 
			586, 1366, 2815, 619, 1485, 2811, 620, 1707, 2867, 608, 1859, 2827, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w14uh"}, {206, 220, 742, 1352, 2960, 776, 1189, 2903, 740, 1118, 2958, 721, 1330, 2969, 742, 1352, 2960, 
			734, 1345, 2940, 734, 1517, 2810, 694, 1709, 2866, 624, 1819, 2770, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w14uw"}, {241, 239, 483, 1117, 2707, 486, 1181, 2720, 488, 1120, 2706, 479, 1117, 2708, 465, 1117, 2736, 
			452, 1120, 2741, 442, 1120, 2711, 436, 1084, 2712, 426, 1067, 2692, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19, 0, 0}},
		{{"w15ae"}, {400, 226, 620, 2316, 2849, 573, 2349, 2955, 602, 2378, 2931, 631, 2304, 2823, 662, 2129, 2752, 
			687, 1962, 2749, 725, 1897, 2775, 738, 1776, 2793, 709, 1774, 2819, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w15ah"}, {340, 226, 955, 1615, 2678, 901, 1676, 2593, 908, 1640, 2628, 943, 1620, 2665, 960, 1616, 2678, 
			947, 1623, 2682, 921, 1629, 2679, 876, 1691, 2687, 803, 1743, 2726, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"w15aw"}, {394, 229, 827, 1189, 2633, 825, 1183, 2640, 838, 1225, 2641, 813, 1305, 2683, 800, 1298, 2660, 
			848, 1370, 2693, 843, 1429, 2684, 832, 1479, 2691, 803, 1621, 2733, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"w15eh"}, {217, 222, 684, 1992, 2951, 683, 1966, 2890, 687, 1978, 2971, 688, 2007, 2973, 687, 2010, 2911, 
			688, 1997, 2918, 684, 1992, 2951, 676, 1989, 2963, 648, 1991, 2979, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w15ei"}, {325, 244, 500, 2482, 3060, 505, 2512, 3039, 497, 2466, 3085, 492, 2556, 3109, 474, 2607, 3117, 
			451, 2740, 3166, 441, 2740, 3147, 437, 2822, 3249, 434, 2810, 3238, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w15er"}, {355, 234, 484, 1652, 1928, 500, 1682, 1939, 497, 1634, 1922, 495, 1641, 1921, 484, 1652, 1928, 
			472, 1633, 1954, 458, 1628, 1913, 457, 1644, 1910, 460, 1859, 1972, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w15ih"}, {218, 234, 485, 2129, 2967, 499, 2192, 2985, 489, 2133, 2965, 480, 2125, 2989, 464, 2115, 2992, 
			457, 2115, 2965, 455, 2137, 2932, 452, 2116, 2912, 459, 2076, 2887, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w15iy"}, {321, 259, 455, 2797, 3129, 469, 2795, 3054, 440, 2767, 3142, 441, 2768, 3105, 440, 2806, 3116, 
			446, 2779, 3135, 447, 2780, 3143, 466, 2828, 3129, 466, 2851, 3143, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w15oa"}, {357, 226, 449, 924, 2694, 688, 1141, 2749, 613, 1130, 2685, 563, 1087, 2722, 511, 1026, 2748, 
			491, 980, 2725, 455, 930, 2697, 447, 922, 2699, 447, 939, 2725, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w15oo"}, {240, 248, 501, 1189, 2647, 502, 1189, 2666, 500, 1191, 2653, 500, 1237, 2676, 501, 1369, 2663, 
			502, 1476, 2652, 504, 1537, 2671, 509, 1701, 2694, 519, 1786, 2756, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w15uh"}, {288, 226, 700, 1338, 2750, 696, 1374, 2765, 722, 1332, 2798, 700, 1338, 2750, 691, 1309, 2758, 
			682, 1328, 2824, 680, 1492, 2841, 648, 1650, 2833, 626, 1745, 2823, 95, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 19, 0}},
		{{"w15uw"}, {298, 262, 466, 778, 2699, 375, 861, 2643, 368, 843, 2630, 362, 765, 2799, 466, 778, 2699, 
			509, 815, 2716, 493, 865, 2742, 486, 915, 2679, 472, 946, 2750, 95, 0, 0, 0, 0, 0, 0, 0, 1, 0, 19, 0, 0}},
		{{"w16ae"}, {356, 202, 634, 2596, 3255, 634, 2569, 3208, 651, 2545, 3169, 664, 2555, 3026, 764, 2459, 2983, 
			778, 2276, 2954, 927, 1939, 2861, 941, 1880, 2987, 842, 1913, 3064, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w16ah"}, {305, 189, 1085, 1687, 2870, 1258, 1781, 2848, 1149, 1762, 2890, 1124, 1745, 2877, 1085, 1687, 2870, 
			1072, 1733, 2898, 1024, 1769, 2837, 961, 1831, 2855, 775, 1910, 2973, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"w16aw"}, {342, 201, 847, 1192, 2999, 868, 1248, 3005, 863, 1233, 2961, 857, 1218, 2970, 850, 1200, 3010, 
			844, 1185, 2996, 839, 1170, 2997, 833, 1255, 2997, 921, 1492, 2982, 40, 0, 0, 0, 0, 0, 12, 8, 0, 0, 0, 0, 0}},
		{{"w16eh"}, {253, 197, 756, 2008, 3089, 783, 2193, 2904, 778, 2166, 3082, 775, 2120, 3110, 770, 2058, 3084, 
			766, 2013, 3095, 748, 2013, 3103, 731, 2014, 3104, 688, 2058, 3119, 95, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0}},
		{{"w16ei"}, {336, 207, 617, 2697, 3154, 630, 2664, 3244, 623, 2663, 3159, 608, 2688, 3149, 570, 2727, 3140, 
			509, 2767, 3128, 460, 2797, 3213, 439, 2798, 3192, 430, 2812, 3258, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w16er"}, {295, 219, 597, 1588, 1905, 617, 1594, 1987, 612, 1585, 1913, 579, 1604, 1875, 554, 1584, 1875, 
			542, 1610, 1871, 511, 1619, 1876, 481, 1682, 1904, 462, 1668, 1930, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w16ih"}, {290, 214, 457, 2590, 3232, 448, 2700, 3284, 457, 2590, 3232, 525, 2501, 3074, 580, 2364, 3075, 
			619, 2252, 2982, 621, 2215, 3034, 618, 2173, 3013, 619, 2158, 3041, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w16iy"}, {278, 227, 443, 2944, 3546, 461, 3007, 3565, 448, 2957, 3555, 445, 2953, 3548, 440, 2932, 3575, 
			439, 2913, 3567, 437, 2894, 3558, 437, 2888, 3528, 441, 2908, 3459, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w16oa"}, {346, 209, 612, 1026, 2896, 644, 1084, 2814, 622, 1051, 2910, 612, 1026, 2896, 581, 999, 2907, 
			567, 989, 2887, 530, 987, 2888, 496, 999, 2835, 446, 1020, 2808, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w16oo"}, {273, 219, 562, 1109, 2904, 526, 1107, 2905, 562, 1109, 2904, 580, 1096, 2948, 621, 1178, 2939, 
			624, 1235, 2940, 625, 1274, 2930, 626, 1440, 2887, 626, 1570, 2861, 90, 0, 0, 0, 0, 0, 0, 1, 0, 18, 0, 1, 0}},
		{{"w16uh"}, {250, 198, 824, 1459, 2924, 846, 1452, 2937, 814, 1457, 2930, 798, 1463, 2964, 806, 1468, 2985, 
			795, 1473, 2992, 792, 1480, 2955, 766, 1503, 2821, 735, 1698, 2786, 75, 0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 15, 0}},
		{{"w16uw"}, {268, 225, 440, 1122, 2856, 456, 1268, 2845, 453, 1173, 2823, 446, 1116, 2844, 442, 1116, 2849, 
			440, 1122, 2856, 440, 1126, 2873, 440, 1136, 2889, 441, 1195, 2893, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w17ae"}, {323, 212, 672, 2145, 2764, 684, 2197, 2716, 678, 2218, 2772, 672, 2156, 2796, 672, 2145, 2764, 
			688, 2136, 2751, 741, 2095, 2791, 777, 1977, 2803, 794, 1884, 2784, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w17ah"}, {267, 205, 937, 1623, 2844, 913, 1610, 2897, 939, 1617, 2844, 939, 1607, 2859, 942, 1614, 2870, 
			941, 1608, 2870, 934, 1612, 2869, 912, 1650, 2875, 838, 1658, 2860, 95, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0}},
		{{"w17aw"}, {341, 211, 794, 1185, 2896, 772, 1035, 2855, 777, 1001, 2844, 776, 1039, 2810, 796, 1115, 2817, 
			794, 1181, 2884, 796, 1205, 2934, 814, 1229, 2896, 830, 1420, 2864, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"w17eh"}, {188, 216, 750, 1935, 2930, 696, 1985, 2892, 730, 1961, 2877, 748, 1942, 2885, 752, 1940, 2887, 
			746, 1934, 2943, 746, 1906, 2950, 738, 1896, 2955, 682, 1884, 2959, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w17ei"}, {261, 210, 558, 2505, 2855, 553, 2402, 2915, 558, 2505, 2855, 507, 2514, 2935, 472, 2522, 2947, 
			448, 2529, 2924, 434, 2606, 2942, 429, 2630, 2981, 428, 2638, 2966, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w17er"}, {308, 212, 491, 1506, 1727, 482, 1464, 1795, 483, 1516, 1739, 491, 1510, 1727, 488, 1506, 1722, 
			461, 1504, 1723, 464, 1501, 1750, 461, 1514, 1728, 444, 1515, 1802, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w17ih"}, {203, 221, 474, 2362, 2830, 471, 2350, 2936, 474, 2362, 2830, 496, 2289, 2859, 514, 2222, 2865, 
			565, 2170, 2869, 580, 2121, 2869, 590, 2005, 2916, 587, 1973, 2938, 95, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w17iy"}, {240, 210, 429, 2698, 3509, 433, 2625, 3462, 432, 2678, 3508, 430, 2695, 3510, 428, 2681, 3503, 
			429, 2708, 3450, 430, 2710, 3426, 430, 2702, 3411, 432, 2661, 3319, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w17oa"}, {298, 206, 616, 1034, 2785, 626, 1105, 2758, 616, 1034, 2785, 593, 1018, 2780, 569, 997, 2803, 
			549, 980, 2825, 493, 884, 2819, 450, 863, 2746, 435, 875, 2701, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w17oo"}, {202, 232, 492, 1161, 2766, 495, 1161, 2766, 489, 1166, 2765, 489, 1183, 2760, 492, 1209, 2763, 
			498, 1298, 2777, 505, 1332, 2768, 506, 1392, 2759, 502, 1614, 2784, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 0, 1, 0}},
		{{"w17uh"}, {183, 212, 705, 1166, 2890, 705, 1166, 2890, 742, 1194, 2895, 748, 1266, 2902, 742, 1332, 2892, 
			739, 1403, 2883, 734, 1503, 2867, 710, 1618, 2866, 672, 1679, 2866, 95, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 19, 0}},
		{{"w17uw"}, {266, 216, 436, 911, 2724, 488, 908, 2677, 447, 974, 2723, 439, 949, 2686, 437, 929, 2676, 
			436, 916, 2744, 434, 877, 2726, 434, 890, 2677, 434, 975, 2720, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w19ae"}, {363, 237, 698, 2339, 3075, 684, 2414, 3188, 691, 2357, 3073, 699, 2331, 3094, 747, 2205, 3067, 
			831, 2101, 3008, 886, 1923, 3002, 902, 1922, 3037, 889, 1950, 3021, 95, 0, 0, 1, 0, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"w19ah"}, {316, 239, 918, 1640, 2801, 916, 1510, 2788, 939, 1584, 2803, 945, 1621, 2793, 937, 1639, 2851, 
			918, 1640, 2801, 910, 1664, 2823, 914, 1803, 2836, 890, 1899, 2867, 95, 0, 0, 0, 0, 0, 19, 0, 0, 0, 0, 1, 0}},
		{{"w19aw"}, {327, 241, 848, 1175, 2892, 828, 1171, 2909, 848, 1182, 2883, 838, 1185, 2896, 831, 1204, 2942, 
			846, 1182, 2980, 868, 1284, 2991, 879, 1417, 3060, 881, 1646, 2939, 65, 0, 0, 0, 0, 0, 7, 13, 0, 0, 0, 0, 0}},
		{{"w19eh"}, {317, 238, 706, 1986, 3228, 696, 2008, 3190, 705, 2100, 3191, 704, 2078, 3226, 708, 2024, 3227, 
			707, 1990, 3264, 704, 1984, 3221, 701, 1991, 3200, 696, 2043, 3119, 90, 0, 0, 0, 18, 2, 0, 0, 0, 0, 0, 0, 0}},
		{{"w19ei"}, {334, 244, 572, 2428, 3120, 643, 2429, 3140, 602, 2425, 3144, 572, 2428, 3120, 546, 2430, 3121, 
			508, 2436, 3126, 499, 2440, 3146, 495, 2447, 3107, 492, 2477, 3093, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w19er"}, {319, 246, 524, 1547, 2066, 557, 1561, 2085, 584, 1624, 2068, 549, 1572, 2045, 530, 1551, 2066, 
			523, 1518, 2103, 510, 1526, 2121, 504, 1567, 2129, 500, 1626, 2134, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w19ih"}, {256, 252, 501, 2355, 3161, 501, 2366, 3164, 501, 2329, 3158, 498, 2369, 3163, 496, 2334, 3103, 
			497, 2261, 3086, 499, 2218, 3091, 504, 2189, 3091, 508, 2185, 3093, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w19iy"}, {302, 247, 492, 2695, 3371, 469, 2659, 3342, 496, 2595, 3301, 497, 2625, 3290, 497, 2686, 3339, 
			496, 2698, 3355, 492, 2697, 3368, 489, 2686, 3380, 486, 2680, 3346, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w19oa"}, {340, 241, 631, 1185, 2942, 705, 1218, 2996, 685, 1205, 2988, 657, 1196, 2947, 622, 1177, 2926, 
			586, 1164, 2918, 551, 1131, 2922, 511, 1098, 2918, 496, 1117, 2870, 95, 0, 0, 0, 0, 0, 1, 0, 19, 0, 0, 0, 0}},
		{{"w19oo"}, {332, 249, 505, 1404, 2774, 539, 1393, 2741, 505, 1404, 2774, 503, 1418, 2770, 515, 1441, 2766, 
			555, 1449, 2832, 591, 1502, 2851, 615, 1630, 2860, 609, 1693, 2850, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"w19uh"}, {286, 235, 839, 1555, 3003, 798, 1290, 2944, 818, 1376, 3009, 845, 1432, 3059, 853, 1526, 2990, 
			839, 1555, 3003, 810, 1608, 3005, 744, 1707, 2998, 699, 1875, 3019, 70, 0, 0, 0, 0, 0, 2, 4, 0, 0, 0, 14, 0}},
		{{"w19uw"}, {293, 257, 506, 1030, 2575, 535, 1024, 2618, 518, 1034, 2586, 510, 1044, 2603, 509, 1042, 2584, 
			506, 1030, 2575, 504, 1012, 2591, 502, 1011, 2601, 501, 1011, 2587, 95, 0, 0, 0, 0, 0, 0, 0, 1, 0, 19, 0, 0}},
		{{"w20ae"}, {307, 187, 586, 2299, 2748, 586, 2306, 2744, 593, 2299, 2740, 582, 2294, 2754, 624, 2272, 2722, 
			724, 2132, 2697, 835, 1988, 2626, 940, 1826, 2622, 930, 1817, 2610, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w20ah"}, {365, 193, 818, 1351, 2933, 824, 1378, 2928, 842, 1320, 2935, 828, 1352, 2943, 859, 1345, 2939, 
			922, 1384, 2971, 909, 1419, 3007, 900, 1444, 3006, 814, 1585, 2872, 65, 0, 0, 0, 0, 0, 13, 7, 0, 0, 0, 0, 0}},
		{{"w20aw"}, {344, 185, 713, 1132, 2948, 698, 1031, 2933, 690, 1038, 3270, 740, 1110, 3053, 737, 1160, 2908, 
			713, 1132, 2948, 699, 1116, 2942, 743, 1363, 2940, 728, 1506, 2895, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"w20eh"}, {263, 173, 717, 2256, 2896, 745, 2104, 2966, 734, 2097, 2897, 733, 2211, 2889, 720, 2228, 2845, 
			686, 2216, 2859, 681, 2205, 2856, 653, 2232, 2924, 653, 2255, 2959, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w20ei"}, {305, 203, 574, 2360, 2686, 551, 2358, 2704, 578, 2321, 2683, 571, 2386, 2690, 561, 2372, 2697, 
			553, 2379, 2667, 534, 2418, 2719, 505, 2447, 2766, 488, 2500, 2826, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w20er"}, {325, 175, 508, 1562, 0, 563, 1438, 0, 556, 1609, 0, 552, 1532, 0, 545, 1499, 0, 
			523, 1508, 0, 509, 1556, 0, 507, 1567, 0, 506, 1591, 0, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w20ih"}, {240, 213, 449, 2173, 2840, 450, 2205, 2887, 452, 2171, 2836, 456, 2164, 2878, 491, 2225, 2886, 
			519, 2278, 2891, 579, 2228, 2906, 572, 2137, 2915, 553, 2138, 2944, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w20iy"}, {287, 195, 382, 2563, 3477, 429, 2585, 3310, 421, 2617, 3350, 397, 2610, 3417, 392, 2578, 3449, 
			385, 2571, 3463, 380, 2562, 3477, 381, 2562, 3493, 402, 2616, 3616, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w20oa"}, {317, 174, 504, 1047, 2749, 626, 1069, 2775, 621, 1063, 2735, 585, 1056, 2748, 563, 1050, 2758, 
			540, 1058, 2761, 508, 1049, 2778, 504, 1047, 2749, 503, 1046, 2684, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w20oo"}, {301, 233, 500, 1095, 2804, 500, 1090, 2804, 503, 1037, 2797, 560, 1109, 2807, 575, 1309, 2829, 
			554, 1358, 2746, 545, 1477, 2674, 546, 1595, 2626, 512, 1731, 2658, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w20uh"}, {249, 216, 642, 1338, 2938, 570, 1354, 2933, 679, 1373, 2953, 793, 1427, 2949, 728, 1446, 2946, 
			761, 1550, 2965, 757, 1728, 2962, 581, 1924, 2926, 540, 2152, 2980, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w20uw"}, {283, 171, 360, 1034, 2731, 411, 959, 2820, 376, 1011, 2798, 375, 1047, 2815, 385, 1049, 2834, 
			373, 1046, 2777, 386, 1009, 2764, 357, 1038, 2707, 365, 1046, 2753, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19, 0, 0}},
		{{"w21ae"}, {382, 217, 616, 2156, 2865, 620, 2143, 2901, 616, 2151, 2865, 621, 2134, 2927, 629, 2117, 2837, 
			676, 1997, 2821, 726, 1918, 2812, 748, 1875, 2836, 746, 1874, 2867, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"w21ah"}, {402, 216, 769, 1451, 2898, 753, 1449, 2855, 763, 1379, 2861, 757, 1424, 2906, 764, 1439, 2907, 
			771, 1472, 2863, 764, 1501, 2830, 762, 1531, 2791, 761, 1635, 2801, 90, 0, 0, 0, 0, 0, 18, 1, 0, 0, 0, 1, 0}},
		{{"w21aw"}, {465, 226, 698, 1127, 2886, 712, 1124, 2876, 698, 1133, 2896, 697, 1126, 2876, 701, 1124, 2874, 
			696, 1159, 2869, 700, 1216, 2860, 713, 1276, 2876, 707, 1446, 2888, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"w21eh"}, {281, 233, 727, 1871, 2902, 733, 1886, 2895, 723, 1882, 2900, 738, 1856, 2899, 741, 1843, 2919, 
			734, 1809, 2922, 725, 1820, 2934, 700, 1870, 2907, 692, 1923, 2886, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w21ei"}, {346, 238, 498, 2218, 2894, 498, 2218, 2894, 490, 2272, 2937, 457, 2266, 2947, 440, 2287, 2925, 
			441, 2303, 2936, 441, 2329, 2932, 442, 2342, 2913, 445, 2347, 2873, 95, 1, 0, 19, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w21er"}, {370, 224, 469, 1499, 0, 483, 1391, 1598, 464, 1450, 0, 447, 1379, 0, 446, 1412, 0, 
			450, 1450, 0, 457, 1495, 0, 469, 1503, 0, 468, 1543, 0, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w21ih"}, {345, 233, 458, 2143, 2986, 481, 2205, 2989, 445, 2216, 2954, 457, 2156, 2986, 469, 2112, 2972, 
			486, 2115, 2949, 496, 2120, 2940, 501, 2008, 2935, 499, 1983, 2936, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w21iy"}, {353, 223, 434, 2512, 2998, 429, 2484, 2972, 432, 2505, 3022, 434, 2470, 3044, 434, 2497, 3055, 
			434, 2511, 2993, 435, 2501, 2981, 436, 2497, 2904, 437, 2493, 2869, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w21oa"}, {374, 234, 469, 934, 2571, 621, 1136, 2757, 585, 1122, 2746, 527, 1097, 2729, 493, 1007, 2679, 
			477, 970, 2617, 461, 932, 2545, 465, 934, 2569, 471, 936, 2563, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w21oo"}, {370, 242, 496, 1169, 2654, 498, 1131, 2645, 496, 1171, 2651, 492, 1165, 2639, 496, 1177, 2659, 
			500, 1277, 2635, 510, 1412, 2619, 518, 1549, 2653, 507, 1681, 2679, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w21uh"}, {330, 221, 701, 1318, 2881, 705, 1299, 2869, 700, 1334, 2878, 701, 1383, 2859, 690, 1477, 2867, 
			673, 1525, 2869, 661, 1596, 2881, 661, 1656, 2878, 645, 1803, 2862, 75, 0, 0, 0, 0, 0, 2, 3, 0, 0, 0, 15, 0}},
		{{"w21uw"}, {378, 247, 491, 1260, 2502, 482, 1240, 2480, 489, 1263, 2505, 493, 1257, 2472, 497, 1269, 2499, 
			497, 1175, 2501, 496, 993, 2522, 500, 1005, 2546, 506, 1064, 2549, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w22ae"}, {338, 186, 576, 2429, 3101, 579, 2414, 3100, 573, 2431, 3104, 579, 2424, 3056, 625, 2373, 3024, 
			685, 2308, 2970, 755, 2124, 2905, 844, 1918, 2907, 866, 1811, 2938, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w22ah"}, {350, 178, 1008, 1495, 3018, 949, 1486, 3037, 998, 1482, 3065, 1010, 1488, 3042, 1010, 1489, 3016, 
			1005, 1498, 3016, 994, 1490, 2994, 973, 1508, 2977, 912, 1573, 2952, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"w22aw"}, {318, 191, 736, 1120, 3055, 700, 973, 3056, 714, 1025, 3066, 721, 1086, 3063, 741, 1124, 3053, 
			762, 1159, 3051, 804, 1192, 3051, 829, 1281, 3020, 834, 1471, 3004, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"w22eh"}, {239, 180, 584, 2175, 2930, 580, 2194, 2953, 617, 2144, 2935, 634, 2076, 2901, 669, 1989, 2911, 
			689, 1931, 2911, 695, 1865, 2914, 697, 1832, 2952, 692, 1847, 2988, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w22ei"}, {269, 166, 511, 2574, 3085, 511, 2574, 3085, 500, 2573, 3107, 499, 2575, 3063, 498, 2590, 3057, 
			495, 2618, 3080, 488, 2636, 3029, 479, 2655, 3042, 453, 2673, 3051, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w22er"}, {279, 173, 505, 1568, 1845, 504, 1536, 1862, 505, 1568, 1845, 505, 1570, 1835, 506, 1580, 1830, 
			508, 1596, 1851, 510, 1632, 1871, 511, 1658, 1893, 516, 1755, 1950, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w22ih"}, {171, 187, 440, 2394, 3136, 440, 2394, 3136, 449, 2394, 3054, 468, 2398, 3087, 487, 2393, 3100, 
			501, 2348, 3097, 525, 2278, 3082, 552, 2214, 3063, 555, 2172, 3059, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w22iy"}, {304, 214, 435, 2621, 3116, 437, 2590, 3181, 436, 2604, 3163, 436, 2621, 3118, 433, 2622, 3084, 
			433, 2628, 3073, 431, 2616, 3035, 432, 2605, 3039, 435, 2577, 3043, 95, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w22oa"}, {334, 181, 578, 1061, 2882, 570, 1059, 2878, 569, 1056, 2875, 562, 1057, 2888, 553, 1042, 2947, 
			526, 995, 2962, 507, 933, 2949, 506, 886, 2943, 487, 871, 2992, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w22oo"}, {270, 203, 458, 1042, 3007, 460, 1028, 3001, 476, 1085, 3017, 493, 1114, 3005, 519, 1179, 3002, 
			572, 1215, 2984, 610, 1366, 2937, 620, 1428, 2914, 623, 1590, 2881, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w22uh"}, {209, 188, 777, 1397, 3033, 780, 1274, 3037, 783, 1294, 3037, 773, 1353, 3037, 786, 1414, 3030, 
			808, 1441, 3037, 817, 1497, 3058, 807, 1598, 3045, 724, 1780, 3047, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w22uw"}, {310, 199, 391, 971, 2945, 432, 1028, 2907, 427, 1025, 2920, 419, 1003, 2902, 400, 977, 2914, 
			391, 959, 2953, 392, 977, 2948, 411, 1008, 2969, 426, 1053, 2954, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w23ae"}, {365, 213, 674, 2256, 2818, 666, 2278, 2835, 669, 2297, 2795, 676, 2252, 2794, 741, 2198, 2745, 
			772, 2131, 2712, 807, 2056, 2736, 804, 2011, 2794, 791, 2008, 2877, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w23ah"}, {329, 207, 905, 1514, 2848, 894, 1511, 2783, 930, 1494, 2849, 903, 1525, 2862, 900, 1504, 2838, 
			851, 1536, 2817, 806, 1636, 2844, 792, 1841, 2884, 735, 1896, 2885, 90, 0, 0, 0, 0, 0, 18, 1, 0, 0, 0, 1, 0}},
		{{"w23aw"}, {362, 209, 835, 1326, 2667, 827, 1255, 2589, 834, 1319, 2623, 835, 1326, 2667, 831, 1314, 2612, 
			819, 1301, 2609, 815, 1352, 2643, 807, 1506, 2661, 765, 1662, 2698, 50, 0, 0, 0, 0, 0, 10, 10, 0, 0, 0, 0, 0}},
		{{"w23eh"}, {294, 210, 762, 1902, 2887, 751, 1888, 2807, 763, 1888, 2810, 769, 1874, 2806, 757, 1902, 2890, 
			744, 1941, 2897, 683, 2007, 2897, 630, 2054, 2896, 620, 2022, 2903, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w23ei"}, {342, 211, 513, 2435, 2939, 512, 2435, 2935, 496, 2472, 2971, 478, 2463, 2881, 452, 2525, 2910, 
			444, 2561, 2893, 432, 2575, 2866, 414, 2578, 2889, 394, 2586, 2825, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w23er"}, {345, 212, 508, 1687, 1913, 541, 1698, 1858, 526, 1696, 1883, 511, 1692, 1903, 501, 1685, 1939, 
			460, 1695, 1961, 445, 1702, 1960, 441, 1730, 1968, 440, 1754, 1973, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w23ih"}, {227, 218, 535, 2134, 2929, 490, 2192, 2870, 524, 2156, 2880, 546, 2146, 2913, 551, 2137, 2924, 
			547, 2134, 2925, 526, 2135, 2937, 493, 2160, 2908, 473, 2098, 2893, 95, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w23iy"}, {307, 219, 433, 2634, 3307, 433, 2599, 3266, 433, 2621, 3294, 433, 2634, 3307, 433, 2632, 3281, 
			435, 2631, 3295, 437, 2664, 3232, 440, 2710, 3131, 444, 2673, 3029, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w23oa"}, {345, 201, 560, 987, 2311, 687, 1140, 2533, 633, 1136, 2447, 613, 1104, 2405, 577, 1008, 2364, 
			566, 987, 2313, 558, 989, 2304, 526, 996, 2243, 490, 1116, 2242, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w23oo"}, {208, 231, 617, 1373, 2579, 617, 1347, 2571, 612, 1305, 2589, 612, 1326, 2607, 617, 1436, 2616, 
			618, 1533, 2619, 610, 1558, 2623, 575, 1673, 2648, 543, 1779, 2701, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 0, 1, 0}},
		{{"w23uh"}, {220, 211, 686, 1477, 2698, 634, 1473, 2615, 679, 1488, 2666, 686, 1477, 2698, 689, 1475, 2705, 
			683, 1498, 2675, 650, 1615, 2631, 633, 1742, 2667, 623, 1846, 2736, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w23uw"}, {363, 210, 431, 1003, 2366, 438, 1068, 2315, 436, 1016, 2375, 434, 1015, 2366, 430, 999, 2364, 
			427, 996, 2387, 427, 1001, 2385, 426, 1014, 2342, 431, 1056, 2340, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w24ae"}, {331, 248, 738, 2378, 3305, 743, 2457, 3253, 737, 2417, 3316, 740, 2378, 3306, 743, 2331, 3266, 
			754, 2233, 3270, 780, 2189, 3258, 817, 2173, 3247, 816, 2140, 3263, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w24ah"}, {303, 259, 1063, 1680, 2785, 1035, 1671, 2844, 1042, 1667, 2791, 1063, 1680, 2785, 1060, 1707, 2864, 
			1042, 1740, 2965, 1012, 1751, 3021, 1000, 1762, 3030, 993, 1808, 3152, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"w24aw"}, {366, 247, 907, 1527, 3075, 1044, 1597, 2690, 1008, 1566, 2797, 980, 1547, 2880, 958, 1540, 2991, 
			907, 1527, 3075, 878, 1538, 3133, 933, 1681, 3177, 944, 1755, 3101, 40, 0, 0, 0, 0, 0, 12, 8, 0, 0, 0, 0, 0}},
		{{"w24eh"}, {231, 251, 981, 2075, 3215, 1032, 2058, 3300, 992, 2028, 3205, 981, 2075, 3215, 957, 2108, 3203, 
			938, 2130, 3219, 905, 2136, 3242, 823, 2177, 3301, 753, 2216, 3406, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w24ei"}, {349, 263, 617, 2668, 3319, 607, 2686, 3288, 614, 2652, 3331, 592, 2663, 3299, 572, 2693, 3303, 
			556, 2769, 3240, 536, 2798, 3278, 510, 2822, 3251, 506, 2837, 3273, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w24er"}, {327, 262, 538, 1813, 2236, 557, 1661, 2166, 554, 1666, 2131, 549, 1694, 2075, 549, 1749, 2152, 
			546, 1799, 2194, 544, 1812, 2225, 535, 1813, 2222, 537, 1818, 2220, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w24ih"}, {258, 256, 516, 2567, 3485, 553, 2619, 3390, 529, 2608, 3420, 518, 2584, 3461, 516, 2567, 3497, 
			521, 2528, 3479, 543, 2475, 3440, 551, 2438, 3447, 551, 2401, 3495, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w24iy"}, {334, 272, 531, 2996, 3831, 524, 3010, 3890, 520, 3027, 3816, 513, 3101, 3775, 517, 3046, 3756, 
			524, 3007, 3806, 532, 2997, 3796, 530, 2995, 3693, 533, 3000, 3600, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w24oa"}, {362, 242, 508, 991, 3121, 643, 1175, 3114, 629, 1147, 3220, 596, 1101, 3231, 564, 1047, 3184, 
			543, 1008, 3128, 509, 993, 3117, 501, 982, 3110, 497, 983, 3111, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w24oo"}, {245, 262, 559, 1246, 3127, 559, 1240, 3180, 561, 1246, 3083, 563, 1230, 3167, 596, 1283, 3173, 
			620, 1463, 3185, 630, 1495, 3175, 657, 1623, 3104, 644, 1749, 3057, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w24uh"}, {250, 239, 769, 1382, 3035, 800, 1442, 2983, 793, 1394, 3004, 768, 1394, 3046, 763, 1435, 3078, 
			760, 1516, 3117, 759, 1601, 3123, 749, 1737, 3119, 705, 1921, 3154, 90, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 18, 0}},
		{{"w24uw"}, {327, 268, 525, 1065, 2942, 551, 1117, 2971, 546, 1102, 2980, 538, 1085, 2964, 537, 1083, 2958, 
			538, 1082, 2950, 526, 1067, 2946, 525, 1066, 2932, 530, 1084, 2925, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19, 0, 0}},
		{{"w25ae"}, {461, 220, 646, 2406, 3283, 641, 2459, 3395, 652, 2322, 3268, 678, 2313, 3228, 761, 2203, 3181, 
			802, 2084, 3154, 811, 1976, 3084, 809, 1864, 2972, 776, 1935, 3083, 95, 0, 0, 0, 0, 19, 0, 0, 1, 0, 0, 0, 0}},
		{{"w25ah"}, {430, 212, 1053, 1677, 2868, 1100, 1724, 2866, 1066, 1679, 2866, 1053, 1671, 2868, 1032, 1660, 2866, 
			994, 1649, 2841, 957, 1643, 2825, 926, 1613, 2805, 898, 1797, 2744, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"w25aw"}, {414, 205, 811, 1254, 2824, 799, 1148, 2869, 777, 1167, 2769, 780, 1180, 2799, 809, 1248, 2804, 
			811, 1254, 2824, 836, 1281, 2831, 862, 1371, 2786, 805, 1619, 2846, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"w25eh"}, {386, 211, 794, 2046, 3082, 870, 1962, 3124, 813, 2005, 3112, 782, 2026, 3099, 794, 2025, 3087, 
			801, 2047, 3085, 801, 2077, 3099, 767, 2088, 3140, 711, 2155, 3206, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w25ei"}, {399, 212, 610, 2598, 3323, 657, 2487, 3267, 643, 2571, 3272, 626, 2531, 3290, 610, 2598, 3323, 
			573, 2613, 3182, 524, 2635, 3170, 481, 2697, 3197, 445, 2675, 3167, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w25er"}, {403, 215, 617, 1565, 1930, 517, 1557, 2144, 556, 1545, 2049, 611, 1563, 1995, 619, 1567, 1968, 
			617, 1564, 1927, 614, 1601, 1888, 611, 1656, 1913, 604, 1681, 1981, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w25ih"}, {302, 222, 552, 2453, 3346, 547, 2414, 3345, 552, 2476, 3323, 566, 2448, 3350, 594, 2353, 3256, 
			611, 2334, 3228, 614, 2323, 3289, 613, 2311, 3323, 605, 2280, 3312, 65, 0, 13, 0, 6, 1, 0, 0, 0, 0, 0, 0, 0}},
		{{"w25iy"}, {346, 237, 467, 3009, 0, 495, 2935, 0, 488, 2860, 0, 487, 2901, 0, 474, 2951, 0, 
			463, 3066, 0, 464, 3011, 0, 462, 2984, 0, 482, 2886, 0, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w25oa"}, {443, 213, 658, 1045, 2928, 657, 993, 2836, 661, 1013, 2942, 639, 1048, 2934, 623, 975, 2956, 
			622, 983, 2933, 619, 991, 2863, 611, 988, 2909, 563, 1033, 2877, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w25oo"}, {331, 228, 509, 1244, 2980, 509, 1244, 2980, 521, 1224, 2956, 582, 1324, 2930, 630, 1340, 2884, 
			649, 1500, 3194, 635, 1678, 3174, 626, 1851, 3192, 618, 1984, 3230, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w25uh"}, {309, 217, 815, 1427, 3002, 884, 1360, 3005, 807, 1451, 3003, 775, 1495, 2999, 788, 1498, 2993, 
			789, 1507, 2964, 760, 1659, 2993, 744, 1765, 3013, 677, 1998, 3229, 60, 0, 0, 0, 0, 0, 3, 5, 0, 0, 0, 12, 0}},
		{{"w25uw"}, {382, 226, 448, 1369, 2921, 489, 1607, 2832, 475, 1484, 2872, 465, 1433, 2908, 444, 1348, 2901, 
			442, 1320, 2881, 440, 1302, 2930, 440, 1267, 2921, 443, 1231, 2934, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w26ae"}, {346, 168, 746, 1944, 2927, 745, 1994, 2918, 744, 2008, 2912, 748, 1983, 2933, 747, 1944, 2927, 
			749, 1956, 2940, 741, 1925, 2925, 708, 1890, 2930, 721, 1891, 2920, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w26ah"}, {333, 170, 798, 1351, 2851, 794, 1374, 2833, 805, 1363, 2803, 828, 1284, 2837, 803, 1350, 2852, 
			782, 1370, 2875, 759, 1391, 2915, 726, 1434, 2918, 707, 1491, 2900, 85, 0, 0, 0, 0, 0, 17, 2, 0, 0, 0, 1, 0}},
		{{"w26aw"}, {378, 180, 656, 1075, 2684, 682, 1063, 2766, 644, 1024, 2697, 665, 1017, 2700, 663, 1068, 2699, 
			671, 1073, 2707, 657, 1128, 2727, 654, 1154, 2785, 669, 1269, 2795, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"w26eh"}, {183, 195, 620, 2071, 2933, 671, 2139, 2954, 655, 2124, 2966, 630, 2100, 2955, 620, 2071, 2933, 
			590, 2042, 2920, 571, 2028, 2910, 575, 2001, 2911, 578, 1999, 2906, 95, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w26ei"}, {420, 171, 485, 2423, 2912, 542, 2305, 2988, 510, 2301, 2884, 471, 2323, 2901, 466, 2373, 2884, 
			485, 2423, 2912, 451, 2393, 2897, 427, 2436, 2922, 390, 2440, 2909, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w26er"}, {340, 184, 522, 1490, 2094, 566, 1346, 2114, 556, 1396, 2075, 529, 1425, 2077, 532, 1489, 2102, 
			514, 1474, 2078, 510, 1491, 2060, 502, 1556, 2098, 497, 1622, 2106, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w26ih"}, {209, 223, 455, 2304, 2999, 460, 2306, 2999, 449, 2305, 3000, 436, 2286, 2993, 434, 2260, 2960, 
			446, 2252, 2914, 486, 2201, 2891, 497, 2202, 2865, 495, 2183, 2870, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w26iy"}, {336, 177, 375, 2577, 3276, 439, 2565, 3279, 437, 2534, 3219, 432, 2575, 3254, 409, 2592, 3272, 
			379, 2589, 3277, 370, 2596, 3284, 357, 2566, 3300, 332, 2589, 3295, 95, 19, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w26oa"}, {378, 167, 498, 970, 2876, 569, 939, 2965, 561, 1002, 2928, 532, 960, 2900, 517, 876, 2904, 
			500, 948, 2894, 487, 968, 2767, 447, 915, 2765, 430, 920, 2686, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w26oo"}, {171, 195, 490, 1122, 2824, 498, 1117, 2822, 493, 1091, 2791, 490, 1036, 2813, 492, 1152, 2824, 
			503, 1140, 2821, 516, 1186, 2763, 521, 1329, 2741, 506, 1462, 2681, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w26uh"}, {187, 195, 678, 1397, 2861, 725, 1337, 2849, 691, 1300, 2887, 681, 1388, 2875, 678, 1401, 2861, 
			691, 1468, 2838, 690, 1494, 2831, 688, 1551, 2808, 658, 1712, 2849, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w26uw"}, {333, 230, 445, 881, 2580, 448, 881, 2598, 442, 890, 2642, 436, 927, 2631, 395, 973, 2628, 
			375, 938, 2593, 364, 918, 2523, 319, 927, 2536, 317, 955, 2496, 95, 0, 0, 0, 0, 0, 0, 0, 1, 0, 19, 0, 0}},
		{{"w27ae"}, {272, 246, 734, 2518, 3176, 756, 2450, 3184, 728, 2530, 3177, 730, 2534, 3168, 728, 2518, 3123, 
			727, 2504, 3095, 730, 2141, 3063, 832, 2075, 3061, 835, 2028, 3029, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w27ah"}, {284, 250, 869, 1751, 2762, 830, 1757, 2761, 891, 1749, 2780, 928, 1757, 2833, 932, 1795, 2858, 
			930, 1864, 2817, 932, 1880, 2816, 918, 1892, 2817, 916, 1941, 2833, 95, 0, 0, 0, 1, 0, 19, 0, 0, 0, 0, 0, 0}},
		{{"w27aw"}, {284, 244, 890, 1292, 2801, 779, 1212, 2818, 887, 1252, 2773, 891, 1305, 2828, 895, 1345, 2840, 
			890, 1398, 2875, 886, 1423, 2861, 887, 1559, 2846, 855, 1737, 2790, 55, 0, 0, 0, 0, 0, 7, 11, 0, 0, 0, 2, 0}},
		{{"w27eh"}, {236, 244, 741, 1997, 2567, 762, 2131, 2574, 759, 2042, 2549, 753, 2024, 2567, 749, 2008, 2551, 
			741, 1997, 2562, 731, 1997, 2565, 729, 2003, 2580, 734, 2033, 2535, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w27ei"}, {284, 243, 578, 2474, 3055, 559, 2485, 3059, 539, 2530, 3030, 504, 2572, 3026, 498, 2612, 3078, 
			496, 2606, 2980, 492, 2611, 3059, 487, 2614, 3032, 490, 2661, 3106, 95, 0, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0}},
		{{"w27er"}, {242, 257, 507, 1749, 2028, 529, 1745, 2074, 514, 1752, 2049, 507, 1749, 2028, 506, 1752, 2009, 
			508, 1768, 1991, 510, 1775, 2015, 516, 1805, 2066, 536, 1846, 2116, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w27ih"}, {197, 251, 515, 2378, 3063, 515, 2378, 3063, 500, 2466, 3132, 498, 2427, 3124, 499, 2286, 3222, 
			500, 2254, 3219, 504, 2200, 3217, 508, 2190, 3222, 510, 2193, 3187, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w27iy"}, {250, 252, 497, 2795, 3266, 497, 2768, 3278, 497, 2819, 3245, 497, 2855, 3268, 497, 2795, 3266, 
			498, 2769, 3253, 499, 2765, 3252, 503, 2801, 3241, 508, 2824, 3202, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w27oa"}, {265, 243, 582, 946, 2704, 650, 965, 2728, 582, 946, 2704, 567, 932, 2707, 557, 931, 2748, 
			526, 948, 2818, 521, 954, 2829, 509, 985, 2832, 501, 1004, 2841, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w27oo"}, {167, 262, 616, 1331, 2696, 616, 1331, 2696, 607, 1342, 2728, 606, 1364, 2752, 613, 1387, 2776, 
			625, 1452, 2789, 656, 1594, 2758, 650, 1696, 2730, 615, 1895, 2842, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w27uh"}, {195, 244, 823, 1419, 3051, 807, 1481, 3004, 822, 1451, 3001, 823, 1419, 3051, 822, 1444, 3078, 
			822, 1525, 3103, 813, 1673, 3081, 779, 1742, 3017, 753, 1835, 2958, 95, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 19, 0}},
		{{"w27uw"}, {234, 256, 503, 963, 2776, 507, 1033, 2775, 502, 963, 2776, 501, 942, 2758, 502, 956, 2723, 
			503, 983, 2725, 503, 1119, 2745, 503, 1229, 2746, 504, 1349, 2745, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w28ae"}, {260, 225, 662, 2276, 2955, 671, 2278, 2991, 662, 2270, 2938, 676, 2251, 2936, 689, 2211, 2945, 
			746, 2068, 3004, 800, 2037, 3006, 787, 1930, 3083, 749, 1890, 3093, 90, 0, 0, 1, 0, 18, 1, 0, 0, 0, 0, 0, 0}},
		{{"w28ah"}, {217, 235, 863, 1538, 3038, 854, 1509, 3009, 863, 1538, 3038, 842, 1573, 3058, 842, 1589, 3075, 
			866, 1625, 3078, 869, 1733, 3095, 863, 1830, 3104, 771, 1920, 3118, 75, 0, 0, 0, 0, 0, 15, 1, 0, 0, 0, 4, 0}},
		{{"w28aw"}, {262, 217, 860, 1504, 2834, 845, 1429, 2751, 871, 1474, 2780, 862, 1502, 2834, 861, 1517, 2853, 
			867, 1539, 2871, 875, 1589, 2871, 864, 1722, 2918, 789, 1866, 3031, 50, 0, 0, 0, 1, 0, 6, 10, 0, 0, 0, 3, 0}},
		{{"w28eh"}, {177, 223, 693, 2012, 3003, 688, 2032, 2971, 693, 2012, 3003, 693, 2009, 3017, 689, 2022, 3030, 
			687, 2037, 3054, 686, 2055, 3129, 685, 2060, 3181, 682, 2073, 3246, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w28ei"}, {231, 234, 507, 2573, 3121, 506, 2485, 3126, 503, 2582, 3201, 491, 2617, 3179, 480, 2621, 3073, 
			491, 2773, 3100, 499, 2772, 0, 500, 2916, 0, 501, 2813, 3063, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w28er"}, {223, 238, 514, 1831, 2192, 557, 1738, 2206, 552, 1739, 2153, 535, 1796, 2165, 517, 1824, 2192, 
			509, 1841, 2200, 502, 1855, 2242, 498, 1896, 2294, 498, 1940, 2389, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w28ih"}, {167, 238, 504, 2334, 3063, 504, 2334, 3063, 510, 2300, 3054, 538, 2298, 3071, 560, 2218, 3107, 
			562, 2182, 3116, 553, 2185, 3152, 527, 2198, 3240, 509, 2216, 3320, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w28iy"}, {222, 246, 495, 2910, 0, 496, 2783, 0, 493, 2859, 0, 493, 2883, 0, 493, 2902, 0, 
			495, 2910, 0, 498, 2934, 0, 501, 2945, 0, 507, 2828, 3335, 95, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w28oa"}, {225, 240, 613, 1184, 3063, 613, 1184, 3063, 591, 1183, 3068, 549, 1178, 3076, 506, 1167, 3040, 
			502, 1135, 2998, 500, 1137, 2992, 500, 1170, 2965, 500, 1217, 2969, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w28oo"}, {181, 241, 521, 1428, 3156, 514, 1313, 3068, 517, 1375, 3119, 520, 1412, 3132, 521, 1438, 3168, 
			520, 1462, 3105, 521, 1493, 3097, 519, 1630, 3085, 508, 1752, 3013, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w28uh"}, {194, 231, 700, 1580, 3018, 700, 1576, 3013, 697, 1584, 3048, 698, 1610, 3081, 694, 1638, 3101, 
			691, 1711, 3122, 689, 1778, 3055, 685, 1854, 3044, 679, 1882, 3090, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w28uw"}, {182, 261, 508, 1001, 2763, 508, 1309, 2825, 503, 1288, 2801, 507, 1090, 2771, 510, 1036, 2752, 
			510, 1035, 2755, 510, 1036, 2772, 513, 979, 2807, 518, 1082, 2847, 95, 0, 0, 1, 0, 0, 0, 0, 0, 0, 19, 0, 0}},
		{{"w29ae"}, {333, 238, 696, 2447, 3122, 710, 2465, 3207, 700, 2456, 3136, 695, 2413, 3154, 692, 2445, 3146, 
			747, 2344, 3013, 941, 2183, 2833, 1009, 2036, 2778, 982, 2027, 2830, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w29ah"}, {303, 248, 938, 1462, 2825, 936, 1451, 2779, 914, 1395, 2836, 1047, 1405, 2833, 1016, 1470, 2768, 
			1064, 1469, 2797, 1035, 1549, 2824, 1043, 1585, 2803, 1030, 1697, 2836, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"w29aw"}, {327, 212, 740, 1037, 2736, 736, 1040, 2750, 742, 1034, 2710, 745, 1054, 2722, 807, 1068, 2779, 
			877, 1182, 2774, 929, 1298, 2891, 922, 1370, 2898, 927, 1595, 2920, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"w29eh"}, {193, 203, 597, 2426, 3018, 597, 2426, 3018, 614, 2398, 2984, 618, 2384, 2973, 664, 2275, 2945, 
			761, 2099, 2958, 791, 2010, 2981, 768, 1974, 2967, 724, 2006, 2977, 70, 0, 0, 0, 14, 6, 0, 0, 0, 0, 0, 0, 0}},
		{{"w29ei"}, {280, 246, 504, 2501, 3050, 504, 2502, 3100, 504, 2501, 3050, 502, 2495, 3027, 496, 2546, 3080, 
			479, 2567, 3133, 491, 2622, 3170, 493, 2658, 3174, 504, 2652, 3209, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w29er"}, {311, 264, 580, 1407, 1837, 586, 1370, 1916, 580, 1407, 1837, 568, 1423, 1805, 566, 1511, 1803, 
			582, 1398, 1775, 616, 1461, 1810, 569, 1504, 1867, 536, 1633, 2017, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w29ih"}, {219, 260, 544, 2570, 3135, 542, 2571, 3177, 537, 2571, 3115, 507, 2552, 3074, 502, 2488, 2998, 
			523, 2346, 2899, 622, 2240, 2864, 616, 2157, 2865, 613, 2126, 2841, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w29iy"}, {284, 275, 508, 2731, 3669, 506, 2719, 3619, 509, 2725, 3663, 509, 2744, 3665, 508, 2693, 3650, 
			500, 2741, 3636, 476, 2741, 3767, 436, 2750, 3793, 400, 2752, 3813, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w29oa"}, {285, 274, 636, 981, 2944, 634, 978, 2910, 636, 981, 2944, 619, 957, 2929, 590, 934, 2943, 
			549, 931, 2992, 568, 939, 3023, 544, 923, 3076, 499, 950, 3009, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w29oo"}, {205, 285, 570, 1117, 2820, 570, 1063, 2826, 571, 1119, 2828, 570, 1116, 2798, 572, 1150, 2770, 
			585, 1309, 2765, 644, 1464, 2757, 642, 1631, 2738, 618, 1797, 2738, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w29uh"}, {200, 307, 885, 1580, 3078, 888, 1585, 3065, 879, 1560, 3075, 838, 1492, 3011, 837, 1472, 2869, 
			921, 1500, 2906, 926, 1599, 2932, 850, 1804, 2927, 796, 1919, 2972, 75, 0, 0, 0, 0, 0, 5, 0, 0, 0, 0, 15, 0}},
		{{"w29uw"}, {274, 297, 510, 917, 2757, 510, 917, 2757, 525, 912, 2799, 561, 924, 2839, 544, 980, 2788, 
			512, 1002, 2814, 477, 992, 2810, 435, 1055, 2781, 386, 1140, 2758, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19, 0, 0}},
		{{"w30ae"}, {310, 238, 687, 2378, 2913, 678, 2376, 2980, 703, 2379, 2942, 750, 2373, 2979, 839, 2284, 2922, 
			922, 2125, 2926, 938, 1945, 2933, 938, 1883, 2969, 902, 1858, 2995, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w30ah"}, {293, 232, 957, 1591, 2733, 957, 1591, 2733, 967, 1579, 2713, 962, 1612, 2705, 933, 1624, 2680, 
			903, 1637, 2697, 892, 1684, 2745, 907, 1750, 2778, 896, 1797, 2785, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"w30aw"}, {319, 233, 867, 1207, 2746, 867, 1207, 2746, 844, 1191, 2660, 835, 1250, 2688, 861, 1323, 2718, 
			875, 1409, 2736, 879, 1529, 2728, 874, 1621, 2740, 825, 1721, 2775, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"w30eh"}, {221, 244, 886, 2186, 2956, 885, 2104, 2946, 873, 2024, 2923, 863, 2012, 2951, 827, 1997, 2960, 
			806, 1996, 2969, 753, 1973, 2949, 735, 1984, 2987, 708, 2065, 3088, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w30ei"}, {313, 245, 494, 2680, 3156, 494, 2545, 3121, 540, 2574, 3002, 517, 2558, 3130, 493, 2555, 3093, 
			491, 2615, 3126, 489, 2665, 3136, 494, 2680, 3156, 497, 2698, 3162, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w30er"}, {332, 246, 557, 1546, 1929, 525, 1518, 2007, 560, 1525, 1950, 548, 1541, 1931, 517, 1525, 1910, 
			516, 1553, 1854, 517, 1598, 0, 535, 1763, 0, 552, 1819, 0, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w30ih"}, {208, 250, 501, 2449, 3185, 499, 2473, 3165, 503, 2446, 3188, 519, 2363, 3148, 545, 2287, 3126, 
			556, 2270, 3109, 560, 2224, 3097, 560, 2173, 3112, 557, 2141, 3157, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w30iy"}, {306, 233, 452, 2754, 3511, 433, 2885, 3573, 482, 2850, 3572, 471, 2829, 3650, 459, 2794, 3639, 
			453, 2755, 3540, 451, 2748, 3499, 452, 2747, 3456, 466, 2786, 3365, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w30oa"}, {313, 230, 636, 1096, 2684, 651, 1116, 2709, 640, 1115, 2742, 624, 1099, 2771, 591, 1023, 2810, 
			552, 982, 2805, 507, 953, 2804, 499, 982, 2765, 505, 995, 2759, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w30oo"}, {216, 254, 532, 1354, 2810, 502, 1287, 2904, 525, 1340, 2853, 549, 1372, 2801, 559, 1423, 2806, 
			571, 1484, 2807, 583, 1528, 2817, 585, 1584, 2840, 576, 1816, 2991, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 0, 0, 1}},
		{{"w30uh"}, {208, 235, 835, 1507, 2834, 803, 1496, 2870, 835, 1466, 2848, 810, 1430, 2828, 806, 1472, 2855, 
			802, 1539, 2863, 789, 1553, 2836, 757, 1727, 2838, 707, 1916, 2961, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w30uw"}, {274, 268, 502, 1050, 2695, 491, 1026, 2713, 502, 1039, 2700, 502, 1055, 2692, 515, 1071, 2722, 
			507, 1102, 2752, 507, 1112, 2778, 502, 1115, 2809, 486, 1120, 2813, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w31ae"}, {364, 205, 645, 2154, 3139, 666, 2193, 3097, 671, 2166, 3036, 711, 2156, 2928, 740, 2162, 3022, 
			749, 2098, 3030, 750, 2066, 2953, 751, 2037, 2898, 756, 1920, 2931, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w31ah"}, {315, 202, 827, 1543, 2980, 817, 1560, 3026, 878, 1591, 3082, 882, 1598, 3062, 842, 1546, 3003, 
			823, 1566, 2958, 823, 1606, 2991, 829, 1629, 3044, 822, 1684, 2889, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"w31aw"}, {384, 194, 776, 1324, 2852, 722, 1222, 2848, 797, 1217, 2870, 807, 1292, 2820, 797, 1324, 2825, 
			769, 1303, 2815, 813, 1353, 2844, 809, 1384, 2807, 768, 1457, 2753, 90, 0, 0, 0, 0, 0, 1, 18, 0, 0, 0, 1, 0}},
		{{"w31eh"}, {248, 214, 588, 2130, 2870, 538, 2126, 2962, 605, 2130, 2869, 623, 2118, 2909, 626, 2042, 2887, 
			663, 1963, 2854, 708, 1883, 2789, 686, 1863, 2837, 658, 1865, 2928, 80, 0, 2, 0, 16, 2, 0, 0, 0, 0, 0, 0, 0}},
		{{"w31ei"}, {338, 212, 487, 2408, 3094, 534, 2340, 3123, 532, 2334, 3047, 493, 2385, 3090, 454, 2441, 3088, 
			439, 2457, 3089, 436, 2456, 3061, 434, 2456, 3042, 433, 2441, 2996, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w31er"}, {335, 184, 555, 1491, 1998, 563, 1551, 2012, 560, 1502, 2044, 555, 1491, 1998, 557, 1501, 2003, 
			550, 1532, 2000, 544, 1569, 2040, 557, 1610, 2056, 549, 1674, 2096, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w31ih"}, {261, 206, 431, 2190, 3060, 431, 2190, 3060, 428, 2195, 3053, 434, 2174, 3013, 470, 2164, 3011, 
			524, 2086, 3007, 564, 2066, 3029, 567, 2050, 3055, 561, 1988, 3051, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w31iy"}, {316, 237, 461, 2359, 3218, 454, 2347, 3169, 459, 2362, 3196, 457, 2344, 3140, 443, 2327, 3133, 
			435, 2364, 3149, 397, 2348, 3104, 370, 2391, 3124, 333, 2389, 3112, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w31oa"}, {363, 203, 584, 936, 2985, 607, 852, 2916, 601, 894, 2921, 580, 946, 2985, 573, 944, 2984, 
			566, 927, 2971, 560, 922, 2971, 500, 859, 2924, 443, 856, 2947, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w31oo"}, {245, 228, 482, 1124, 2704, 490, 1144, 2668, 474, 1124, 2738, 498, 1106, 2665, 560, 1198, 2682, 
			603, 1378, 2688, 593, 1434, 2637, 570, 1588, 2636, 594, 1699, 2616, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w31uh"}, {199, 216, 756, 1531, 2871, 738, 1524, 2871, 741, 1541, 2885, 750, 1570, 2893, 756, 1584, 2885, 
			758, 1622, 2886, 748, 1684, 2897, 698, 1782, 2844, 655, 1813, 2892, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w31uw"}, {277, 237, 488, 1119, 2702, 497, 1141, 2727, 494, 1131, 2675, 492, 1124, 2666, 472, 1114, 2714, 
			445, 1109, 2742, 436, 1064, 2693, 411, 1145, 2625, 381, 1307, 2648, 90, 0, 0, 0, 0, 0, 0, 0, 1, 1, 18, 0, 0}},
		{{"w32ae"}, {239, 208, 626, 2374, 2836, 626, 2413, 2858, 620, 2373, 2795, 627, 2334, 2733, 687, 2263, 2672, 
			747, 2090, 2600, 779, 1959, 2574, 801, 1859, 2537, 779, 1871, 2559, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w32ah"}, {240, 213, 708, 1547, 2426, 660, 1570, 2486, 695, 1541, 2418, 708, 1547, 2426, 675, 1559, 2439, 
			715, 1571, 2459, 816, 1583, 2483, 765, 1607, 2519, 747, 1613, 2521, 95, 0, 0, 0, 0, 0, 19, 0, 0, 0, 0, 1, 0}},
		{{"w32aw"}, {287, 226, 699, 1104, 2513, 717, 1108, 2486, 728, 1090, 2463, 692, 1119, 2501, 697, 1129, 2485, 
			705, 1176, 2513, 744, 1293, 2537, 748, 1302, 2511, 732, 1620, 2557, 90, 0, 0, 0, 0, 0, 2, 18, 0, 0, 0, 0, 0}},
		{{"w32eh"}, {191, 189, 755, 1762, 2646, 704, 1869, 2577, 783, 1831, 2596, 769, 1775, 2653, 755, 1762, 2646, 
			747, 1789, 2672, 725, 1806, 2683, 720, 1815, 2711, 683, 1810, 2748, 75, 0, 0, 1, 15, 3, 1, 0, 0, 0, 0, 0, 0}},
		{{"w32ei"}, {277, 215, 547, 2262, 2708, 551, 2273, 2752, 550, 2275, 2722, 547, 2262, 2708, 510, 2301, 2758, 
			493, 2310, 2743, 474, 2417, 2773, 455, 2430, 2795, 445, 2450, 2818, 95, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w32er"}, {272, 212, 588, 1509, 0, 505, 1551, 1893, 573, 1512, 0, 584, 1499, 0, 576, 1452, 0, 
			568, 1497, 0, 561, 1566, 0, 556, 1517, 0, 546, 1527, 0, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w32ih"}, {213, 215, 478, 2157, 2793, 450, 2312, 2945, 467, 2214, 2813, 488, 2150, 2758, 529, 2137, 2731, 
			547, 2109, 2725, 564, 2161, 2755, 565, 2100, 2744, 557, 2014, 2748, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w32iy"}, {261, 216, 433, 2720, 3035, 436, 2689, 3007, 435, 2692, 3041, 431, 2724, 3023, 429, 2753, 3000, 
			423, 2741, 3019, 411, 2775, 3061, 407, 2696, 3065, 408, 2660, 3039, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w32oa"}, {239, 217, 499, 1046, 2697, 639, 1125, 2628, 627, 1056, 2620, 608, 1024, 2616, 565, 1036, 2650, 
			554, 1042, 2702, 507, 1040, 2697, 480, 1048, 2698, 447, 1067, 2727, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w32oo"}, {183, 205, 541, 1183, 2780, 467, 1256, 2777, 500, 1207, 2779, 557, 1206, 2780, 578, 1270, 2676, 
			584, 1363, 2671, 589, 1419, 2675, 591, 1468, 2656, 594, 1555, 2681, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"w32uh"}, {224, 209, 685, 1100, 2510, 636, 1111, 2492, 685, 1100, 2510, 693, 1106, 2546, 701, 1112, 2560, 
			690, 1137, 2611, 714, 1230, 2551, 700, 1330, 2500, 687, 1564, 2487, 55, 0, 0, 0, 0, 0, 4, 5, 0, 0, 0, 11, 0}},
		{{"w32uw"}, {242, 238, 477, 1184, 2610, 468, 1182, 2625, 477, 1184, 2623, 466, 1180, 2593, 449, 1157, 2704, 
			440, 1125, 2754, 435, 1087, 2638, 429, 1093, 2692, 426, 1195, 2703, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w33ae"}, {261, 235, 689, 2701, 3490, 689, 2701, 3490, 689, 2774, 3534, 711, 2698, 3482, 863, 2629, 3473, 
			905, 2495, 3527, 924, 2311, 3540, 937, 2133, 3606, 962, 2174, 3602, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w33ah"}, {283, 241, 1163, 1685, 3250, 1156, 1685, 3233, 1152, 1676, 3352, 1127, 1679, 3297, 1110, 1683, 3360, 
			1117, 1707, 3357, 1058, 1771, 3370, 977, 1907, 3438, 1046, 2078, 3433, 90, 0, 0, 0, 0, 0, 18, 2, 0, 0, 0, 0, 0}},
		{{"w33aw"}, {328, 241, 906, 1386, 3066, 906, 1386, 3066, 915, 1413, 3137, 905, 1412, 3184, 934, 1437, 3263, 
			938, 1551, 3319, 942, 1653, 3363, 939, 1708, 3379, 938, 1842, 3318, 55, 0, 0, 0, 0, 0, 9, 11, 0, 0, 0, 0, 0}},
		{{"w33eh"}, {194, 236, 885, 2322, 3652, 887, 2285, 3579, 895, 2282, 3615, 888, 2315, 3636, 880, 2325, 3664, 
			859, 2321, 3689, 794, 2307, 3772, 764, 2311, 3780, 759, 2338, 3783, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w33ei"}, {296, 247, 500, 2889, 3452, 528, 2779, 3583, 516, 2834, 3514, 502, 2883, 3521, 501, 2929, 3423, 
			500, 2951, 3455, 502, 3014, 3479, 511, 3058, 3568, 544, 2994, 3492, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w33er"}, {345, 252, 549, 1749, 2209, 566, 1752, 2236, 553, 1750, 2220, 551, 1749, 2205, 555, 1751, 2222, 
			554, 1759, 2242, 554, 1787, 2243, 558, 1821, 2259, 560, 1888, 2258, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w33ih"}, {227, 252, 556, 2541, 3684, 527, 2671, 3668, 532, 2628, 3644, 553, 2557, 3655, 559, 2517, 3717, 
			565, 2538, 3738, 571, 2539, 3744, 571, 2517, 3753, 570, 2416, 3745, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w33iy"}, {257, 268, 505, 2982, 3741, 506, 2986, 3737, 504, 2984, 3739, 505, 2997, 3736, 506, 3006, 3727, 
			505, 3039, 3717, 504, 3023, 3677, 499, 3020, 3706, 495, 3006, 3774, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w33oa"}, {277, 247, 504, 990, 3324, 682, 1150, 3221, 641, 1143, 3207, 614, 1122, 3250, 539, 993, 3312, 
			507, 989, 3319, 502, 994, 3320, 502, 1000, 3293, 516, 1063, 3297, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w33oo"}, {187, 244, 615, 1375, 3295, 615, 1375, 3295, 613, 1425, 3334, 605, 1449, 3304, 598, 1484, 3260, 
			608, 1537, 3247, 622, 1660, 3270, 628, 1799, 3303, 621, 2001, 3346, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"w33uh"}, {193, 244, 828, 1454, 3287, 829, 1454, 3287, 833, 1504, 3243, 833, 1558, 3237, 818, 1690, 3237, 
			791, 1744, 3245, 754, 1804, 3283, 737, 2000, 3328, 708, 2054, 3377, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w33uw"}, {291, 272, 504, 1068, 3121, 513, 1070, 3040, 497, 1095, 3078, 499, 1084, 3087, 502, 1064, 3146, 
			504, 1064, 3201, 504, 1075, 3197, 499, 1101, 3196, 470, 1118, 3161, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w34ae"}, {277, 218, 557, 2586, 3202, 563, 2553, 3064, 606, 2514, 2902, 611, 2459, 2818, 626, 2359, 2872, 
			689, 2104, 2802, 804, 2013, 2857, 839, 1896, 2849, 731, 1916, 2854, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w34ah"}, {247, 211, 1011, 1541, 2803, 1014, 1542, 2801, 1013, 1548, 2798, 999, 1558, 2820, 993, 1589, 2818, 
			980, 1616, 2815, 954, 1658, 2812, 951, 1720, 2801, 931, 1808, 2852, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"w34aw"}, {253, 210, 814, 1144, 2729, 814, 1144, 2729, 791, 1161, 2703, 800, 1166, 2726, 813, 1255, 2729, 
			813, 1297, 2725, 817, 1376, 2679, 799, 1521, 2658, 744, 1667, 2735, 45, 0, 0, 0, 0, 0, 10, 9, 0, 0, 0, 1, 0}},
		{{"w34eh"}, {179, 214, 804, 1998, 2775, 818, 2004, 2839, 806, 1982, 2796, 804, 2000, 2794, 802, 2001, 2786, 
			794, 2034, 2858, 767, 2032, 2885, 761, 2020, 2919, 693, 2036, 2983, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w34ei"}, {267, 211, 433, 2773, 3297, 493, 2649, 3170, 470, 2687, 3176, 443, 2646, 3186, 437, 2666, 3186, 
			434, 2764, 3294, 432, 2769, 3295, 432, 2793, 3352, 434, 2791, 3371, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w34er"}, {240, 214, 461, 1534, 1878, 487, 1541, 1806, 499, 1535, 1863, 495, 1552, 1822, 461, 1534, 1878, 
			450, 1543, 1878, 447, 1592, 1871, 440, 1656, 1886, 437, 1697, 1970, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w34ih"}, {175, 231, 486, 2454, 3106, 486, 2475, 3163, 480, 2446, 3091, 464, 2458, 3107, 477, 2424, 3098, 
			486, 2360, 3084, 493, 2288, 3067, 493, 2240, 3076, 484, 2220, 3111, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w34iy"}, {236, 220, 434, 2854, 3614, 438, 2820, 3524, 437, 2907, 3595, 435, 2916, 3605, 434, 2869, 3619, 
			432, 2831, 3684, 433, 2837, 3702, 433, 2839, 3660, 434, 2858, 3669, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w34oa"}, {254, 219, 453, 880, 2924, 591, 1058, 2724, 600, 1062, 2759, 570, 1006, 2859, 521, 924, 2860, 
			483, 887, 2865, 443, 886, 2886, 441, 928, 2873, 442, 1068, 2838, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w34oo"}, {187, 232, 579, 1392, 2760, 577, 1291, 2813, 571, 1373, 2763, 573, 1375, 2790, 595, 1455, 2762, 
			597, 1568, 2761, 595, 1628, 2752, 560, 1803, 2760, 515, 1877, 2746, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w34uh"}, {164, 222, 764, 1359, 2875, 873, 1366, 2849, 764, 1359, 2873, 769, 1405, 2874, 774, 1506, 2879, 
			774, 1540, 2896, 753, 1651, 2902, 711, 1749, 2901, 658, 1894, 2933, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w34uw"}, {244, 242, 483, 938, 2835, 497, 1002, 2804, 496, 995, 2863, 492, 986, 2901, 489, 967, 2892, 
			485, 950, 2848, 484, 934, 2843, 486, 935, 2867, 485, 950, 2861, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w35ae"}, {328, 211, 668, 2296, 2711, 664, 2296, 2706, 666, 2281, 2685, 675, 2258, 2663, 687, 2218, 2648, 
			745, 2078, 2654, 772, 1987, 2664, 771, 1883, 2662, 769, 1871, 2680, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w35ah"}, {310, 201, 1035, 1486, 2865, 968, 1489, 2862, 1026, 1490, 2866, 1028, 1494, 2870, 1008, 1508, 2854, 
			1004, 1538, 2817, 986, 1556, 2787, 967, 1614, 2784, 946, 1676, 2754, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"w35aw"}, {331, 207, 801, 1187, 2935, 814, 1187, 2896, 783, 1190, 2909, 801, 1185, 2944, 828, 1188, 2961, 
			862, 1190, 2902, 879, 1222, 2875, 862, 1286, 2866, 840, 1473, 2803, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"w35eh"}, {223, 202, 718, 2005, 2906, 626, 2297, 3073, 630, 2244, 3066, 653, 2196, 3046, 685, 2084, 2989, 
			705, 2014, 2912, 727, 2003, 2884, 717, 2000, 2844, 696, 1989, 2818, 55, 0, 0, 0, 11, 9, 0, 0, 0, 0, 0, 0, 0}},
		{{"w35ei"}, {314, 215, 591, 2562, 3156, 616, 2483, 3147, 591, 2562, 3156, 566, 2570, 3172, 532, 2576, 3148, 
			472, 2604, 3158, 438, 2643, 3137, 438, 2686, 3112, 439, 2687, 3124, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w35er"}, {300, 218, 537, 1509, 1682, 555, 1529, 1730, 541, 1520, 1705, 511, 1542, 1666, 504, 1555, 1692, 
			497, 1562, 1733, 489, 1559, 1758, 475, 1568, 1795, 456, 1606, 1809, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w35ih"}, {176, 218, 442, 2472, 3010, 442, 2472, 3010, 440, 2442, 2932, 448, 2396, 2867, 499, 2296, 2804, 
			532, 2230, 2773, 557, 2149, 2785, 564, 2134, 2822, 564, 2133, 2867, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w35iy"}, {276, 215, 426, 2775, 3535, 431, 2728, 3532, 428, 2787, 3560, 422, 2754, 3570, 420, 2749, 3542, 
			421, 2748, 3531, 423, 2755, 3527, 426, 2782, 3530, 425, 2810, 3505, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w35oa"}, {309, 210, 641, 1195, 2890, 699, 1169, 2818, 667, 1218, 2902, 635, 1186, 2916, 625, 1125, 2934, 
			613, 1025, 2977, 570, 973, 3010, 516, 994, 2962, 473, 1053, 2984, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w35oo"}, {203, 236, 552, 1364, 2809, 549, 1425, 2769, 552, 1397, 2809, 549, 1349, 2807, 550, 1326, 2814, 
			558, 1340, 2818, 572, 1422, 2827, 584, 1467, 2829, 603, 1694, 2811, 95, 0, 0, 0, 1, 0, 0, 0, 0, 19, 0, 0, 0}},
		{{"w35uh"}, {215, 207, 729, 1435, 3109, 780, 1446, 2996, 742, 1436, 3025, 733, 1433, 3098, 721, 1435, 3107, 
			720, 1455, 3099, 713, 1519, 3021, 695, 1661, 2898, 681, 1708, 2862, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w35uw"}, {294, 232, 448, 1463, 2693, 445, 1457, 2688, 443, 1408, 2736, 447, 1367, 2755, 449, 1363, 2766, 
			451, 1355, 2782, 453, 1302, 2781, 456, 1254, 2783, 466, 1208, 2779, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w36ae"}, {413, 200, 746, 2371, 2984, 741, 2368, 3037, 746, 2385, 2983, 756, 2362, 2909, 762, 2355, 3013, 
			789, 2204, 2999, 808, 2028, 2913, 810, 1952, 2918, 807, 1944, 2915, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w36ah"}, {455, 208, 952, 1676, 2862, 951, 1665, 2872, 939, 1616, 2909, 931, 1615, 2927, 916, 1637, 2949, 
			892, 1681, 2906, 874, 1743, 2928, 867, 1791, 2964, 825, 1836, 2928, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"w36aw"}, {422, 213, 822, 1177, 2950, 835, 1117, 2946, 828, 1137, 2946, 822, 1167, 2948, 824, 1215, 2947, 
			831, 1243, 2968, 856, 1312, 2962, 861, 1468, 2996, 844, 1724, 2983, 75, 0, 0, 0, 0, 0, 5, 15, 0, 0, 0, 0, 0}},
		{{"w36eh"}, {368, 205, 773, 2042, 3009, 769, 2038, 3016, 762, 2046, 3049, 756, 2041, 3038, 747, 2038, 3037, 
			734, 2037, 3042, 711, 2011, 3015, 683, 2020, 3003, 641, 2038, 2996, 85, 0, 0, 0, 17, 3, 0, 0, 0, 0, 0, 0, 0}},
		{{"w36ei"}, {402, 211, 596, 2314, 3113, 584, 2271, 3045, 587, 2313, 3117, 569, 2303, 3099, 552, 2342, 3110, 
			500, 2392, 3102, 454, 2433, 3087, 439, 2456, 3052, 436, 2462, 3046, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w36er"}, {438, 206, 492, 1635, 1871, 466, 1698, 1983, 484, 1641, 1933, 478, 1592, 1885, 473, 1623, 1865, 
			460, 1591, 1841, 443, 1568, 1847, 437, 1615, 1868, 434, 1617, 1870, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w36ih"}, {353, 225, 486, 2253, 3212, 443, 2430, 3183, 444, 2409, 3201, 453, 2393, 3167, 472, 2335, 3213, 
			478, 2298, 3216, 484, 2252, 3220, 486, 2269, 3209, 483, 2299, 3178, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w36iy"}, {324, 220, 424, 2692, 3336, 428, 2749, 3387, 426, 2714, 3439, 426, 2700, 3394, 425, 2693, 3424, 
			424, 2692, 3336, 424, 2680, 3351, 424, 2695, 3275, 428, 2701, 3230, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w36oa"}, {340, 217, 620, 1057, 2833, 659, 1148, 2809, 641, 1047, 2823, 629, 1048, 2846, 620, 1057, 2833, 
			579, 1057, 2846, 546, 1032, 2862, 501, 999, 2743, 499, 1069, 2748, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w36oo"}, {328, 214, 449, 1396, 2702, 441, 1318, 2672, 478, 1441, 2744, 511, 1468, 2734, 564, 1496, 2726, 
			605, 1617, 2748, 613, 1685, 2760, 599, 1730, 2735, 569, 1866, 2800, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w36uh"}, {365, 206, 761, 1380, 3042, 755, 1374, 3032, 760, 1386, 3002, 754, 1424, 2989, 755, 1527, 2934, 
			748, 1636, 2896, 692, 1754, 2862, 669, 1822, 2852, 635, 1860, 2813, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w36uw"}, {405, 219, 438, 1711, 2636, 440, 1681, 2652, 437, 1668, 2628, 435, 1545, 2637, 430, 1467, 2653, 
			429, 1365, 2643, 428, 1265, 2659, 428, 1216, 2667, 433, 1252, 2676, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w37ae"}, {405, 192, 893, 2070, 3024, 947, 2070, 2882, 898, 2088, 2916, 881, 2072, 3043, 885, 2064, 3037, 
			907, 2051, 3016, 914, 2038, 2995, 912, 2046, 3003, 874, 2026, 3028, 80, 0, 0, 0, 4, 16, 0, 0, 0, 0, 0, 0, 0}},
		{{"w37ah"}, {419, 175, 810, 1314, 3236, 864, 1263, 3261, 833, 1238, 3279, 832, 1271, 3231, 810, 1314, 3175, 
			838, 1330, 3170, 903, 1404, 3153, 940, 1525, 3157, 907, 1719, 3115, 55, 0, 0, 0, 0, 0, 11, 9, 0, 0, 0, 0, 0}},
		{{"w37aw"}, {413, 196, 851, 1105, 3249, 824, 1106, 3287, 843, 1081, 3254, 866, 1121, 3256, 883, 1168, 3257, 
			878, 1207, 3279, 874, 1234, 3190, 892, 1307, 3201, 901, 1424, 3183, 90, 0, 0, 0, 0, 0, 2, 18, 0, 0, 0, 0, 0}},
		{{"w37eh"}, {343, 191, 694, 2160, 3144, 678, 2338, 3086, 663, 2291, 3107, 650, 2264, 3129, 656, 2228, 3113, 
			691, 2177, 3111, 694, 2160, 3144, 693, 2138, 3132, 672, 2112, 3123, 95, 1, 0, 0, 19, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w37ei"}, {367, 193, 500, 2622, 3305, 573, 2561, 3204, 558, 2585, 3252, 519, 2582, 3316, 495, 2633, 3294, 
			447, 2644, 3310, 432, 2686, 3318, 403, 2746, 3298, 376, 2879, 3264, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w37er"}, {371, 198, 455, 1769, 2070, 494, 1816, 2309, 487, 1785, 2108, 476, 1802, 2109, 454, 1772, 2060, 
			446, 1759, 2054, 438, 1770, 2037, 444, 1831, 2085, 437, 1905, 2157, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w37ih"}, {363, 201, 440, 2453, 3123, 440, 2453, 3123, 447, 2423, 3110, 494, 2324, 3108, 515, 2265, 3081, 
			545, 2242, 3068, 551, 2245, 3097, 558, 2159, 3073, 561, 2121, 3071, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w37iy"}, {396, 201, 380, 2816, 3312, 415, 2778, 3314, 396, 2831, 3322, 385, 2859, 3358, 383, 2837, 3341, 
			382, 2828, 3335, 381, 2816, 3314, 380, 2816, 3312, 382, 2818, 3314, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w37oa"}, {406, 195, 563, 1198, 3046, 621, 1309, 3004, 592, 1305, 3007, 570, 1252, 3036, 560, 1197, 3032, 
			546, 1160, 3044, 502, 1147, 3052, 447, 1130, 3041, 426, 1112, 3057, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w37oo"}, {333, 228, 464, 1169, 2985, 461, 1172, 2981, 467, 1204, 2983, 504, 1184, 3031, 552, 1364, 3020, 
			564, 1422, 3008, 588, 1554, 2989, 609, 1640, 2943, 603, 1785, 2873, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"w37uh"}, {325, 206, 809, 1468, 3153, 817, 1539, 3232, 803, 1472, 3141, 817, 1468, 3169, 832, 1541, 3210, 
			817, 1599, 3208, 806, 1695, 3189, 791, 1763, 3190, 759, 1819, 3198, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w37uw"}, {406, 220, 437, 1178, 2852, 451, 1170, 2879, 438, 1174, 2871, 436, 1176, 2852, 436, 1188, 2832, 
			435, 1179, 2820, 435, 1161, 2814, 435, 1119, 2815, 435, 1111, 2843, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w38ae"}, {294, 219, 665, 2408, 3034, 678, 2391, 3074, 659, 2436, 3026, 646, 2412, 2973, 686, 2288, 2991, 
			734, 2114, 2950, 751, 2020, 2909, 756, 1993, 2874, 747, 2005, 2859, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"w38ah"}, {336, 217, 993, 1671, 2751, 989, 1624, 2686, 997, 1671, 2758, 993, 1668, 2751, 969, 1651, 2726, 
			928, 1647, 2724, 914, 1702, 2813, 844, 1746, 2840, 810, 1819, 2859, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"w38aw"}, {327, 219, 825, 1213, 2751, 835, 1234, 2697, 841, 1248, 2792, 862, 1350, 2801, 866, 1405, 2825, 
			849, 1485, 2880, 827, 1601, 2889, 822, 1743, 2892, 755, 1941, 2929, 90, 0, 0, 0, 1, 0, 1, 18, 0, 0, 0, 0, 0}},
		{{"w38eh"}, {268, 210, 680, 2068, 3072, 808, 2099, 2985, 812, 2103, 2948, 807, 2107, 3009, 796, 2098, 3116, 
			759, 2100, 3089, 723, 2090, 3061, 680, 2068, 3072, 634, 2091, 3104, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w38ei"}, {310, 224, 645, 2554, 3047, 645, 2554, 3047, 611, 2603, 3046, 510, 2595, 3086, 491, 2616, 3104, 
			457, 2672, 3131, 452, 2690, 3098, 454, 2761, 3088, 481, 2802, 3078, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w38er"}, {303, 230, 656, 1614, 2008, 729, 1644, 2122, 679, 1620, 2058, 659, 1613, 2024, 622, 1647, 1988, 
			588, 1705, 1988, 578, 1690, 1988, 578, 1675, 2043, 566, 1793, 2041, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w38ih"}, {239, 231, 477, 2554, 3296, 495, 2593, 3305, 478, 2561, 3293, 478, 2540, 3305, 489, 2554, 3303, 
			493, 2549, 3282, 500, 2508, 3248, 507, 2463, 3202, 507, 2372, 3130, 95, 0, 19, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w38iy"}, {276, 223, 439, 2927, 3283, 391, 2770, 0, 492, 2906, 0, 496, 2855, 0, 495, 2912, 0, 
			478, 2906, 0, 450, 2986, 0, 441, 2942, 3300, 438, 2919, 3268, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w38oa"}, {296, 221, 632, 1108, 2869, 691, 1155, 2818, 660, 1122, 2873, 632, 1108, 2869, 619, 1065, 2917, 
			590, 1052, 2954, 566, 1024, 2971, 512, 1026, 2978, 489, 1060, 2913, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w38oo"}, {251, 258, 552, 1214, 2882, 549, 1165, 2887, 557, 1242, 2928, 578, 1405, 2924, 613, 1448, 2863, 
			627, 1601, 2808, 621, 1701, 2871, 614, 1854, 2873, 578, 1981, 2924, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"w38uh"}, {214, 228, 818, 1516, 2970, 818, 1516, 2970, 814, 1530, 2926, 794, 1532, 2950, 775, 1525, 2950, 
			756, 1573, 2952, 736, 1727, 2933, 690, 1763, 2928, 680, 1982, 3054, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w38uw"}, {277, 260, 504, 1041, 2838, 502, 1210, 2799, 503, 1107, 2804, 505, 1090, 2811, 503, 1059, 2811, 
			503, 1043, 2816, 504, 1044, 2852, 509, 1056, 2869, 514, 1076, 2916, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w39ae"}, {365, 192, 564, 2442, 0, 567, 2558, 0, 565, 2446, 0, 566, 2432, 0, 575, 2337, 0, 
			710, 2057, 2591, 922, 1880, 2613, 955, 1746, 2632, 988, 1777, 2717, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w39ah"}, {338, 212, 931, 1348, 2698, 988, 1346, 2630, 958, 1301, 2603, 955, 1337, 2656, 950, 1307, 2658, 
			947, 1363, 2671, 942, 1374, 2691, 949, 1482, 2675, 873, 1646, 2723, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"w39aw"}, {369, 211, 752, 1101, 2616, 693, 1066, 2622, 713, 1058, 2674, 710, 1063, 2616, 726, 1072, 2650, 
			750, 1094, 2628, 755, 1114, 2574, 778, 1202, 2566, 782, 1482, 2610, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"w39eh"}, {259, 193, 694, 2052, 2838, 694, 2052, 2838, 687, 2008, 2810, 688, 2006, 2784, 698, 1990, 2803, 
			686, 2005, 2825, 649, 2005, 2837, 608, 2019, 2876, 567, 2007, 2866, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w39ei"}, {344, 198, 434, 2568, 3053, 531, 2514, 3046, 485, 2562, 3057, 438, 2564, 3055, 421, 2580, 3051, 
			415, 2601, 3067, 422, 2578, 3076, 429, 2635, 3092, 433, 2663, 3066, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w39er"}, {340, 189, 509, 1365, 0, 505, 1373, 1603, 519, 1318, 1661, 513, 1318, 1620, 509, 1365, 0, 
			510, 1458, 0, 506, 1480, 0, 478, 1504, 0, 432, 1567, 1738, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w39ih"}, {247, 216, 436, 2559, 3104, 437, 2540, 3087, 436, 2517, 3037, 434, 2439, 3018, 446, 2392, 2978, 
			494, 2338, 2923, 545, 2199, 2897, 553, 2145, 2901, 515, 2151, 2909, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w39iy"}, {338, 210, 430, 2606, 0, 439, 2748, 3171, 435, 2592, 3162, 433, 2576, 3072, 431, 2582, 3041, 
			429, 2606, 0, 430, 2615, 0, 433, 2591, 0, 436, 2586, 0, 95, 19, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w39oa"}, {353, 204, 430, 820, 2783, 559, 960, 2648, 557, 944, 2611, 520, 942, 2673, 475, 917, 2704, 
			435, 855, 2743, 430, 820, 2783, 436, 850, 2690, 438, 861, 2684, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w39oo"}, {236, 187, 486, 1088, 2663, 430, 983, 2729, 437, 1017, 2767, 445, 1054, 2732, 486, 1088, 2663, 
			501, 1087, 2591, 519, 1115, 2592, 542, 1397, 2639, 512, 1643, 2674, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w39uh"}, {233, 198, 710, 1293, 2623, 770, 1272, 2623, 742, 1320, 2532, 719, 1247, 2634, 721, 1296, 2629, 
			705, 1297, 2615, 683, 1405, 2580, 624, 1663, 2923, 587, 1809, 2884, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w39uw"}, {266, 215, 429, 827, 2719, 429, 906, 2753, 417, 916, 2688, 412, 851, 2683, 420, 800, 2706, 
			425, 815, 2704, 429, 827, 2719, 432, 845, 2727, 435, 930, 2668, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19, 0, 0}},
		{{"w40ae"}, {301, 197, 714, 2254, 2625, 714, 2254, 2625, 727, 2248, 2604, 728, 2131, 2551, 722, 2105, 2628, 
			747, 2069, 2698, 800, 1966, 2610, 807, 1907, 2617, 790, 1895, 2748, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w40ah"}, {297, 198, 822, 1371, 3130, 802, 1407, 3222, 816, 1390, 3205, 816, 1396, 3087, 802, 1350, 3260, 
			823, 1367, 3179, 824, 1390, 3196, 800, 1385, 3147, 833, 1554, 3176, 85, 0, 0, 0, 0, 0, 17, 3, 0, 0, 0, 0, 0}},
		{{"w40aw"}, {303, 200, 765, 997, 3477, 764, 1003, 3488, 754, 1036, 3539, 748, 975, 3513, 766, 0, 3517, 
			763, 0, 3475, 777, 0, 3385, 777, 1449, 3135, 749, 1655, 2949, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"w40eh"}, {237, 185, 726, 2123, 2959, 699, 2356, 3160, 732, 2181, 2949, 737, 2186, 2922, 732, 2135, 2952, 
			719, 2081, 2945, 709, 2043, 2881, 697, 2041, 2867, 684, 2047, 2911, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w40ei"}, {296, 202, 543, 2399, 2775, 543, 2399, 2775, 536, 2386, 2733, 507, 2454, 2736, 485, 2454, 2738, 
			433, 2459, 2761, 403, 2519, 2754, 381, 2717, 0, 384, 2778, 0, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w40er"}, {304, 207, 584, 1486, 1783, 520, 1512, 1859, 600, 1491, 1804, 584, 1486, 1783, 572, 1471, 1795, 
			563, 1487, 1792, 547, 1482, 1798, 515, 1492, 1823, 494, 1556, 1867, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w40ih"}, {227, 205, 501, 2292, 2863, 488, 2304, 2903, 502, 2285, 2863, 518, 2304, 2905, 536, 2306, 2893, 
			552, 2265, 2884, 558, 2231, 2894, 559, 2200, 2929, 557, 2210, 2926, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w40iy"}, {271, 198, 383, 2733, 3326, 425, 2661, 3272, 407, 2637, 3278, 390, 2698, 3330, 382, 2722, 3340, 
			380, 2714, 3282, 377, 2717, 3189, 377, 2759, 3220, 379, 2740, 3179, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w40oa"}, {305, 207, 438, 812, 3068, 625, 1004, 3195, 599, 970, 3196, 529, 854, 3130, 465, 801, 3120, 
			448, 809, 3081, 438, 812, 3068, 436, 835, 3017, 437, 874, 3031, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w40oo"}, {233, 205, 516, 1019, 2890, 498, 1005, 2879, 516, 1019, 2890, 514, 961, 2888, 533, 943, 2883, 
			560, 985, 2915, 574, 1200, 2867, 609, 1457, 2793, 599, 1641, 2763, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w40uh"}, {233, 194, 755, 1427, 2965, 809, 1391, 2927, 767, 1400, 2965, 738, 1449, 2915, 714, 1633, 2873, 
			624, 1816, 2947, 508, 1874, 3256, 258, 1932, 3076, 311, 2082, 3226, 75, 0, 0, 0, 0, 0, 2, 3, 0, 0, 0, 15, 0}},
		{{"w40uw"}, {296, 223, 438, 1078, 2774, 439, 1100, 2602, 438, 1078, 2774, 437, 1059, 2751, 436, 1052, 2750, 
			435, 1011, 2698, 434, 1035, 2773, 436, 1019, 2789, 437, 1024, 2757, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w41ae"}, {301, 216, 625, 2594, 3146, 627, 2607, 3170, 625, 2625, 3158, 625, 2624, 3155, 631, 2547, 3144, 
			653, 2464, 3146, 742, 2339, 3143, 781, 2215, 3141, 773, 2080, 3187, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"w41ah"}, {298, 200, 875, 1491, 2979, 885, 1447, 2991, 890, 1513, 2984, 882, 1503, 2966, 876, 1502, 2986, 
			864, 1537, 3000, 837, 1598, 2986, 839, 1693, 2953, 847, 1782, 2933, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"w41aw"}, {332, 198, 957, 0, 3066, 954, 0, 3045, 992, 0, 3054, 972, 0, 3079, 966, 0, 3075, 
			961, 0, 3044, 943, 1241, 3050, 948, 1349, 3031, 937, 1559, 2938, 95, 0, 0, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0}},
		{{"w41eh"}, {203, 217, 714, 2255, 2941, 666, 2347, 2938, 714, 2255, 2941, 739, 2202, 2936, 767, 2107, 2966, 
			773, 2006, 2993, 766, 1995, 3031, 752, 2021, 3069, 719, 2037, 3103, 35, 0, 0, 0, 7, 13, 0, 0, 0, 0, 0, 0, 0}},
		{{"w41ei"}, {298, 214, 555, 2766, 3181, 555, 2766, 3181, 539, 2752, 3282, 502, 2892, 3194, 483, 2879, 3218, 
			444, 2943, 3236, 439, 2916, 3260, 437, 2894, 3282, 437, 2997, 3323, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w41er"}, {262, 208, 491, 1655, 0, 578, 1686, 0, 561, 1655, 0, 502, 1655, 0, 485, 1648, 0, 
			467, 1644, 0, 445, 1686, 0, 443, 1696, 0, 443, 1740, 0, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w41ih"}, {200, 219, 443, 2644, 3306, 444, 2760, 3345, 444, 2684, 3335, 443, 2601, 3282, 466, 2515, 3240, 
			496, 2407, 3228, 516, 2342, 3241, 512, 2278, 3245, 501, 2230, 3239, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w41iy"}, {263, 229, 438, 2964, 3632, 459, 2970, 3615, 447, 3033, 3623, 443, 3031, 3626, 441, 3020, 3645, 
			439, 3007, 3679, 438, 2983, 3663, 438, 2960, 3629, 439, 2967, 3626, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w41oa"}, {288, 226, 450, 889, 2904, 615, 1018, 2609, 577, 949, 2767, 551, 892, 2845, 518, 872, 2895, 
			483, 871, 2937, 456, 875, 2922, 448, 898, 2912, 445, 922, 2938, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w41oo"}, {217, 226, 444, 1081, 3049, 445, 1113, 3029, 444, 1106, 3030, 453, 1087, 2993, 488, 1204, 2985, 
			530, 1388, 3023, 565, 1513, 2979, 568, 1622, 2943, 565, 1758, 2896, 90, 0, 0, 0, 0, 0, 0, 0, 0, 18, 2, 0, 0}},
		{{"w41uh"}, {201, 215, 827, 1416, 3022, 832, 1416, 3022, 826, 1431, 3038, 797, 1453, 3111, 768, 1466, 3110, 
			757, 1499, 3095, 753, 1583, 3072, 723, 1699, 3036, 679, 1785, 3019, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w41uw"}, {235, 237, 450, 916, 2959, 460, 962, 2657, 455, 919, 2664, 451, 916, 2930, 447, 936, 3001, 
			445, 930, 2885, 444, 935, 2865, 442, 979, 2894, 443, 1012, 2944, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w42ae"}, {312, 222, 552, 2227, 2978, 562, 2232, 2991, 569, 2204, 3004, 570, 2187, 2982, 605, 2144, 2890, 
			654, 2070, 2844, 736, 1898, 2806, 778, 1855, 2798, 766, 1746, 2809, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w42ah"}, {268, 226, 884, 1568, 2819, 884, 1610, 2828, 900, 1612, 2812, 886, 1588, 2825, 881, 1566, 2822, 
			880, 1561, 2820, 879, 1572, 2806, 874, 1634, 2804, 869, 1722, 2917, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"w42aw"}, {387, 237, 799, 1146, 2770, 799, 1146, 2770, 869, 1236, 2771, 879, 1291, 2722, 874, 1314, 2766, 
			873, 1302, 2782, 870, 1306, 2733, 867, 1451, 2795, 861, 1507, 2749, 85, 0, 0, 0, 0, 0, 3, 17, 0, 0, 0, 0, 0}},
		{{"w42eh"}, {250, 209, 623, 1882, 3007, 655, 1956, 3040, 637, 1954, 3041, 622, 1937, 3041, 624, 1918, 3030, 
			622, 1889, 3007, 622, 1872, 3011, 620, 1867, 3020, 618, 1876, 3035, 95, 0, 0, 0, 19, 0, 0, 0, 0, 1, 0, 0, 0}},
		{{"w42ei"}, {343, 232, 484, 2276, 3151, 498, 2192, 3121, 479, 2298, 3151, 452, 2370, 3053, 467, 2423, 3027, 
			456, 2537, 3050, 443, 2554, 3025, 438, 2573, 2938, 438, 2559, 3008, 70, 6, 0, 14, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w42er"}, {331, 240, 482, 1676, 0, 488, 1681, 0, 481, 1646, 0, 484, 1656, 0, 481, 1685, 0, 
			486, 1674, 0, 489, 1688, 0, 496, 1742, 0, 498, 1727, 0, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w42ih"}, {170, 236, 489, 2175, 2990, 489, 2196, 2990, 469, 2130, 3018, 459, 2116, 3010, 463, 2108, 3002, 
			471, 2105, 3003, 473, 2099, 3040, 481, 2086, 3064, 489, 2050, 3087, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w42iy"}, {376, 250, 427, 2772, 3762, 325, 2820, 3690, 369, 2797, 3708, 406, 2747, 3778, 423, 2780, 3770, 
			434, 2761, 3794, 440, 2779, 3806, 445, 2775, 3814, 446, 2859, 3859, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w42oa"}, {318, 219, 444, 987, 2727, 556, 1074, 2716, 525, 1076, 2693, 500, 1059, 2772, 489, 1047, 2758, 
			459, 997, 2751, 444, 987, 2727, 438, 980, 2704, 437, 998, 2666, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w42oo"}, {246, 240, 497, 1440, 2713, 494, 1440, 2713, 487, 1431, 2724, 486, 1449, 2757, 490, 1466, 2752, 
			495, 1538, 2753, 500, 1605, 2770, 500, 1628, 2783, 497, 1733, 2842, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w42uh"}, {156, 236, 736, 1300, 2829, 736, 1300, 2829, 746, 1399, 2878, 769, 1203, 2821, 806, 1373, 2809, 
			798, 1491, 2784, 776, 1687, 2828, 743, 1746, 2837, 683, 1824, 2893, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w42uw"}, {272, 255, 474, 974, 2540, 443, 1298, 2562, 445, 1243, 2562, 452, 1153, 2565, 466, 1063, 2557, 
			473, 974, 2544, 477, 982, 2530, 483, 980, 2521, 489, 986, 2518, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w44ae"}, {353, 230, 685, 2205, 2813, 685, 2205, 2813, 681, 2232, 2839, 667, 2187, 2898, 671, 2132, 2844, 
			728, 1978, 2743, 785, 1768, 2610, 803, 1670, 2591, 770, 1494, 2604, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w44ah"}, {340, 212, 938, 1492, 2673, 922, 1501, 2673, 942, 1492, 2677, 949, 1483, 2655, 955, 1503, 2634, 
			954, 1531, 2641, 943, 1547, 2660, 900, 1588, 2686, 834, 1627, 2730, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"w44aw"}, {390, 222, 715, 963, 2455, 694, 954, 2498, 711, 1007, 2449, 714, 1028, 2433, 711, 1193, 2426, 
			709, 1298, 2406, 707, 1316, 2394, 705, 1453, 2562, 703, 1548, 2622, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"w44eh"}, {255, 221, 718, 1952, 2817, 708, 1980, 2872, 727, 1966, 2857, 730, 1963, 2812, 718, 1952, 2817, 
			698, 1942, 2805, 686, 1941, 2820, 683, 1943, 2835, 681, 1933, 2933, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w44ei"}, {300, 217, 442, 2464, 3067, 472, 2432, 3069, 442, 2464, 3067, 437, 2488, 3053, 438, 2498, 3039, 
			443, 2554, 3029, 448, 2562, 3078, 457, 2571, 3058, 466, 2579, 3080, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w44er"}, {319, 237, 487, 1434, 1911, 503, 1490, 1898, 496, 1623, 0, 492, 1517, 1866, 492, 1446, 1896, 
			488, 1438, 1873, 487, 1451, 1883, 487, 1454, 1872, 494, 1663, 0, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w44ih"}, {266, 225, 451, 2247, 2929, 457, 2255, 2935, 444, 2240, 2891, 444, 2217, 2891, 448, 2199, 2877, 
			471, 2109, 2865, 495, 2028, 2855, 500, 2027, 2836, 497, 1874, 2812, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w44iy"}, {368, 226, 437, 2555, 3583, 480, 2579, 3589, 488, 2591, 3605, 486, 2632, 3586, 457, 2582, 3591, 
			439, 2549, 3572, 435, 2584, 3581, 434, 2599, 3578, 434, 2590, 3545, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w44oa"}, {347, 212, 435, 862, 2516, 603, 1037, 2426, 569, 1027, 2495, 501, 963, 2515, 444, 872, 2521, 
			435, 862, 2516, 436, 865, 2475, 438, 871, 2483, 443, 913, 2464, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w44oo"}, {242, 224, 499, 1144, 2511, 503, 1189, 2540, 496, 1125, 2519, 505, 1153, 2524, 514, 1182, 2572, 
			524, 1216, 2538, 500, 1299, 2571, 531, 1492, 2574, 499, 1617, 2506, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w44uh"}, {251, 214, 678, 1229, 2590, 657, 1236, 2602, 688, 1104, 2582, 694, 1067, 2575, 693, 1091, 2621, 
			695, 1197, 2679, 715, 1338, 2690, 688, 1444, 2711, 635, 1647, 2743, 70, 0, 0, 0, 0, 0, 0, 6, 0, 0, 0, 14, 0}},
		{{"w44uw"}, {346, 254, 430, 954, 2491, 449, 1086, 2442, 430, 940, 2485, 438, 995, 2462, 443, 1006, 2506, 
			452, 1014, 2526, 466, 1027, 2560, 459, 1042, 2582, 479, 1004, 2611, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w45ae"}, {333, 208, 657, 2192, 2654, 657, 2192, 2654, 649, 2235, 2698, 662, 2253, 2689, 734, 2146, 2677, 
			754, 2028, 2574, 852, 1938, 2597, 882, 1839, 2669, 880, 1809, 2706, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w45ah"}, {312, 195, 914, 1383, 2854, 950, 1382, 2814, 944, 1379, 2810, 914, 1391, 2842, 926, 1369, 2852, 
			930, 1438, 2829, 921, 1451, 2835, 924, 1505, 2810, 907, 1651, 2817, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"w45aw"}, {351, 196, 790, 1044, 2933, 832, 964, 2940, 815, 1018, 2934, 798, 1047, 2907, 786, 1062, 2931, 
			785, 1047, 2888, 838, 1118, 2844, 915, 1436, 2787, 867, 1529, 2760, 95, 0, 0, 0, 0, 0, 0, 19, 0, 0, 1, 0, 0}},
		{{"w45eh"}, {247, 198, 758, 2091, 2829, 786, 2080, 2790, 764, 2085, 2814, 765, 2081, 2825, 763, 2102, 2826, 
			758, 2091, 2829, 754, 2086, 2830, 734, 2119, 2888, 653, 2130, 2913, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w45ei"}, {319, 205, 606, 2438, 2723, 609, 2435, 2839, 630, 2365, 2734, 608, 2475, 2723, 602, 2436, 2659, 
			575, 2430, 2678, 511, 2446, 2644, 456, 2516, 2702, 427, 2455, 2744, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w45er"}, {310, 204, 537, 1638, 1932, 548, 1625, 1937, 537, 1679, 1936, 543, 1658, 1934, 537, 1638, 1933, 
			534, 1631, 1930, 531, 1643, 1930, 536, 1679, 1930, 512, 1710, 1957, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w45ih"}, {287, 213, 442, 2316, 2899, 446, 2379, 2925, 441, 2310, 2890, 442, 2269, 2854, 467, 2256, 2828, 
			501, 2221, 2804, 532, 2198, 2809, 554, 2162, 2780, 554, 2146, 2786, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w45iy"}, {326, 217, 431, 2522, 3008, 435, 2519, 3017, 431, 2522, 3008, 427, 2531, 3041, 424, 2517, 3055, 
			421, 2511, 3052, 417, 2517, 2959, 401, 2584, 2918, 417, 2561, 2932, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w45oa"}, {323, 206, 608, 1007, 2866, 641, 1059, 2814, 625, 1045, 2807, 613, 1009, 2866, 596, 1003, 2862, 
			569, 991, 2812, 561, 954, 2817, 522, 952, 2782, 492, 995, 2740, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w45oo"}, {239, 224, 480, 1108, 2717, 480, 1108, 2717, 495, 1050, 2713, 513, 1095, 2724, 551, 1172, 2679, 
			564, 1176, 2649, 568, 1368, 2679, 570, 1548, 2682, 563, 1701, 2689, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w45uh"}, {274, 199, 780, 1380, 2854, 769, 1297, 2851, 784, 1348, 2850, 780, 1380, 2854, 777, 1380, 2816, 
			783, 1431, 2854, 777, 1505, 2845, 756, 1554, 2847, 720, 1689, 2875, 55, 0, 0, 0, 0, 0, 6, 2, 0, 0, 1, 11, 0}},
		{{"w45uw"}, {340, 204, 415, 1001, 2556, 437, 981, 2535, 436, 978, 2548, 432, 1001, 2565, 427, 1001, 2577, 
			418, 1000, 2546, 408, 999, 2606, 404, 999, 2587, 405, 1005, 2577, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w46ae"}, {365, 156, 649, 2508, 3050, 634, 2515, 3056, 612, 2532, 2973, 595, 2512, 3018, 674, 2489, 3067, 
			736, 2419, 3082, 795, 2328, 3049, 823, 2138, 3011, 824, 2126, 3018, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w46ah"}, {334, 150, 901, 1509, 2670, 976, 1481, 2710, 956, 1515, 2783, 933, 1487, 2663, 941, 1582, 2698, 
			909, 1613, 2679, 884, 1568, 2662, 872, 1615, 2662, 846, 1743, 2709, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"w46aw"}, {396, 162, 850, 1182, 2909, 850, 1182, 2909, 855, 1188, 2906, 835, 1222, 2942, 838, 1247, 2955, 
			856, 1335, 2923, 849, 1363, 2946, 850, 1475, 2935, 818, 1643, 2934, 40, 0, 0, 0, 0, 0, 12, 8, 0, 0, 0, 0, 0}},
		{{"w46eh"}, {284, 168, 676, 2196, 3124, 701, 2390, 3230, 680, 2207, 3171, 676, 2196, 3124, 669, 2155, 3162, 
			672, 2164, 3168, 657, 2109, 3223, 656, 2096, 3227, 629, 2099, 3101, 95, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0, 0, 0}},
		{{"w46ei"}, {377, 160, 485, 2619, 0, 482, 2595, 0, 477, 2668, 0, 479, 2679, 0, 456, 2842, 0, 
			440, 2794, 0, 433, 2742, 0, 385, 2797, 0, 357, 2829, 0, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w46er"}, {382, 166, 500, 1637, 1928, 523, 1620, 1939, 508, 1690, 1912, 502, 1671, 1932, 500, 1642, 1923, 
			499, 1631, 1938, 495, 1636, 1942, 490, 1723, 1996, 490, 1757, 2065, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w46ih"}, {252, 149, 449, 2492, 3040, 462, 2525, 3103, 449, 2492, 3040, 450, 2461, 3044, 480, 2448, 3026, 
			506, 2405, 3030, 539, 2372, 2993, 554, 2310, 2952, 548, 2268, 2984, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w46iy"}, {316, 155, 331, 3049, 3736, 360, 3055, 3776, 332, 3057, 3750, 327, 3018, 3742, 345, 3015, 3646, 
			363, 2980, 3772, 368, 3005, 3793, 371, 2996, 3764, 373, 2996, 3620, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w46oa"}, {350, 183, 568, 1112, 2827, 566, 1098, 2825, 549, 1059, 2826, 511, 1030, 2868, 500, 996, 2904, 
			490, 973, 2891, 458, 934, 2914, 446, 937, 2825, 444, 1082, 2760, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w46oo"}, {265, 177, 520, 1421, 2998, 517, 1410, 2986, 505, 1438, 2953, 505, 1459, 2856, 506, 1467, 2911, 
			514, 1483, 2949, 529, 1595, 2915, 547, 1747, 2892, 510, 1885, 2939, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w46uh"}, {200, 155, 693, 1277, 2935, 693, 1277, 2935, 703, 1340, 2949, 704, 1370, 2938, 703, 1413, 2902, 
			718, 1456, 3017, 715, 1496, 2956, 670, 1621, 2955, 586, 1739, 2986, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w46uw"}, {302, 166, 362, 1351, 2879, 372, 1400, 2907, 367, 1365, 2913, 358, 1324, 2885, 348, 1266, 2964, 
			341, 1208, 2910, 332, 1166, 2920, 322, 1078, 2943, 322, 1238, 2935, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w47ae"}, {327, 211, 817, 2102, 2711, 806, 2174, 2812, 818, 2122, 2679, 819, 2082, 2723, 833, 2052, 2701, 
			866, 1989, 2704, 878, 1941, 2703, 868, 1910, 2708, 832, 1900, 2720, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"w47ah"}, {302, 207, 975, 1534, 2658, 906, 1463, 2690, 963, 1473, 2607, 971, 1472, 2634, 981, 1513, 2646, 
			975, 1534, 2658, 957, 1543, 2662, 935, 1552, 2662, 845, 1635, 2671, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"w47aw"}, {310, 206, 813, 1117, 2585, 792, 1050, 2527, 799, 1097, 2561, 791, 1100, 2560, 813, 1117, 2585, 
			837, 1162, 2600, 864, 1205, 2629, 866, 1359, 2671, 793, 1513, 2699, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"w47eh"}, {244, 217, 770, 1866, 2653, 710, 2028, 2856, 733, 2014, 2852, 752, 1978, 2705, 767, 1880, 2582, 
			770, 1866, 2653, 752, 1875, 2716, 709, 1922, 2759, 675, 1960, 2770, 75, 0, 0, 0, 15, 5, 0, 0, 0, 0, 0, 0, 0}},
		{{"w47ei"}, {313, 218, 472, 2616, 0, 506, 2694, 0, 487, 2629, 0, 450, 2643, 0, 436, 2716, 0, 
			433, 2744, 0, 431, 2747, 0, 429, 2749, 0, 430, 2758, 0, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w47er"}, {327, 211, 456, 1615, 1826, 462, 1620, 1882, 456, 1620, 1864, 460, 1616, 1819, 463, 1627, 1873, 
			493, 1616, 1927, 514, 1619, 1929, 511, 1647, 1933, 501, 1696, 2017, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w47ih"}, {208, 221, 444, 2237, 2711, 463, 2345, 2860, 445, 2270, 2748, 445, 2189, 2674, 470, 2155, 2712, 
			554, 2085, 2737, 593, 1986, 2726, 612, 1917, 2714, 573, 1933, 2726, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w47iy"}, {336, 230, 440, 2757, 3175, 462, 2813, 3423, 446, 2801, 3405, 442, 2754, 3311, 439, 2753, 3247, 
			440, 2754, 3179, 440, 2757, 3172, 442, 2770, 0, 452, 2821, 0, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w47oa"}, {380, 224, 624, 1109, 2691, 639, 1132, 2686, 624, 1109, 2691, 568, 1072, 2725, 516, 1070, 2755, 
			479, 1072, 2809, 448, 1040, 2759, 485, 947, 2801, 493, 945, 2754, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w47oo"}, {231, 237, 498, 1162, 2708, 499, 1133, 2707, 498, 1170, 2734, 499, 1131, 2756, 527, 1325, 2749, 
			558, 1382, 2742, 584, 1527, 2711, 571, 1625, 2698, 560, 1769, 2713, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w47uh"}, {206, 213, 712, 1499, 2757, 699, 1420, 2713, 692, 1395, 2729, 687, 1481, 2697, 712, 1489, 2758, 
			717, 1501, 2748, 741, 1511, 2725, 696, 1640, 2709, 634, 1684, 2699, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w47uw"}, {272, 238, 482, 987, 2781, 495, 1017, 2796, 494, 1006, 2727, 483, 991, 2755, 479, 980, 2812, 
			483, 969, 2821, 489, 979, 2733, 490, 980, 2721, 493, 1026, 2727, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w48ae"}, {310, 210, 626, 2331, 2826, 626, 2331, 2826, 623, 2266, 2852, 624, 2201, 2816, 654, 2082, 2816, 
			735, 1897, 2817, 781, 1803, 2829, 804, 1791, 2858, 741, 1811, 2888, 95, 0, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0}},
		{{"w48ah"}, {346, 218, 796, 1502, 2810, 802, 1494, 2805, 818, 1491, 2778, 817, 1493, 2786, 806, 1503, 2812, 
			796, 1502, 2810, 752, 1547, 2848, 732, 1584, 2856, 688, 1720, 2839, 95, 0, 0, 0, 0, 0, 19, 1, 0, 0, 0, 0, 0}},
		{{"w48aw"}, {355, 221, 736, 1098, 2821, 747, 1111, 2742, 736, 1103, 2778, 731, 1106, 2821, 719, 1147, 2838, 
			721, 1231, 2839, 741, 1306, 2830, 749, 1472, 2820, 723, 1557, 2814, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"w48eh"}, {244, 213, 683, 1851, 2877, 669, 2014, 2877, 673, 1983, 2861, 676, 1924, 2856, 682, 1855, 2867, 
			683, 1837, 2888, 682, 1833, 2901, 656, 1828, 2925, 635, 1866, 2962, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w48ei"}, {309, 224, 546, 2237, 2864, 544, 2125, 2865, 566, 2232, 2885, 546, 2237, 2864, 495, 2249, 2845, 
			471, 2293, 2834, 444, 2326, 2833, 449, 2380, 2805, 484, 2423, 2844, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w48er"}, {346, 231, 604, 1441, 1835, 608, 1444, 1839, 592, 1432, 1817, 567, 1417, 1771, 564, 1446, 1684, 
			538, 1459, 1686, 506, 1495, 1719, 492, 1509, 1730, 470, 1550, 1817, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w48ih"}, {209, 238, 498, 2234, 2876, 498, 2234, 2876, 490, 2224, 2884, 497, 2210, 2912, 511, 2194, 2882, 
			550, 2179, 2875, 560, 2181, 2903, 559, 2149, 2937, 518, 2060, 2937, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w48iy"}, {279, 255, 489, 2641, 2952, 485, 2590, 2907, 490, 2653, 2882, 489, 2641, 2952, 488, 2690, 2954, 
			491, 2688, 2944, 490, 2702, 2947, 492, 2703, 2932, 497, 2622, 2913, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w48oa"}, {282, 237, 522, 923, 2871, 583, 949, 2756, 568, 922, 2845, 522, 923, 2871, 500, 935, 2861, 
			493, 946, 2891, 495, 959, 2889, 501, 973, 2800, 502, 990, 2811, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w48oo"}, {239, 255, 519, 1051, 2741, 519, 1051, 2741, 506, 1028, 2763, 508, 1092, 2758, 517, 1151, 2757, 
			544, 1214, 2759, 559, 1353, 2748, 556, 1493, 2725, 530, 1709, 2740, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w48uh"}, {208, 236, 733, 1378, 2891, 825, 1268, 2948, 746, 1359, 2907, 738, 1377, 2897, 726, 1388, 2886, 
			724, 1417, 2884, 718, 1535, 2855, 706, 1644, 2849, 693, 1696, 2864, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w48uw"}, {265, 234, 458, 1163, 2651, 496, 1266, 2658, 488, 1192, 2643, 477, 1153, 2662, 473, 1157, 2670, 
			462, 1161, 2687, 456, 1165, 2646, 459, 1170, 2668, 460, 1174, 2662, 95, 0, 0, 0, 0, 0, 0, 0, 0, 1, 19, 0, 0}},
		{{"w49ae"}, {319, 209, 706, 2400, 2923, 704, 2410, 2918, 724, 2327, 2929, 747, 2340, 3001, 741, 2378, 2952, 
			754, 2244, 2940, 731, 2186, 2948, 747, 2118, 2947, 697, 2084, 3077, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w49ah"}, {326, 224, 1145, 0, 3272, 1129, 0, 3370, 1149, 0, 3276, 1123, 0, 3249, 1106, 0, 3166, 
			974, 0, 3193, 1075, 0, 3149, 978, 1731, 3142, 892, 1829, 3249, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"w49aw"}, {318, 219, 677, 1054, 3278, 684, 1001, 3263, 674, 1066, 3264, 705, 1068, 3254, 791, 1105, 3220, 
			818, 1179, 3209, 810, 1314, 3160, 777, 1525, 3187, 748, 1655, 3204, 100, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0}},
		{{"w49eh"}, {221, 222, 690, 2082, 2647, 690, 2082, 2647, 663, 2104, 2705, 670, 2114, 2706, 683, 2068, 2682, 
			638, 2083, 2682, 634, 2087, 2537, 630, 2001, 2495, 625, 1991, 2539, 100, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w49ei"}, {305, 214, 439, 2801, 3084, 472, 2747, 3153, 445, 2739, 3098, 438, 2772, 3099, 428, 2897, 3134, 
			398, 2900, 3174, 378, 2942, 3200, 375, 2957, 3157, 373, 3000, 3309, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w49er"}, {301, 176, 513, 1569, 1869, 456, 1621, 2025, 461, 1584, 2109, 448, 1559, 1986, 441, 1499, 1847, 
			446, 1548, 1860, 512, 1572, 1925, 502, 1552, 1916, 499, 1567, 1981, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w49ih"}, {216, 225, 463, 2603, 2991, 479, 2630, 3006, 448, 2603, 2975, 439, 2609, 3016, 434, 2596, 3026, 
			429, 2574, 3036, 441, 2547, 3050, 497, 2515, 3079, 500, 2392, 3116, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w49iy"}, {290, 243, 419, 2874, 3367, 419, 2874, 3367, 398, 2885, 3382, 413, 2865, 3305, 418, 2878, 3390, 
			395, 2860, 3408, 373, 2880, 3413, 372, 2881, 3488, 365, 2899, 3495, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w49oa"}, {315, 206, 444, 803, 3233, 486, 795, 3391, 444, 803, 3233, 433, 796, 3209, 423, 733, 3210, 
			440, 0, 3256, 443, 0, 3188, 445, 0, 3115, 441, 762, 3158, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w49oo"}, {217, 256, 509, 1049, 3306, 504, 1112, 3246, 497, 1147, 3187, 500, 1120, 3096, 524, 1092, 3074, 
			550, 1228, 3002, 557, 1426, 2890, 557, 1532, 2792, 552, 1705, 2763, 95, 0, 0, 0, 0, 0, 0, 0, 0, 19, 1, 0, 0}},
		{{"w49uh"}, {215, 220, 796, 1271, 3197, 773, 1274, 3217, 719, 1294, 3276, 807, 1363, 3205, 819, 1511, 3126, 
			794, 1553, 2980, 763, 1623, 3013, 706, 1712, 3075, 663, 1819, 3024, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0}},
		{{"w49uw"}, {248, 240, 467, 993, 2802, 454, 959, 2768, 438, 934, 2752, 436, 887, 2738, 428, 821, 2723, 
			387, 785, 2704, 373, 778, 2696, 367, 856, 2709, 363, 824, 2757, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}},
		{{"w50ae"}, {357, 209, 751, 2432, 2896, 749, 2386, 2926, 750, 2432, 2896, 749, 2387, 2914, 799, 2291, 2924, 
			816, 2093, 2927, 844, 2020, 2947, 858, 1991, 2943, 829, 2044, 2973, 100, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0}},
		{{"w50ah"}, {357, 234, 968, 1433, 2850, 942, 1449, 2824, 965, 1427, 2844, 976, 1428, 2875, 958, 1522, 2839, 
			944, 1520, 2864, 961, 1694, 2900, 944, 1777, 2886, 898, 1849, 2821, 100, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0, 0, 0}},
		{{"w50aw"}, {361, 246, 995, 1359, 2941, 999, 1361, 2945, 987, 1354, 2932, 956, 1346, 2944, 934, 1368, 3016, 
			881, 1351, 2937, 867, 1315, 2945, 848, 1415, 2811, 847, 1704, 2767, 90, 0, 0, 0, 0, 0, 2, 18, 0, 0, 0, 0, 0}},
		{{"w50eh"}, {270, 219, 817, 1984, 2973, 756, 2058, 2943, 760, 2005, 2950, 798, 1951, 2921, 818, 1933, 2932, 
			826, 1939, 2932, 822, 1974, 2933, 813, 1983, 2973, 770, 1979, 2954, 95, 0, 0, 1, 19, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w50ei"}, {361, 252, 664, 2455, 2915, 664, 2455, 2915, 611, 2485, 2883, 559, 2512, 2922, 506, 2558, 2980, 
			492, 2564, 2929, 461, 2647, 2877, 444, 2699, 2919, 440, 2722, 2927, 100, 0, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w50er"}, {356, 224, 503, 1766, 1956, 723, 1714, 1970, 613, 1676, 1961, 567, 1651, 1931, 521, 1632, 1974, 
			498, 1695, 1979, 478, 1725, 1931, 445, 1718, 1920, 437, 1888, 2152, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20}},
		{{"w50ih"}, {245, 239, 499, 2384, 3091, 500, 2396, 3042, 496, 2342, 3086, 498, 2280, 3002, 542, 2229, 2969, 
			580, 2113, 2939, 612, 2033, 2966, 612, 2028, 2962, 589, 2052, 2961, 100, 0, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w50iy"}, {314, 231, 437, 2778, 3139, 436, 2882, 3529, 437, 2823, 3423, 435, 2820, 3321, 438, 2811, 3218, 
			436, 2805, 3129, 437, 2789, 3158, 439, 2805, 3103, 430, 2839, 3153, 100, 20, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
		{{"w50oa"}, {361, 239, 720, 1277, 2812, 728, 1278, 2814, 698, 1267, 2797, 676, 1235, 2784, 632, 1224, 2751, 
			605, 1180, 2691, 568, 1115, 2702, 505, 1073, 2690, 464, 1111, 2650, 100, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0, 0}},
		{{"w50oo"}, {273, 236, 553, 1135, 2859, 557, 1209, 2908, 554, 1122, 2843, 556, 1115, 2819, 553, 1259, 2805, 
			559, 1380, 2798, 568, 1374, 2796, 558, 1579, 2739, 523, 1847, 2990, 100, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0, 0}},
		{{"w50uh"}, {262, 213, 811, 1450, 2938, 825, 1399, 2908, 815, 1441, 2935, 813, 1455, 2944, 815, 1475, 2931, 
			814, 1502, 2894, 809, 1705, 2839, 742, 1792, 2832, 659, 1959, 2853, 80, 0, 0, 0, 0, 0, 2, 2, 0, 0, 0, 16, 0}},
		{{"w50uw"}, {308, 265, 491, 1317, 2680, 495, 1329, 2662, 490, 1314, 2689, 492, 1293, 2748, 492, 1259, 2773, 
			485, 1216, 2745, 471, 1179, 2703, 457, 1197, 2703, 453, 1294, 2679, 100, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20, 0, 0}}
	};
	autoSTRVEC vowels ({U"ae", U"ah", U"aw", U"eh", U"ei", U"er", U"ih", U"iy", U"oa", U"oo", U"uh", U"uw"});
	autoSTRVEC ipa    ({U"æ",  U"ɑ",  U"ɔ",  U"ɛ",  U"e",  U"ɝ",  U"ɪ",  U"i",  U"o",  U"ʊ",  U"ʌ",  U"u"});
	const integer numberOfRows = 1668, numberOfColumns = 47;
	autoTable me = Table_create (numberOfRows, numberOfColumns);
	for (integer irow = 1; irow <= numberOfRows; irow ++) {
		const TableRow row = my rows.at [irow];
		conststring32 name = Melder_peek8to32 (hillenbranddata [irow - 1].name);
		if (name [0] == U'm') {
			row -> cells [1]. string = Melder_dup (U"m"); // Type
			row -> cells [2]. string = Melder_dup (U"m"); // Sex
		} else if (name[0] == U'w') {
			row -> cells [1]. string = Melder_dup (U"w");
			row -> cells [2]. string = Melder_dup (U"f");
		} else if (name[0] == U'b') {
			row -> cells [1]. string = Melder_dup (U"c");
			row -> cells [2]. string = Melder_dup (U"m");
		} else if (name[0] == U'g') {
			row -> cells [1]. string = Melder_dup (U"c");
			row -> cells [2]. string = Melder_dup (U"f");
		}
		char32 tmp [3] = U"\0\0";
		if (name [1] == U'0') {
			tmp [0] = name [2]; tmp [1] = U'\0';
		} else {
			tmp [0] = name [1]; tmp [1] = name [2];
		}
		row -> cells [3]. string = Melder_dup (tmp); // Speaker
		tmp [0] = name [3]; tmp [1] = name [4];
		row -> cells [4]. string = Melder_dup (tmp); // Vowel
		for (integer ivowel = 1; ivowel <= vowels.size; ivowel ++) {
			if (Melder_cmp (vowels [ivowel].get(), tmp) == 0) {
				row -> cells [5]. string = Melder_dup (ipa [ivowel].get()); // Vowel
			}
		}
		for (integer j = 0; j < 42; j ++)
			row -> cells [j + 6]. string = Melder_dup (Melder_integer (hillenbranddata [irow - 1].data [j]));
		const double dur = hillenbranddata [irow - 1]. data [1] / 1000.0;
		row ->  cells [7]. string = Melder_dup (Melder_fixed (dur, 3));
	}
			
	for (integer icol = 1; icol <= numberOfColumns; icol ++) {
		Table_renameColumn_e (me.get(), icol, columnLabels [icol - 1]);
		my columnHeaders [icol]. numericized = false;
	}
	return me;
}

/* End of file Table_dataSets.cpp */
